package org.eclipse.hyades.logging.events;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamField;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.hyades.logging.core.ISerializableAsXml;
import org.eclipse.hyades.logging.events.exceptions.InvalidXMLAnyElementException;
import org.eclipse.hyades.logging.events.exceptions.UnsupportedVersionException;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CommonBaseEventImpl.java,v 1.25 2005/02/16 22:20:41 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * This is the class that embodies the Common Base Event as described in 
 * "Canonical Situation Data Format: The Common Base Event V1.0.1 
 * ACAB.BO0301.0.1"
 * 
 * 
 * @author Richard K. Duggan
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0
 * @see ICommonBaseEvent 
 * @deprecated This class is deprecated as of 2004-05-17. Use the corresponding class available in org.eclipse.hyades.logging.events.cbe*
 */
public class CommonBaseEventImpl implements ICommonBaseEvent, ISerializableAsXml, Serializable, Cloneable {

	//Define the serialiVersionUID for serialization:
	//NOTE: If this object is ever changed which breaks backward compatibility,
	//this value MUST be changed.  Every effort should be made not to break 
	//backward compatibility.
	static final long serialVersionUID = -4234730049423367771L;

	//Define the fields that will be serialized: 
	//NOTE: If any new variables are added to the class this table should be 
	//updated along with the readObject and writeObject methods.
	private static final ObjectStreamField[] serialPersistentFields =
		{
			new ObjectStreamField("extensionName", String.class),
			new ObjectStreamField("localInstanceId", String.class),
			new ObjectStreamField("globalInstanceId", String.class),
			new ObjectStreamField("msg", String.class),
			new ObjectStreamField("otherData", String[].class),
			new ObjectStreamField("elapsedTime", long.class),
			new ObjectStreamField("severity", short.class),
			new ObjectStreamField("creationTimeValue", long.class),
			new ObjectStreamField("creationTime", String.class),
			new ObjectStreamField("priority", short.class),
			new ObjectStreamField("situationType", String.class),
			new ObjectStreamField("repeatCount", short.class),
			new ObjectStreamField("sequenceNumber", long.class),
			new ObjectStreamField("msgDataElement", IMsgDataElement.class),
			new ObjectStreamField("reporterComponentId", IComponentIdentification.class),
			new ObjectStreamField("sourceComponentId", IComponentIdentification.class),
			new ObjectStreamField("contextDataElements", IContextDataElement[].class),
			new ObjectStreamField("extendedDataElements", IExtendedDataElement[].class),
			new ObjectStreamField("associatedEvents", IAssociatedEvent[].class),
			new ObjectStreamField("anyElements", ICommonBaseEventAnyElement[].class),
			new ObjectStreamField("situation", ISituation.class),
			new ObjectStreamField("version", String.class)};

	//The current version of the event API.
	private String version = null;

	//Class instance variable(s):
	protected IMsgDataElement msgDataElement = null;
	protected IComponentIdentification reporterComponentId = null;
	protected IComponentIdentification sourceComponentId = null;
	protected ArrayList contextDataElements = null;
	protected ArrayList extendedDataElements = null;
	protected ArrayList associatedEvents = null;
	protected ArrayList anyElements = null;
	protected ISituation situation = null;

	protected String extensionName = null;
	protected String localInstanceId = null;
	protected String globalInstanceId = null;
	protected String creationTime = null;
	protected String msg = null;
	protected String[] otherData = null;
	protected long creationTimeValue = 0;
	protected long elapsedTime = 0;
	protected short severity = 0;
	protected short priority = 0;
	protected short repeatCount = 0;
	protected long sequenceNumber = 0;

	private boolean creationTimeIsSet = false;
	private boolean elapsedTimeIsSet = false;
	private boolean severityIsSet = false;
	private boolean priorityIsSet = false;
	private boolean repeatCountIsSet = false;
	private boolean sequenceNumberIsSet = false;

	//Deprecated: Use the ISituation as of 1.0.1.
	protected String situationType = null;


	/**
	 * No-argument protected constructor for CommonBaseEventImpl.
	 */
	protected CommonBaseEventImpl() {

		super();

		version = VERSION_1_0_1;
		globalInstanceId = (SimpleEventFactoryImpl.getInstance().createGlobalInstanceId());
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addContextDataElement(IContextDataElement)
	 */
	public void addContextDataElement(IContextDataElement element) {

		if (contextDataElements == null) {
			contextDataElements = new ArrayList();
		}

		this.contextDataElements.add(element);
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElement(IExtendedDataElement)
	 */
	public void addExtendedDataElement(IExtendedDataElement element) {

		if (extendedDataElements == null) {
			extendedDataElements = new ArrayList();
		}

		this.extendedDataElements.add(element);
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addAssociatedEvent(IAssociatedEvent)
	 */
	public void addAssociatedEvent(IAssociatedEvent event) {

		String associatedEngineName = null;
		IAssociationEngine associationEngine = event.getAssociationEngineInfo();

		if (associationEngine != null) {
			associatedEngineName = associationEngine.getName();
		}

		if (associatedEngineName == null)
			associatedEngineName = event.getAssociationEngine();

		if (associatedEngineName != null) {

			if (associatedEvents == null) {
				associatedEvents = new ArrayList();
			}

			this.associatedEvents.add(event);
		}
		else {
			throw new RuntimeException("IAssociatedEvent must be assigned an AssociatineEngine to add it to an instance of ICommonBaseEvent");
		}
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getContextDataElements()
	 */
	public List getContextDataElements() {
    	
		if (contextDataElements == null) {
			contextDataElements = new ArrayList();
		}

		return contextDataElements;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getExtendedDataElements()
	 */
	public List getExtendedDataElements() {
		
		if (extendedDataElements == null) {
			extendedDataElements = new ArrayList();
		}
        
		return extendedDataElements;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getAssociatedEvents()
	 */
	public List getAssociatedEvents() {
       
		if (associatedEvents == null) {
			associatedEvents = new ArrayList();
		}

		return associatedEvents;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getContextDataElement(String)
	 */
	public IContextDataElement getContextDataElement(String name) {

		if (contextDataElements != null) {

			for (int counter = 0; counter < contextDataElements.size(); counter++) {

				if (((IContextDataElement) (contextDataElements.get(counter))).getName().equals(name)) {
					return ((IContextDataElement) (contextDataElements.get(counter)));
				}
			}
		}

		return null;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getExtendedDataElement(String)
	 */
	public IExtendedDataElement getExtendedDataElement(String name) {

		if (extendedDataElements != null) {

			for (int counter = 0; counter < extendedDataElements.size(); counter++) {

				if (((IExtendedDataElement) (extendedDataElements.get(counter))).getName().equals(name)) {
					return ((IExtendedDataElement) (extendedDataElements.get(counter)));
				}
			}
		}

		return null;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#removeContextDataElement(String)
	 */
	public void removeContextDataElement(String name) {

		if (contextDataElements != null) {

			for (int counter = 0; counter < contextDataElements.size(); counter++) {

				if (((IContextDataElement) (contextDataElements.get(counter))).getName().equals(name)) {
					contextDataElements.remove(counter--);
				}
			}
		}
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#removeExtendedDataElement(String)
	 */
	public void removeExtendedDataElement(String name) {

		if (extendedDataElements != null) {

			for (int counter = 0; counter < extendedDataElements.size(); counter++) {

				if (((IExtendedDataElement) (extendedDataElements.get(counter))).getName().equals(name)) {
					extendedDataElements.remove(counter--);
				}
			}
		}
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#clearAssociatedEvents()
	 */
	public void clearAssociatedEvents() {
		associatedEvents = null;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#clearContextDataElements()
	 */
	public void clearContextDataElements() {
		contextDataElements = null;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#clearExtendedDataElements()
	 */
	public void clearExtendedDataElements() {
		extendedDataElements = null;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addAnyElement(ICommonBaseEventAnyElement)
	 * @since 1.0 
	 */
	public void addAnyElement(ICommonBaseEventAnyElement element) {

		if (anyElements == null) {
			anyElements = new ArrayList();
		}

		this.anyElements.add(element);
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getAnyElements()
	 * @since 1.0 
	 */
	public List getAnyElements() {
        
		if (anyElements == null) {
			anyElements = new ArrayList();
		}

		return anyElements;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getAnyElement(String)
	 * @since 1.0 
	 */
	public ICommonBaseEventAnyElement getAnyElement(String name) {

		if (anyElements != null) {

			for (int counter = 0; counter < anyElements.size(); counter++) {

				if (((ICommonBaseEventAnyElement) (anyElements.get(counter))).getElementName().equals(name)) {
					return ((ICommonBaseEventAnyElement) (anyElements.get(counter)));
				}
			}
		}

		return null;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#removeAnyElement(String)
	 * @since 1.0 
	 */
	public void removeAnyElement(String name) {

		if (anyElements != null) {

			for (int counter = 0; counter < anyElements.size(); counter++) {

				if (((ICommonBaseEventAnyElement) (anyElements.get(counter))).getElementName().equals(name)) {
					anyElements.remove(counter--);
				}
			}
		}
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#clearAnyElements()
	 * @since 1.0 
	 */
	public void clearAnyElements() {
		anyElements = null;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getCreationTime()
	 */
	public String getCreationTime() {
		if ((creationTime == null) && (creationTimeValue > 0)) {
			creationTime = CbeFormatter.convertDateToXmlSchemaDateTime(creationTimeValue);
		}
		return creationTime;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getCreationTimeAsLong()
	 */
	public long getCreationTimeAsLong() {

		if ((creationTimeValue == 0) && (creationTime != null)) {
			creationTimeValue = CbeFormatter.convertXmlSchemaDateTimeToDate(creationTime);
		}

		return creationTimeValue;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getElapsedTime()
	 */
	public long getElapsedTime() {
		return elapsedTime;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getExtensionName()
	 */
	public String getExtensionName() {
		return extensionName;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getGlobalInstanceId()
	 */
	public String getGlobalInstanceId() {
		return globalInstanceId;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getLocalInstanceId()
	 */
	public String getLocalInstanceId() {
		return localInstanceId;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getMsg()
	 */
	public String getMsg() {
		return msg;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getPriority()
	 */
	public short getPriority() {
		return priority;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getRepeatCount()
	 */
	public short getRepeatCount() {
		return repeatCount;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getSequenceNumber()
	 */
	public long getSequenceNumber() {
		return sequenceNumber;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getSeverity()
	 */
	public short getSeverity() {
		return severity;
	}

	/**
	 * Returns the situationType.
	 * 
	 * The situationType specifies the type of the situation that caused the event 
	 * to be reported. This is an extensible string value. Proposed reserved keywords
	 *  are:
	 * 1. START
	 * 2. STOP
	 * 3. FEATURE
	 * 4. DEPENDENCY
	 * 5. REQUEST
	 * 6. CONFIGURE
	 * 7. CONNECT
	 * 8. CREATE
	 * 9. UNKNOWN
	 * This is an OPTIONAL property. The maximum string length for situationType MUST
	 * not exceed 64 bytes.
	 * NOTE: The situations type is being defined in parallel to the CommonBaseEventImpl
	 * effort in a separate document, Canonical Situations ' ACAB.CT0304.0.1. ' The type
	 * and the values of situationType are subject to change.
	 * 
	 * @return String The returned situationType.
	 * @since 1.0
	 * @deprecated Use the ISituation as of 1.0.1.
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getSituationType()
	 */
	public String getSituationType() {
		return situationType;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getSourceComponentId()
	 */
	public IComponentIdentification getSourceComponentId() {
		return sourceComponentId;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getMsgDataElement()
	 */
	public IMsgDataElement getMsgDataElement() {
		return msgDataElement;
	}

	/**
	 * Returns the otherData.
	 * 
	 * @param String[] The returned otherData.
	 * @since 1.0.1
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getOtherData()
	 */
	public String[] getOtherData() {
		return otherData;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getReporterComponentId()
	 */
	public IComponentIdentification getReporterComponentId() {
		return reporterComponentId;
	}

	/**
	 * Returns the situation.
	 * 
	 * The situation specifies the type of the situation that caused the 
	 * event to be reported. 
	 * This is an REQUIRED property.
	 * 
	 * @return String The returned situation.
	 * @since 1.0.1
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#getSituation()
	 */
	public ISituation getSituation() {
		return situation;
	}

	/**
	 * Returns the list of supported Common Base Event implementation version(s).
	 * 
	 * The list of supported Common Base Event implementation version(s) are all the 
	 * current and previous supported version(s) of the Common Base Event implementation.
	 * 
	 * A string identifying the version of this event.  This field is OPTIONAL 
	 * in that if it isnt specified, the version of the CBE is said to be 1.0.  
	 * Otherwise this field MUST be filled in and MUST be consistent with the 
	 * version specified in the schema header.The string length for version MUST 
	 * NOT exceed 16 characters. When an application is creating an event following 
	 * this version of the specification this field should be set to 1.0.1.
	 * 
	 * @return List The returned list of supported Common Base Event implementation version(s).
	 * @since 1.0.1
	 * @see ICommonBaseEvent#VERSION_1_0
	 * @see ICommonBaseEvent#VERSION_1_0_1
	 */
	public List getVersions() {
		return (Arrays.asList(new String[] { VERSION_1_0, VERSION_1_0_1 }));
	}

	/**
	 * Sets the preferredVersion.
	 * 
	 * The preferred version of the Common Base Event implementation as desired by the user.
	 * 
	 * A string identifying the version of this event.  This field is OPTIONAL 
	 * in that if it isnt specified, the version of the CBE is said to be 1.0.  
	 * Otherwise this field MUST be filled in and MUST be consistent with the 
	 * version specified in the schema header.The string length for version MUST 
	 * NOT exceed 16 characters. When an application is creating an event following 
	 * this version of the specification this field should be set to 1.0.1.
	 * 
	 * @param preferredVersion The preferredVersion to be set.
	 * @throws UnsupportedVersionException The preferredVersion is not a valid version.
	 * @since 1.0.1
	 * @see ICommonBaseEvent#VERSION_1_0
	 * @see ICommonBaseEvent#VERSION_1_0_1
	 */
	public void setPreferredVersion(String preferredVersion) throws UnsupportedVersionException {

		if ((preferredVersion.equals(VERSION_1_0)) || (preferredVersion.equals(VERSION_1_0_1)))
			version = preferredVersion;

		else
			throw new UnsupportedVersionException("Invalid preferred version (".concat(preferredVersion).concat(")."));
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setCreationTime(String)
	 */
	public void setCreationTime(String creationTime) {
		this.creationTime = creationTime;
		creationTimeIsSet = true;
		creationTimeValue = 0;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setCreationTime(long)
	 */
	public void setCreationTime(long creationTime) {
		this.creationTimeValue = creationTime;
		creationTimeIsSet = true;
		this.creationTime = null;
	}
	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setElapsedTime(long)
	 */
	public void setElapsedTime(long elapsedTime) {
		this.elapsedTime = elapsedTime;
		elapsedTimeIsSet = true;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setExtensionName(String)
	 */
	public void setExtensionName(String extensionName) {
		this.extensionName = extensionName;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setGlobalInstanceId(String)
	 */
	public void setGlobalInstanceId(String globalInstanceId) {
		this.globalInstanceId = globalInstanceId;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setLocalInstanceId(String)
	 */
	public void setLocalInstanceId(String localInstanceId) {
		this.localInstanceId = localInstanceId;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setMsg(String)
	 */
	public void setMsg(String msg) {
		this.msg = msg;
	}

	/**
	 * Sets the otherData.
	 * 
	 * @param situation The otherData to be set.
	 * @since 1.0.1
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setOtherData(String[])
	 */
	public void setOtherData(String[] otherData) {
		this.otherData = otherData;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setPriority(short)
	 */
	public void setPriority(short priority) {
		this.priority = priority;
		priorityIsSet = true;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setRepeatCount(short)
	 */
	public void setRepeatCount(short repeatCount) {
		this.repeatCount = repeatCount;
		repeatCountIsSet = true;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setSequenceNumber(long)
	 */
	public void setSequenceNumber(long sequenceNumber) {
		this.sequenceNumber = sequenceNumber;
		sequenceNumberIsSet = true;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setSeverity(short)
	 */
	public void setSeverity(short severity) {
		this.severity = severity;
		severityIsSet = true;
	}

	/**
	 * Sets the situationType.
	 * 
	 * The situationType specifies the type of the situation that caused the event 
	 * to be reported. This is an extensible string value. Proposed reserved keywords
	 *  are:
	 * 1. START
	 * 2. STOP
	 * 3. FEATURE
	 * 4. DEPENDENCY
	 * 5. REQUEST
	 * 6. CONFIGURE
	 * 7. CONNECT
	 * 8. CREATE
	 * 9. UNKNOWN
	 * This is an OPTIONAL property. The maximum string length for situationType MUST
	 * not exceed 64 bytes.
	 * NOTE: The situations type is being defined in parallel to the CommonBaseEventImpl
	 * effort in a separate document, Canonical Situations ' ACAB.CT0304.0.1. ' The type
	 * and the values of situationType are subject to change.
	 * 
	 * @param situationType The situationType to set
	 * @since 1.0
	 * @deprecated Use the ISituation as of 1.0.1.
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setSituationType(String)
	 */
	public void setSituationType(String situationType) {
		this.situationType = situationType;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setSourceComponentId(IComponentIdentification)
	 */
	public void setSourceComponentId(IComponentIdentification sourceComponentId) {
		this.sourceComponentId = sourceComponentId;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setMsgDataElement(IMsgDataElement)
	 */
	public void setMsgDataElement(IMsgDataElement msgDataElement) {
		this.msgDataElement = msgDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setReporterComponentId(IComponentIdentification)
	 */
	public void setReporterComponentId(IComponentIdentification reporterComponentId) {
		this.reporterComponentId = reporterComponentId;
	}

	/**
	 * Sets the situation.
	 * 
	 * The situation specifies the type of the situation that caused the 
	 * event to be reported. 
	 * This is an REQUIRED property.
	 * 
	 * @param situation The situation to be set.
	 * @since 1.0.1
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setSituation(ISituation)
	 */
	public void setSituation(ISituation situation) {
		this.situation = situation;
	}

	/**
	 * Resets the object's properties to their initial (e.g. null) state.
	 * 
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#init()
	 */
	public void init() {

		version = VERSION_1_0_1;

		//Clear the local fields:
        
		unsetCreationTime();
		unsetElapsedTime();
		unsetPriority();
		unsetRepeatCount();
		unsetSequenceNumber();
		unsetSeverity();

		associatedEvents = null;
		contextDataElements = null;
		extendedDataElements = null;
		anyElements = null;
		sourceComponentId = null;
		reporterComponentId = null;
		msgDataElement = null;
		situation = null;
		extensionName = null;
		localInstanceId = null;
		situationType = null;
		msg = null;
		otherData = null;
		
		globalInstanceId = (SimpleEventFactoryImpl.getInstance().createGlobalInstanceId());
	}

	/**
	 * @see org.eclipse.hyades.logging.core.IExternalizableToXml#externalizeCanonicalXmlString()
	 */
	public String externalizeCanonicalXmlString() {
		return CbeFormatter.toCanonicalXMLString(this);
	}

	/**
	 * @see org.eclipse.hyades.logging.core.IExternalizableToXml#externalizeCanonicalXmlDocString()
	 */
	public String externalizeCanonicalXmlDocString() {
		return CbeFormatter.toCanonicalXMLDocString(this);
	}

	/**
	 * @see org.eclipse.hyades.logging.core.IInternalizableFromXml#internalizeCanonicalXmlString(java.lang.String)
	 */
	public void internalizeCanonicalXmlString(String value) {
		CbeFormatter.fromCanonicalXMLString(this, value);
		return;
	}

	/**
	 * @see org.eclipse.hyades.logging.core.IInternalizableFromXml#internalizeCanonicalXmlDocString(java.lang.String)
	 */
	public void internalizeCanonicalXmlDocString(String value) {
		CbeFormatter.fromCanonicalXMLDocString(this, value);
		return;
	}

	/**
	 * Returns the Common Base Event implementation version.
	 * 
	 * A string identifying the version of this event.  This field is OPTIONAL 
	 * in that if it isnt specified, the version of the CBE is said to be 1.0.  
	 * Otherwise this field MUST be filled in and MUST be consistent with the 
	 * version specified in the schema header.The string length for version MUST 
	 * NOT exceed 16 characters. When an application is creating an event following 
	 * this version of the specification this field should be set to 1.0.1.
	 * 
	 * @return String The version of the Common Base Event implementation.
	 * @deprecated Use getVersion() as of 1.0.1.
	 * @see ICommonBaseEvent#VERSION_1_0
	 * @see ICommonBaseEvent#VERSION_1_0_1
	 */
	public String getCommonBaseEventImplVersion() {
		return version;
	}

	/**
	 * Returns the Common Base Event implementation version.
	 * 
	 * A string identifying the version of this event.  This field is OPTIONAL 
	 * in that if it isnt specified, the version of the CBE is said to be 1.0.  
	 * Otherwise this field MUST be filled in and MUST be consistent with the 
	 * version specified in the schema header.The string length for version MUST 
	 * NOT exceed 16 characters. When an application is creating an event following 
	 * this version of the specification this field should be set to 1.0.1.
	 * 
	 * @return String The version of the Common Base Event implementation.
	 * @see ICommonBaseEvent#VERSION_1_0
	 * @see ICommonBaseEvent#VERSION_1_0_1
	 */
	public String getVersion() {
		return version;
	}

	/**
	 * Returns the Common Base Event XML schema version.
	 * 
	 * @return String The version of the Common Base Event XML schema.
	 * @deprecated Use getVersion() as of 1.0.1.
	 * @see ICommonBaseEvent#VERSION_1_0
	 * @see ICommonBaseEvent#VERSION_1_0_1
	 */
	public String getEventSchemaVersion() {
		return CbeFormatter.getEventFormatterVersion();
	}

	/**
	 * Returns an XML string representation of the CommonBaseEvent
	 */
	public String toString() {
		return CbeFormatter.toCanonicalXMLString(this);
	}

	/**
	 * Determines is two different event instance are equal.
	 */
	public boolean equals(Object obj) {
		boolean result = false;

		//It's the same object reference, so it HAS to be equal
		if (this == obj) {
			result = true;
		}
		else {
			//If the object is null or is not the correct type of object, it can't be
			//equal
			if ((obj != null) && (obj instanceof CommonBaseEventImpl) && (getClass() == obj.getClass())) {
				CommonBaseEventImpl event = (CommonBaseEventImpl) obj;

				Object event1Property;
				Object event2Property;

				event1Property = getGlobalInstanceId();
				event2Property = event.getGlobalInstanceId();

				if (((event1Property != null) && (event1Property.equals(event2Property))) || ((event1Property == null) && (event2Property == null))) {
					if ((getCreationTimeAsLong() == event.getCreationTimeAsLong()) && (getElapsedTime() == event.getElapsedTime()) && (getSeverity() == event.getSeverity()) && (getPriority() == event.getPriority()) && (getRepeatCount() == event.getRepeatCount()) && (getSequenceNumber() == event.getSequenceNumber())) {
						event1Property = getLocalInstanceId();
						event2Property = event.getLocalInstanceId();

						if (((event1Property != null) && (event1Property.equals(event2Property))) || ((event1Property == null) && (event2Property == null))) {

							event1Property = getVersion();
							event2Property = event.getVersion();

							if (((event1Property != null) && (event1Property.equals(event2Property))) || ((event1Property == null) && (event2Property == null))) {

								event1Property = getExtensionName();
								event2Property = event.getExtensionName();

								if (((event1Property != null) && (event1Property.equals(event2Property))) || ((event1Property == null) && (event2Property == null))) {
									event1Property = getSituationType();
									event2Property = event.getSituationType();

									if (((event1Property != null) && (event1Property.equals(event2Property))) || ((event1Property == null) && (event2Property == null))) {
										event1Property = getMsg();
										event2Property = event.getMsg();

										if (((event1Property != null) && (event1Property.equals(event2Property))) || ((event1Property == null) && (event2Property == null))) {
											event1Property = getMsgDataElement();
											event2Property = event.getMsgDataElement();

											if (((event1Property != null) && (event1Property.equals(event2Property))) || ((event1Property == null) && (event2Property == null))) {

												event1Property = getSituation();
												event2Property = event.getSituation();

												if (((event1Property != null) && (event1Property.equals(event2Property))) || ((event1Property == null) && (event2Property == null))) {

													event1Property = getReporterComponentId();
													event2Property = event.getReporterComponentId();

													if (((event1Property != null) && (event1Property.equals(event2Property))) || ((event1Property == null) && (event2Property == null))) {
														event1Property = getSourceComponentId();
														event2Property = event.getSourceComponentId();

														if (((event1Property != null) && (event1Property.equals(event2Property))) || ((event1Property == null) && (event2Property == null))) {
															List event1ListProperty = contextDataElements;
															List event2ListProperty = event.contextDataElements;

															if (((event1ListProperty != null) && (event1ListProperty.equals(event2ListProperty))) || (((event1ListProperty == null) || (event1ListProperty.size() == 0)) && ((event2ListProperty == null) || (event2ListProperty.size() == 0)))) {
																event1ListProperty = extendedDataElements;
																event2ListProperty = event.extendedDataElements;

																if (((event1ListProperty != null) && (event1ListProperty.equals(event2ListProperty))) || (((event1ListProperty == null) || (event1ListProperty.size() == 0)) && ((event2ListProperty == null) || (event2ListProperty.size() == 0)))) {
																	event1ListProperty = associatedEvents;
																	event2ListProperty = event.associatedEvents;

																	if (((event1ListProperty != null) && (event1ListProperty.equals(event2ListProperty))) || (((event1ListProperty == null) || (event1ListProperty.size() == 0)) && ((event2ListProperty == null) || (event2ListProperty.size() == 0)))) {
																		event1ListProperty = anyElements;
																		event2ListProperty = event.anyElements;

																		if (((event1ListProperty != null) && (event1ListProperty.equals(event2ListProperty))) || (((event1ListProperty == null) || (event1ListProperty.size() == 0)) && ((event2ListProperty == null) || (event2ListProperty.size() == 0)))) {

																			event1Property = getOtherData();
																			event2Property = event.getOtherData();

																			if ((event1Property == null) && (event2Property == null))
																				result = true;

																			else if ((event1Property != null) && (event2Property != null) && (((String[]) (event1Property)).length == ((String[]) (event2Property)).length)) {

																				result = true;

																				for (int counter = 0; counter < ((String[]) (event1Property)).length; counter++) {

																					if (!((String[]) (event1Property))[counter].equals(((String[]) (event2Property))[counter])) {

																						result = false;
																						break;
																					}
																				}
																			}
																			else
																				result = false;
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}

		return result;
	}

	private void writeObject(ObjectOutputStream out) throws IOException {
		// set the values of the Serializable fields
		ObjectOutputStream.PutField fields = out.putFields();

		fields.put("extensionName", extensionName);
		fields.put("localInstanceId", localInstanceId);
		fields.put("globalInstanceId", globalInstanceId);
		fields.put("msg", msg);
		fields.put("otherData", otherData);
		fields.put("elapsedTime", elapsedTime);
		fields.put("version", version);
		fields.put("severity", severity);
		fields.put("creationTimeValue", creationTimeValue);
		fields.put("creationTime", creationTime);
		fields.put("priority", priority);
		fields.put("situationType", situationType);
		fields.put("repeatCount", repeatCount);
		fields.put("sequenceNumber", sequenceNumber);
		fields.put("sourceComponentId", sourceComponentId);

		if (msgDataElement != null) {
			fields.put("msgDataElement", msgDataElement);
		}
		if (situation != null) {
			fields.put("situation", situation);
		}
		if (reporterComponentId != null) {
			fields.put("reporterComponentId", reporterComponentId);
		}
		if ((contextDataElements != null) && (!contextDataElements.isEmpty())) {
			fields.put("contextDataElements", ((IContextDataElement[]) (contextDataElements.toArray(new IContextDataElement[contextDataElements.size()]))));
		}
		if ((extendedDataElements != null) && (!extendedDataElements.isEmpty())) {
			fields.put("extendedDataElements", ((IExtendedDataElement[]) (extendedDataElements.toArray(new IExtendedDataElement[extendedDataElements.size()]))));
		}
		if ((associatedEvents != null) && (!associatedEvents.isEmpty())) {
			fields.put("associatedEvents", ((IAssociatedEvent[]) (associatedEvents.toArray(new IAssociatedEvent[associatedEvents.size()]))));
		}
		if ((anyElements != null) && (!anyElements.isEmpty())) {
			// need to convert to the xml string
			// no gaurantee the other side contains the object class
			// for the any element
			Object anyElementArray[] = ((ICommonBaseEventAnyElement[]) (anyElements.toArray(new ICommonBaseEventAnyElement[anyElements.size()])));
			int arraySize = anyElementArray.length;
			String anyString[] = new String[arraySize];

			for (int i = 0; i < arraySize; i++) {
				anyString[i] = ((ICommonBaseEventAnyElement) anyElementArray[i]).getXmlAnyElement();
			}
			fields.put("anyElements", anyString);
		}

		// save them
		out.writeFields();
	}

	private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {

		// prepare to read the alternate persistent fields
		ObjectInputStream.GetField fields = null;
		fields = in.readFields();

		// read the alternate persistent fields
		extensionName = (String) fields.get("extensionName", extensionName);
		localInstanceId = (String) fields.get("localInstanceId", localInstanceId);
		globalInstanceId = (String) fields.get("globalInstanceId", globalInstanceId);
		msg = (String) fields.get("msg", msg);
		otherData = ((String[]) (fields.get("otherData", otherData)));
		elapsedTime = (long) fields.get("elapsedTime", elapsedTime);
		version = (String) fields.get("version", version);
		severity = (short) fields.get("severity", severity);
		creationTime = (String) fields.get("creationTime", creationTime);
		creationTimeValue = (long) fields.get("creationTimeValue", creationTimeValue);
		priority = (short) fields.get("priority", priority);
		situationType = (String) fields.get("situationType", situationType);
		repeatCount = (short) fields.get("repeatCount", repeatCount);
		sequenceNumber = (long) fields.get("sequenceNumber", sequenceNumber);
		msgDataElement = (IMsgDataElement) fields.get("msgDataElement", msgDataElement);
		situation = (ISituation) fields.get("situation", situation);
		reporterComponentId = (IComponentIdentification) fields.get("reporterComponentId", reporterComponentId);
		sourceComponentId = (IComponentIdentification) fields.get("sourceComponentId", sourceComponentId);

		Object tempArray[] = null;
		tempArray = (Object[]) fields.get("contextDataElements", null);
		if (tempArray != null) {
			int arraySize = tempArray.length;
			for (int i = 0; i < arraySize; i++) {
				addContextDataElement((IContextDataElement) tempArray[i]);
			}
		}

		tempArray = null;
		tempArray = (Object[]) fields.get("extendedDataElements", null);
		if (tempArray != null) {
			int arraySize = tempArray.length;
			for (int i = 0; i < arraySize; i++) {
				IExtendedDataElement temp = (IExtendedDataElement) tempArray[i];
				addExtendedDataElement(temp);
			}
		}

		tempArray = null;
		tempArray = (Object[]) fields.get("associatedEvents", null);
		if (tempArray != null) {
			int arraySize = tempArray.length;
			for (int i = 0; i < arraySize; i++) {
				addAssociatedEvent((IAssociatedEvent) tempArray[i]);
			}
		}

		tempArray = null;
		tempArray = (Object[]) fields.get("anyElements", null);
		if (tempArray != null) {
			int arraySize = tempArray.length;
			for (int i = 0; i < arraySize; i++) {
				ICommonBaseEventAnyElement anyElement = new CommonBaseEventAnyElementImpl();
				try {
					anyElement.setXmlAnyElement((String) tempArray[i]);
				}
				catch (InvalidXMLAnyElementException exp) {
					throw new IOException(exp.getMessage());
				}
				addAnyElement(anyElement);
			}
		}
	}

	/**
	 * return a cloned (copy by value) object. This is a deep copy version, in which all the
	 * objects within this object will be copied by value.
	 *  
	 * @return Object
	 */
	public Object clone() throws CloneNotSupportedException {

		CommonBaseEventImpl copy = ((CommonBaseEventImpl) (super.clone()));
		Object tempElement = null;

		//Clone the contextDataElements array list:

		if (this.contextDataElements != null) {
			copy.contextDataElements = ((ArrayList) (this.contextDataElements.clone()));
			for (int ctr = 0; ctr < this.contextDataElements.size(); ctr++) {
				tempElement = contextDataElements.get(ctr);
				/* if the Elements in contextDataElements arrayList are some other implementations,
				 * it will only be shallow cloned.
				 */
				if (tempElement instanceof ContextDataElementImpl) {
					copy.contextDataElements.set(ctr, ((IContextDataElement) ((ContextDataElementImpl) (tempElement)).clone()));
				}
			}
		}

		//Clone the extendedDataElements array list:
		if (this.extendedDataElements != null) {
			copy.extendedDataElements = ((ArrayList) (this.extendedDataElements.clone()));
			for (int ctr = 0; ctr < this.extendedDataElements.size(); ctr++) {
				tempElement = extendedDataElements.get(ctr);
				/* if the Elements in extendedDataElements arrayList are some other implementations,
				 * it will only be shallow cloned.
				 */
				if (tempElement instanceof ExtendedDataElementImpl) {
					copy.extendedDataElements.set(ctr, ((IExtendedDataElement) ((ExtendedDataElementImpl) (tempElement)).clone()));
				}
			}
		}

		//Clone the anyElements array list:
		if (this.anyElements != null) {
			copy.anyElements = ((ArrayList) (this.anyElements.clone()));
			for (int ctr = 0; ctr < this.anyElements.size(); ctr++) {
				tempElement = anyElements.get(ctr);
				/* if the Elements in anyElements arrayList are some other implementations,
				 * it will only be shallow cloned.
				 */
				if (tempElement instanceof CommonBaseEventAnyElementImpl) {
					copy.anyElements.set(ctr, ((ICommonBaseEventAnyElement) ((CommonBaseEventAnyElementImpl) (tempElement)).clone()));
				}
			}
		}

		//Clone the associatedEvents array list:
		if (this.associatedEvents != null) {
			copy.associatedEvents = ((ArrayList) (this.associatedEvents.clone()));
			for (int ctr = 0; ctr < this.associatedEvents.size(); ctr++) {
				tempElement = associatedEvents.get(ctr);
				/* if the Elements in associatedEvents arrayList are some other implementations,
				 * it will only be shallow cloned.
				 */
				if (tempElement instanceof AssociatedEventImpl) {
					copy.associatedEvents.set(ctr, ((IAssociatedEvent) ((AssociatedEventImpl) (tempElement)).clone()));
				}
			}
		}

		if (this.msgDataElement instanceof MsgDataElementImpl) {
			copy.msgDataElement = ((IMsgDataElement) ((MsgDataElementImpl) (this.msgDataElement)).clone());
		}

		if (this.situation instanceof SituationImpl) {
			copy.situation = ((ISituation) ((SituationImpl) (this.situation)).clone());
		}

		if (this.sourceComponentId instanceof ComponentIdentificationImpl) {
			copy.sourceComponentId = ((IComponentIdentification) ((ComponentIdentificationImpl) (this.sourceComponentId)).clone());
		}

		if (this.reporterComponentId instanceof ComponentIdentificationImpl) {
			copy.reporterComponentId = ((IComponentIdentification) ((ComponentIdentificationImpl) (this.reporterComponentId)).clone());
		}

		if (this.otherData != null)
			copy.otherData = ((String[]) (this.otherData.clone()));

		return copy;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addAssociatedEvent(java.lang.String, java.lang.String, java.lang.String, java.lang.String[])
	 */
	public IAssociatedEvent addAssociatedEvent(String associationEngineName, String associationEngineType, String associationEngineId, String[] resolvedEvents) {

		IAssociationEngine associationEngine = new AssociationEngineImpl();
		associationEngine.setName(associationEngineName);
		associationEngine.setType(associationEngineType);
		associationEngine.setId(associationEngineId);

		IAssociatedEvent associatedEvent = new AssociatedEventImpl();
		associatedEvent.setAssociationEngineInfo(associationEngine);
		associatedEvent.setResolvedEventsAsArray(resolvedEvents);

		addAssociatedEvent(associatedEvent);

		return associatedEvent;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addContextDataElementWithId(java.lang.String, java.lang.String, java.lang.String)
	 */
	public IContextDataElement addContextDataElementWithId(String contextType, String contextName, String contextId) {

		IContextDataElement contextDataElement = new ContextDataElementImpl();
		contextDataElement.setType(contextType);
		contextDataElement.setName(contextName);
		contextDataElement.setContextId(contextId);

		addContextDataElement(contextDataElement);

		return contextDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addContextDataElementWithValue(java.lang.String, java.lang.String, java.lang.String)
	 */
	public IContextDataElement addContextDataElementWithValue(String contextType, String contextName, String contextValue) {

		IContextDataElement contextDataElement = new ContextDataElementImpl();
		contextDataElement.setType(contextType);
		contextDataElement.setName(contextName);
		contextDataElement.setContextValue(contextValue);

		addContextDataElement(contextDataElement);

		return contextDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElement(java.lang.String, int, java.lang.String)
	 */
	public IExtendedDataElement addExtendedDataElement(String name, int type, String value) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(type);
		extendedDataElement.setValues(new String[] { value });

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElement(java.lang.String, int, java.lang.String[])
	 */
	public IExtendedDataElement addExtendedDataElement(String name, int type, String[] values) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(type);
		extendedDataElement.setValues(values);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElement(java.lang.String, java.lang.String)
	 */
	public IExtendedDataElement addExtendedDataElement(String name, String value) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_STRING_VALUE);
		extendedDataElement.setValues(new String[] { value });

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElement(java.lang.String, java.lang.String[])
	 */
	public IExtendedDataElement addExtendedDataElement(String name, String[] values) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_STRING_ARRAY_VALUE);
		extendedDataElement.setValues(values);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithByteArrayValue(java.lang.String, byte[])
	 */
	public IExtendedDataElement addExtendedDataElementWithByteArrayValue(String name, byte[] values) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_BYTE_ARRAY_VALUE);

		String[] stringValues = new String[values.length];

		for (int counter = 0; counter < stringValues.length; counter++) {
			stringValues[counter] = String.valueOf(values[counter]);
		}

		extendedDataElement.setValues(stringValues);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithByteValue(java.lang.String, byte)
	 */
	public IExtendedDataElement addExtendedDataElementWithByteValue(String name, byte value) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_BYTE_VALUE);
		extendedDataElement.setValues(new String[] { String.valueOf(value)});

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithBooleanArrayValue(java.lang.String, boolean[])
	 */
	public IExtendedDataElement addExtendedDataElementWithBooleanArrayValue(String name, boolean[] values) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_BOOLEAN_ARRAY_VALUE);

		String[] stringValues = new String[values.length];

		for (int counter = 0; counter < stringValues.length; counter++) {
			stringValues[counter] = String.valueOf(values[counter]);
		}

		extendedDataElement.setValues(stringValues);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithBooleanValue(java.lang.String, boolean)
	 */
	public IExtendedDataElement addExtendedDataElementWithBooleanValue(String name, boolean value) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_BOOLEAN_VALUE);
		extendedDataElement.setValues(new String[] { String.valueOf(value)});

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithDateArrayValue(java.lang.String, java.lang.String[])
	 */
	public IExtendedDataElement addExtendedDataElementWithDateArrayValue(String name, String[] values) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);

		String[] stringValues = new String[values.length];

		for (int counter = 0; counter < stringValues.length; counter++) {
			stringValues[counter] = String.valueOf(values[counter]);
		}

		extendedDataElement.setValues(stringValues);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithDateAsLongValue(java.lang.String, long)
	 */
	public IExtendedDataElement addExtendedDataElementWithDateAsLongValue(String name, long value) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_DATE_TIME_VALUE);
		extendedDataElement.setValues(new String[] { CbeFormatter.convertDateToXmlSchemaDateTime(value)});

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithDatesAsLongValue(java.lang.String, long[])
	 */
	public IExtendedDataElement addExtendedDataElementWithDatesAsLongValue(String name, long[] values) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);

		String[] stringValues = new String[values.length];

		for (int counter = 0; counter < stringValues.length; counter++) {
			stringValues[counter] = CbeFormatter.convertDateToXmlSchemaDateTime(values[counter]);
		}

		extendedDataElement.setValues(stringValues);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithDateValue(java.lang.String, java.lang.String)
	 */
	public IExtendedDataElement addExtendedDataElementWithDateValue(String name, String value) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_DATE_TIME_VALUE);
		extendedDataElement.setValues(new String[] { String.valueOf(value)});

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithDoubleArrayValue(java.lang.String, double[])
	 */
	public IExtendedDataElement addExtendedDataElementWithDoubleArrayValue(String name, double[] values) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_DOUBLE_ARRAY_VALUE);

		String[] stringValues = new String[values.length];

		for (int counter = 0; counter < stringValues.length; counter++) {
			stringValues[counter] = String.valueOf(values[counter]);
		}

		extendedDataElement.setValues(stringValues);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithDoubleValue(java.lang.String, double)
	 */
	public IExtendedDataElement addExtendedDataElementWithDoubleValue(String name, double value) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_DOUBLE_VALUE);
		extendedDataElement.setValues(new String[] { String.valueOf(value)});

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithFloatArrayValue(java.lang.String, float[])
	 */
	public IExtendedDataElement addExtendedDataElementWithFloatArrayValue(String name, float[] values) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_FLOAT_ARRAY_VALUE);

		String[] stringValues = new String[values.length];

		for (int counter = 0; counter < stringValues.length; counter++) {
			stringValues[counter] = String.valueOf(values[counter]);
		}

		extendedDataElement.setValues(stringValues);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithFloatValue(java.lang.String, float)
	 */
	public IExtendedDataElement addExtendedDataElementWithFloatValue(String name, float value) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_FLOAT_VALUE);
		extendedDataElement.setValues(new String[] { String.valueOf(value)});

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithHexValue(java.lang.String, byte[])
	 */
	public IExtendedDataElement addExtendedDataElementWithHexValue(String name, byte[] hexValue) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_HEX_BINARY_VALUE);

		char[] hexValues = new char[hexValue.length];

		for (int counter = 0; counter < hexValues.length; counter++) {
			hexValues[counter] = ((char) ((hexValue[counter] + 128)));
		}

		extendedDataElement.setHexValues(hexValues);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithHexValue(java.lang.String, java.lang.String)
	 */
	public IExtendedDataElement addExtendedDataElementWithHexValue(String name, String hexValue) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_HEX_BINARY_VALUE);

		char[] hexValues = new char[hexValue.length()];

		for (int counter = 0; counter < hexValues.length; counter++) {
			hexValues[counter] = hexValue.charAt(counter);
		}

		extendedDataElement.setHexValues(hexValues);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithIntArrayValue(java.lang.String, int[])
	 */
	public IExtendedDataElement addExtendedDataElementWithIntArrayValue(String name, int[] values) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_INT_ARRAY_VALUE);

		String[] stringValues = new String[values.length];

		for (int counter = 0; counter < stringValues.length; counter++) {
			stringValues[counter] = String.valueOf(values[counter]);
		}

		extendedDataElement.setValues(stringValues);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithIntValue(java.lang.String, int)
	 */
	public IExtendedDataElement addExtendedDataElementWithIntValue(String name, int value) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_INT_VALUE);
		extendedDataElement.setValues(new String[] { String.valueOf(value)});

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithLongArrayValue(java.lang.String, long[])
	 */
	public IExtendedDataElement addExtendedDataElementWithLongArrayValue(String name, long[] values) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_LONG_ARRAY_VALUE);

		String[] stringValues = new String[values.length];

		for (int counter = 0; counter < stringValues.length; counter++) {
			stringValues[counter] = String.valueOf(values[counter]);
		}

		extendedDataElement.setValues(stringValues);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithLongValue(java.lang.String, long)
	 */
	public IExtendedDataElement addExtendedDataElementWithLongValue(String name, long value) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_LONG_VALUE);
		extendedDataElement.setValues(new String[] { String.valueOf(value)});

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithShortArrayValue(java.lang.String, short[])
	 */
	public IExtendedDataElement addExtendedDataElementWithShortArrayValue(String name, short[] values) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_SHORT_ARRAY_VALUE);

		String[] stringValues = new String[values.length];

		for (int counter = 0; counter < stringValues.length; counter++) {
			stringValues[counter] = String.valueOf(values[counter]);
		}

		extendedDataElement.setValues(stringValues);

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#addExtendedDataElementWithShortValue(java.lang.String, short)
	 */
	public IExtendedDataElement addExtendedDataElementWithShortValue(String name, short value) {

		IExtendedDataElement extendedDataElement = new ExtendedDataElementImpl();
		extendedDataElement.setName(name);
		extendedDataElement.setType(IExtendedDataElement.TYPE_SHORT_VALUE);
		extendedDataElement.setValues(new String[] { String.valueOf(value)});

		addExtendedDataElement(extendedDataElement);

		return extendedDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#isSetCreationTime()
	 */
	public boolean isSetCreationTime() {
		return creationTimeIsSet;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#isSetElapsedTime()
	 */
	public boolean isSetElapsedTime() {
		return elapsedTimeIsSet;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#isSetPriority()
	 */
	public boolean isSetPriority() {
		return priorityIsSet;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#isSetRepeatCount()
	 */
	public boolean isSetRepeatCount() {
		return repeatCountIsSet;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#isSetSequenceNumber()
	 */
	public boolean isSetSequenceNumber() {
		return sequenceNumberIsSet;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#isSetSeverity()
	 */
	public boolean isSetSeverity() {
		return severityIsSet;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setMsgDataElement(java.lang.String, java.lang.String, org.eclipse.hyades.logging.events.IMsgCatalogToken[], java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public IMsgDataElement setMsgDataElement(String msgCatalog, String msgCatalogId, IMsgCatalogToken[] tokens, String msgCatalogType, String msgId, String msgIdType, String msgCatalogLocale) {

		IMsgDataElement msgDataElement = new MsgDataElementImpl();
		msgDataElement.setMsgCatalog(msgCatalog);
		msgDataElement.setMsgCatalogId(msgCatalogId);
		msgDataElement.setMsgCatalogTokens(tokens);
		msgDataElement.setMsgCatalogType(msgCatalogType);
		msgDataElement.setMsgId(msgId);
		msgDataElement.setMsgIdType(msgIdType);
		msgDataElement.setMsgLocale(msgCatalogLocale);

		setMsgDataElement(msgDataElement);

		return msgDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setMsgDataElement(java.lang.String, java.lang.String, java.lang.String[], java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public IMsgDataElement setMsgDataElement(String msgCatalog, String msgCatalogId, String[] tokens, String msgCatalogType, String msgId, String msgIdType, String msgCatalogLocale) {

		IMsgDataElement msgDataElement = new MsgDataElementImpl();
		msgDataElement.setMsgCatalog(msgCatalog);
		msgDataElement.setMsgCatalogId(msgCatalogId);
		msgDataElement.setMsgCatalogTokensAsStrings(tokens);
		msgDataElement.setMsgCatalogType(msgCatalogType);
		msgDataElement.setMsgId(msgId);
		msgDataElement.setMsgIdType(msgIdType);
		msgDataElement.setMsgLocale(msgCatalogLocale);

		setMsgDataElement(msgDataElement);

		return msgDataElement;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setReporterComponentId(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public IComponentIdentification setReporterComponentId(String application, String component, String subComponent, String componentType, String componentIdType, String executionEnvironment, String instanceId, String location, String locationType, String processId, String threadId) {

		IComponentIdentification reporterComponentId = new ComponentIdentificationImpl();
		reporterComponentId.setApplication(application);
		reporterComponentId.setComponent(component);
		reporterComponentId.setSubComponent(subComponent);
		reporterComponentId.setComponentType(componentType);
		reporterComponentId.setComponentIdType(componentIdType);
		reporterComponentId.setExecutionEnvironment(executionEnvironment);
		reporterComponentId.setInstanceId(instanceId);
		reporterComponentId.setLocation(location);
		reporterComponentId.setLocationType(locationType);
		reporterComponentId.setProcessId(processId);
		reporterComponentId.setThreadId(threadId);

		setReporterComponentId(reporterComponentId);

		return reporterComponentId;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setReporterComponentId(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public IComponentIdentification setReporterComponentId(String application, String component, String subComponent, String componentType, String componentIdType, String location, String locationType) {

		IComponentIdentification reporterComponentId = new ComponentIdentificationImpl();
		reporterComponentId.setApplication(application);
		reporterComponentId.setComponent(component);
		reporterComponentId.setSubComponent(subComponent);
		reporterComponentId.setComponentType(componentType);
		reporterComponentId.setComponentIdType(componentIdType);
		reporterComponentId.setLocation(location);
		reporterComponentId.setLocationType(locationType);

		setReporterComponentId(reporterComponentId);

		return reporterComponentId;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setSourceComponentId(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public IComponentIdentification setSourceComponentId(String application, String component, String subComponent, String componentType, String componentIdType, String executionEnvironment, String instanceId, String location, String locationType, String processId, String threadId) {

		IComponentIdentification sourceComponentId = new ComponentIdentificationImpl();
		sourceComponentId.setApplication(application);
		sourceComponentId.setComponent(component);
		sourceComponentId.setSubComponent(subComponent);
		sourceComponentId.setComponentType(componentType);
		sourceComponentId.setComponentIdType(componentIdType);
		sourceComponentId.setExecutionEnvironment(executionEnvironment);
		sourceComponentId.setInstanceId(instanceId);
		sourceComponentId.setLocation(location);
		sourceComponentId.setLocationType(locationType);
		sourceComponentId.setProcessId(processId);
		sourceComponentId.setThreadId(threadId);

		setSourceComponentId(sourceComponentId);

		return sourceComponentId;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#setSourceComponentId(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public IComponentIdentification setSourceComponentId(String application, String component, String subComponent, String componentType, String componentIdType, String location, String locationType) {

		IComponentIdentification sourceComponentId = new ComponentIdentificationImpl();
		sourceComponentId.setApplication(application);
		sourceComponentId.setComponent(component);
		sourceComponentId.setSubComponent(subComponent);
		sourceComponentId.setComponentType(componentType);
		sourceComponentId.setComponentIdType(componentIdType);
		sourceComponentId.setLocation(location);
		sourceComponentId.setLocationType(locationType);

		setSourceComponentId(sourceComponentId);

		return sourceComponentId;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#unsetCreationTime()
	 */
	public void unsetCreationTime() {
		creationTimeValue = 0;
		creationTime = null;
		creationTimeIsSet = false;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#unsetElapsedTime()
	 */
	public void unsetElapsedTime() {
		elapsedTimeIsSet = false;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#unsetPriority()
	 */
	public void unsetPriority() {
		priorityIsSet = false;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#unsetRepeatCount()
	 */
	public void unsetRepeatCount() {
		repeatCountIsSet = false;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#unsetSequenceNumber()
	 */
	public void unsetSequenceNumber() {
		sequenceNumberIsSet = false;
	}

	/**
	 * @see org.eclipse.hyades.logging.events.ICommonBaseEvent#unsetSeverity()
	 */
	public void unsetSeverity() {
		severityIsSet = false;
	}
}
