package org.eclipse.hyades.logging.java.tests;

import java.util.Enumeration;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import junit.framework.AssertionFailedError;
import junit.framework.TestCase;

import org.eclipse.hyades.internal.logging.core.Constants;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;
import org.eclipse.hyades.logging.java.CommonBaseEventLogRecord;
import org.eclipse.hyades.logging.java.XmlFormatter;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XmlFormatterTest.java,v 1.5 2005/04/19 01:48:00 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * All JUnit test cases for the
 * <code>org.eclipse.hyades.logging.java.XmlFormatter</code>
 * class.
 * <p>
 * 
 * @author  Paul E. Slauenwhite
 * @version April 18, 2005
 * @since   October 27, 2004
 */
public class XmlFormatterTest extends TestCase {

    private EventFactory eventFactory = null;

    /**
     * Private static reference to the platform-dependent line separator character.
     */
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");
   
    /**
     * Constructor for CommonBaseEventLogRecordTest.
     * <p>
     * 
     * @param name
     *            The name of the test case.
     */
    public XmlFormatterTest(String name) {
        super(name);
    }

    /**
     * @see junit.framework.TestCase#setUp()
     */
    protected void setUp() throws Exception {

        super.setUp();

        eventFactory = EventFactoryContext.getInstance().getSimpleEventFactoryHome().getAnonymousEventFactory();
    }

    /**
     * Tests formatting a Common Base Event Log Record.
     */
    public void testXmlFormattingCommonBaseEventLogRecord() {

        try {
            
            LogRecord tempLogRecord = new LogRecord(Level.OFF, null);

            final long SEQUENCE_NUMBER = (tempLogRecord.getSequenceNumber() + 1);

            final int THREAD_ID = tempLogRecord.getThreadID();

            final long CURRENT_TIME = System.currentTimeMillis();

            final String LOGGER_NAME = "Logger Name";

            final String MESSAGE = "Message";

            final String[] PARAMETERS = new String[] { "Paramater1", "Paramater2", "Paramater3"};

            final ResourceBundle RESOURCE_BUNDLE = new ResourceBundle() {

                public Enumeration getKeys() {
                    return null;
                }

                public Locale getLocale() {
                    return null;
                }

                protected Object handleGetObject(String key) {
                    return null;
                }

                protected void setParent(ResourceBundle parent) {
                }
            };

            final String RESOURCE_BUNDLE_NAME = "ResourceBundleName";

            final String SOURCE_CLASS_NAME = "Source Class Name";

            final String SOURCE_METHOD_NAME = "Source Method Name";

            final int THROWABLES_LEVELS = 100;

            Throwable throwable = null;
            Throwable child = new Throwable("ChildThrowable" + THROWABLES_LEVELS);

            for (int counter = (THROWABLES_LEVELS - 1); counter > 0; counter--) {

                throwable = new Throwable("ChildThrowable" + counter, child);
                child = throwable;
            }

            throwable = new Throwable("Throwable", child);

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            commonBaseEventLogRecord.setSequenceNumber(SEQUENCE_NUMBER);

            commonBaseEventLogRecord.setThreadID(THREAD_ID);

            commonBaseEventLogRecord.setMillis(CURRENT_TIME);

            commonBaseEventLogRecord.setLevel(Level.ALL);

            commonBaseEventLogRecord.setLoggerName(LOGGER_NAME);

            commonBaseEventLogRecord.setMessage(MESSAGE);
        
            commonBaseEventLogRecord.setParameters(PARAMETERS);

            commonBaseEventLogRecord.setResourceBundleName(RESOURCE_BUNDLE_NAME);

            commonBaseEventLogRecord.setResourceBundle(RESOURCE_BUNDLE);

            commonBaseEventLogRecord.setSourceClassName(SOURCE_CLASS_NAME);

            commonBaseEventLogRecord.setSourceMethodName(SOURCE_METHOD_NAME);

            commonBaseEventLogRecord.setThrown(throwable);

            String commonBaseEventXML = EventFormatter.toCanonicalXMLString(commonBaseEventLogRecord.getCommonBaseEvent(), true).concat(LINE_SEPARATOR);

            String formattedCommonBaseEventXML = new XmlFormatter().format(commonBaseEventLogRecord);
            
            assertEquals(commonBaseEventXML,formattedCommonBaseEventXML);
            
            try {
                assertNotNull(EventFormatter.eventFromCanonicalXML(commonBaseEventXML,true));
            } 
            catch (Exception e) {
                fail("Invalid Common Base Event XML.");
            }    
        } 
        catch (AssertionFailedError a) {
            throw (a);
        } 
        catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Tests formatting a Log Record.
     */
    public void testXmlFormattingLogRecord() {
        try {
            
            LogRecord tempLogRecord = new LogRecord(Level.OFF, null);

            final long SEQUENCE_NUMBER = (tempLogRecord.getSequenceNumber() + 1);

            final int THREAD_ID = tempLogRecord.getThreadID();

            final long CURRENT_TIME = System.currentTimeMillis();

            final String LOGGER_NAME = "Logger Name";

            final String MESSAGE = "Message";

            final String[] PARAMETERS = new String[] { "Paramater1", "Paramater2", "Paramater3"};

            final ResourceBundle RESOURCE_BUNDLE = new ResourceBundle() {

                public Enumeration getKeys() {
                    return null;
                }

                public Locale getLocale() {
                    return null;
                }

                protected Object handleGetObject(String key) {
                    return null;
                }

                protected void setParent(ResourceBundle parent) {
                }
            };

            final String RESOURCE_BUNDLE_NAME = "ResourceBundleName";

            final String SOURCE_CLASS_NAME = "Source Class Name";

            final String SOURCE_METHOD_NAME = "Source Method Name";

            final int THROWABLES_LEVELS = 100;

            Throwable throwable = null;
            Throwable child = new Throwable("ChildThrowable" + THROWABLES_LEVELS);

            for (int counter = (THROWABLES_LEVELS - 1); counter > 0; counter--) {

                throwable = new Throwable("ChildThrowable" + counter, child);
                child = throwable;
            }

            throwable = new Throwable("Throwable", child);

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            commonBaseEventLogRecord.setSequenceNumber(SEQUENCE_NUMBER);

            commonBaseEventLogRecord.setThreadID(THREAD_ID);

            commonBaseEventLogRecord.setMillis(CURRENT_TIME);

            commonBaseEventLogRecord.setLevel(Level.ALL);

            commonBaseEventLogRecord.setLoggerName(LOGGER_NAME);

            commonBaseEventLogRecord.setMessage(MESSAGE);
        
            commonBaseEventLogRecord.setParameters(PARAMETERS);

            commonBaseEventLogRecord.setResourceBundleName(RESOURCE_BUNDLE_NAME);

            commonBaseEventLogRecord.setResourceBundle(RESOURCE_BUNDLE);

            commonBaseEventLogRecord.setSourceClassName(SOURCE_CLASS_NAME);

            commonBaseEventLogRecord.setSourceMethodName(SOURCE_METHOD_NAME);

            commonBaseEventLogRecord.setThrown(throwable);

            LogRecord logRecord = new LogRecord(Level.ALL,MESSAGE);

            logRecord.setMillis(CURRENT_TIME);

            logRecord.setLoggerName(LOGGER_NAME);

            logRecord.setSequenceNumber(SEQUENCE_NUMBER);

            logRecord.setParameters(PARAMETERS);

            logRecord.setResourceBundle(RESOURCE_BUNDLE);

            logRecord.setResourceBundleName(RESOURCE_BUNDLE_NAME);

            logRecord.setSourceClassName(SOURCE_CLASS_NAME);

            logRecord.setSourceMethodName(SOURCE_METHOD_NAME);

            logRecord.setThreadID(THREAD_ID);
            
            logRecord.setThrown(throwable);

            String commonBaseEventXML = EventFormatter.toCanonicalXMLString(commonBaseEventLogRecord.getCommonBaseEvent(), true).concat(LINE_SEPARATOR);

            String formattedLogRecordXML = new XmlFormatter().format(logRecord);
            
            assertEquals(commonBaseEventXML,formattedLogRecordXML);

            try {
                assertNotNull(EventFormatter.eventFromCanonicalXML(commonBaseEventXML,true));
            } 
            catch (Exception e) {
                fail("Invalid Common Base Event XML.");
            }    
        } 
        catch (AssertionFailedError a) {
            throw (a);
        } 
        catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }  
    
    /**
     * Tests formatting a Log Record using a Common Base Event XML template.
     */
    public void testXmlFormattingLogRecordWithCBEXMLTemplate() {
        
        fail("NOTE: Add 'org.eclipse.hyades.logging.java.tests.XmlFormatterTest.event.xml' to the classpath.");
        
        try {
            
            LogRecord tempLogRecord = new LogRecord(Level.OFF, null);

            final long SEQUENCE_NUMBER = (tempLogRecord.getSequenceNumber() + 1);

            final int THREAD_ID = tempLogRecord.getThreadID();

            final long CURRENT_TIME = System.currentTimeMillis();

            final String LOGGER_NAME = "org.eclipse.hyades.logging.java.tests.XmlFormatterTest";
            final String MESSAGE = "Message";

            final String[] PARAMETERS = new String[] { "Paramater1", "Paramater2", "Paramater3"};

            final ResourceBundle RESOURCE_BUNDLE = new ResourceBundle() {

                public Enumeration getKeys() {
                    return null;
                }

                public Locale getLocale() {
                    return null;
                }

                protected Object handleGetObject(String key) {
                    return null;
                }

                protected void setParent(ResourceBundle parent) {
                }
            };

            final String RESOURCE_BUNDLE_NAME = "ResourceBundleName";

            final String SOURCE_CLASS_NAME = "Source Class Name";

            final String SOURCE_METHOD_NAME = "Source Method Name";

            final int THROWABLES_LEVELS = 100;

            Throwable throwable = null;
            Throwable child = new Throwable("ChildThrowable" + THROWABLES_LEVELS);

            for (int counter = (THROWABLES_LEVELS - 1); counter > 0; counter--) {

                throwable = new Throwable("ChildThrowable" + counter, child);
                child = throwable;
            }

            throwable = new Throwable("Throwable", child);

            LogRecord logRecord = new LogRecord(Level.ALL,MESSAGE);

            logRecord.setMillis(CURRENT_TIME);

            logRecord.setLoggerName(LOGGER_NAME);

            logRecord.setSequenceNumber(SEQUENCE_NUMBER);

            logRecord.setParameters(PARAMETERS);

            logRecord.setResourceBundle(RESOURCE_BUNDLE);

            logRecord.setResourceBundleName(RESOURCE_BUNDLE_NAME);

            logRecord.setSourceClassName(SOURCE_CLASS_NAME);

            logRecord.setSourceMethodName(SOURCE_METHOD_NAME);

            logRecord.setThreadID(THREAD_ID);
            
            logRecord.setThrown(throwable);

            String formattedLogRecordXML = new XmlFormatter().format(logRecord);
            
            CommonBaseEvent commonBaseEvent = null;
            
            try {
                commonBaseEvent = EventFormatter.eventFromCanonicalXML(formattedLogRecordXML,true);
            } 
            catch (Exception e) {
                fail("Invalid Common Base Event XML.");
            }    

            assertNotNull(commonBaseEvent);
            
            assertNotNull(commonBaseEvent.getLocalInstanceId());

            assertEquals(commonBaseEvent.getLocalInstanceId(),"Sample Event");

            assertNotNull(commonBaseEvent.getSourceComponentId());

            assertNotNull(commonBaseEvent.getSourceComponentId().getComponent());

            assertEquals(commonBaseEvent.getSourceComponentId().getComponent(),"XmlFormatterTest");

            assertNotNull(commonBaseEvent.getSourceComponentId().getComponentIdType());

            assertEquals(commonBaseEvent.getSourceComponentId().getComponentIdType(),"Application");

            assertNotNull(commonBaseEvent.getSourceComponentId().getExecutionEnvironment());

            assertEquals(commonBaseEvent.getSourceComponentId().getExecutionEnvironment(),"Java");

            assertNotNull(commonBaseEvent.getSourceComponentId().getLocation());

            assertEquals(commonBaseEvent.getSourceComponentId().getLocation(),"127.0.0.1");

            assertNotNull(commonBaseEvent.getSourceComponentId().getLocationType());

            assertEquals(commonBaseEvent.getSourceComponentId().getLocationType(),"IPV4");

            assertNotNull(commonBaseEvent.getSourceComponentId().getSubComponent());

            assertEquals(commonBaseEvent.getSourceComponentId().getSubComponent(),"testXmlFormattingLogRecordWithCBEXMLTemplate");

            assertNotNull(commonBaseEvent.getSourceComponentId().getComponentType());

            assertEquals(commonBaseEvent.getSourceComponentId().getComponentType(),"Java_Application");

            assertNull(commonBaseEvent.getSourceComponentId().getInstanceId());
            assertNull(commonBaseEvent.getSourceComponentId().getProcessId());
            assertNull(commonBaseEvent.getSourceComponentId().getThreadId());
                        
            assertNotNull(commonBaseEvent.getSituation());

            assertNotNull(commonBaseEvent.getSituation().getCategoryName());

            assertEquals(commonBaseEvent.getSituation().getCategoryName(),Situation.REPORT_SITUATION_CATEGORY);

            assertNotNull(commonBaseEvent.getSituation().getSituationType());

            assertTrue(commonBaseEvent.getSituation().getSituationType() instanceof ReportSituation);

            assertNotNull(((ReportSituation)(commonBaseEvent.getSituation().getSituationType())).getReasoningScope());

            assertEquals(((ReportSituation)(commonBaseEvent.getSituation().getSituationType())).getReasoningScope(),"INTERNAL");

            assertNotNull(((ReportSituation)(commonBaseEvent.getSituation().getSituationType())).getReportCategory());

            assertEquals(((ReportSituation)(commonBaseEvent.getSituation().getSituationType())).getReportCategory(),"LOG");
        } 
        catch (AssertionFailedError a) {
            throw (a);
        } 
        catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }  
    
    /**
     * Tests formatting a Log Record without using a Common Base Event XML template.
     */
    public void testXmlFormattingLogRecordWithoutCBEXMLTemplate() {
        
        try {
            
            LogRecord tempLogRecord = new LogRecord(Level.OFF, null);

            final long SEQUENCE_NUMBER = (tempLogRecord.getSequenceNumber() + 1);

            final int THREAD_ID = tempLogRecord.getThreadID();

            final long CURRENT_TIME = System.currentTimeMillis();

            final String LOGGER_NAME = "org.eclipse.hyades.logging.java.tests.XmlFormatterTest_XXXX";
            final String MESSAGE = "Message";

            final String[] PARAMETERS = new String[] { "Paramater1", "Paramater2", "Paramater3"};

            final ResourceBundle RESOURCE_BUNDLE = new ResourceBundle() {

                public Enumeration getKeys() {
                    return null;
                }

                public Locale getLocale() {
                    return null;
                }

                protected Object handleGetObject(String key) {
                    return null;
                }

                protected void setParent(ResourceBundle parent) {
                }
            };

            final String RESOURCE_BUNDLE_NAME = "ResourceBundleName";

            final String SOURCE_CLASS_NAME = "Source Class Name";

            final String SOURCE_METHOD_NAME = "Source Method Name";

            final int THROWABLES_LEVELS = 100;

            Throwable throwable = null;
            Throwable child = new Throwable("ChildThrowable" + THROWABLES_LEVELS);

            for (int counter = (THROWABLES_LEVELS - 1); counter > 0; counter--) {

                throwable = new Throwable("ChildThrowable" + counter, child);
                child = throwable;
            }

            throwable = new Throwable("Throwable", child);

            LogRecord logRecord = new LogRecord(Level.ALL,MESSAGE);

            logRecord.setMillis(CURRENT_TIME);

            logRecord.setLoggerName(LOGGER_NAME);

            logRecord.setSequenceNumber(SEQUENCE_NUMBER);

            logRecord.setParameters(PARAMETERS);

            logRecord.setResourceBundle(RESOURCE_BUNDLE);

            logRecord.setResourceBundleName(RESOURCE_BUNDLE_NAME);

            logRecord.setSourceClassName(SOURCE_CLASS_NAME);

            logRecord.setSourceMethodName(SOURCE_METHOD_NAME);

            logRecord.setThreadID(THREAD_ID);
            
            logRecord.setThrown(throwable);

            String formattedLogRecordXML = new XmlFormatter().format(logRecord);
            
            CommonBaseEvent commonBaseEvent = null;
            
            try {
                commonBaseEvent = EventFormatter.eventFromCanonicalXML(formattedLogRecordXML,true);
            } 
            catch (Exception e) {
                fail("Invalid Common Base Event XML.");
            }    

            assertNotNull(commonBaseEvent);
            
            assertNull(commonBaseEvent.getLocalInstanceId());

            assertNotNull(commonBaseEvent.getSourceComponentId());

            assertNotNull(commonBaseEvent.getSourceComponentId().getComponent());

            assertEquals(commonBaseEvent.getSourceComponentId().getComponent(),"Logging");

            assertNotNull(commonBaseEvent.getSourceComponentId().getComponentIdType());

            assertEquals(commonBaseEvent.getSourceComponentId().getComponentIdType(),"Application");

            assertNotNull(commonBaseEvent.getSourceComponentId().getExecutionEnvironment());

            assertEquals(commonBaseEvent.getSourceComponentId().getExecutionEnvironment(),"Java");

            assertNotNull(commonBaseEvent.getSourceComponentId().getLocation());

            assertEquals(commonBaseEvent.getSourceComponentId().getLocation(),Constants.LOCAL_HOST_IP_ADDRESS);

            assertNotNull(commonBaseEvent.getSourceComponentId().getLocationType());

            assertEquals(commonBaseEvent.getSourceComponentId().getLocationType(),"IPV4");

            assertNotNull(commonBaseEvent.getSourceComponentId().getSubComponent());

            assertEquals(commonBaseEvent.getSourceComponentId().getSubComponent(),"Logger");

            assertNotNull(commonBaseEvent.getSourceComponentId().getComponentType());

            assertEquals(commonBaseEvent.getSourceComponentId().getComponentType(),"Logging_Application");

            assertNull(commonBaseEvent.getSourceComponentId().getInstanceId());
            assertNull(commonBaseEvent.getSourceComponentId().getProcessId());
            assertNull(commonBaseEvent.getSourceComponentId().getThreadId());
                        
            assertNotNull(commonBaseEvent.getSituation());

            assertNotNull(commonBaseEvent.getSituation().getCategoryName());

            assertEquals(commonBaseEvent.getSituation().getCategoryName(),Situation.REPORT_SITUATION_CATEGORY);

            assertNotNull(commonBaseEvent.getSituation().getSituationType());

            assertTrue(commonBaseEvent.getSituation().getSituationType() instanceof ReportSituation);

            assertNotNull(((ReportSituation)(commonBaseEvent.getSituation().getSituationType())).getReasoningScope());

            assertEquals(((ReportSituation)(commonBaseEvent.getSituation().getSituationType())).getReasoningScope(),"INTERNAL");

            assertNotNull(((ReportSituation)(commonBaseEvent.getSituation().getSituationType())).getReportCategory());

            assertEquals(((ReportSituation)(commonBaseEvent.getSituation().getSituationType())).getReportCategory(),"LOG");
        } 
        catch (AssertionFailedError a) {
            throw (a);
        } 
        catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }  
}