package org.eclipse.hyades.logging.log4j;

import java.util.HashMap;
import java.util.Iterator;

import org.apache.log4j.AppenderSkeleton;
import org.apache.log4j.Layout;
import org.apache.log4j.spi.LoggingEvent;
import org.eclipse.hyades.logging.core.LoggingAgent;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LoggingAgentAppender.java,v 1.3 2005/02/16 22:20:54 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Extension of the Log4J <code>org.apache.log4j.AppenderSkeleton</code> abstract class
 * used by <code>org.apache.log4j.Logger</code> instances to publish logged 
 * <code>org.apache.log4j.spi.LoggingEvent</code> instances to a Logging Agent.
 * <p>
 * An instance of this class will be returned from the
 * <code>getAllAppenders()</code> API on an <code>org.apache.log4j.Logger</code>
 * instance.
 * <p>
 * A Logging Agent is created with the same name as the <code>org.apache.log4j.Logger</code> 
 * instance which logged the <code>org.apache.log4j.spi.LoggingEvent</code> instance.  The 
 * event is implicitly formatted using the 
 * <code>org.eclipse.hyades.logging.log4j.XMLLayout</code> layout. 
 * <p>
 * Users may configure a <code>org.apache.log4j.Logger</code> instance to use the 
 * <code>LoggingAgentAppender</code> by adding the following the entry to the 
 * <code>&lt;log4j:configuration&gt;</code> element in the <code>log4j.xml</code> 
 * properties file:
 * <p>
 * <code>
 * &lt;!-- LoggingAgentAppender --&gt;
 * &lt;!-- Outputs events to a Logging Agent using possible filter(s).  Layout is performed implicitly. --&gt;
 * &lt;appender name="LoggingAgent" class="org.eclipse.hyades.logging.log4j.LoggingAgentAppender" /&gt;
 * </code>
 * <p>
 * Also, the following the entry or appender reference must be added to the <code>&lt;logger&gt;</code> 
 * and/or <code>&lt;root&gt;</code> elements as desired in the <code>log4j.xml</code> properties file:
 * <p>
 * <code>
 * &lt;!-- LoggingAgentAppender for this logger: --&gt;
 * &lt;appender-ref ref="LoggingAgent" /&gt; 
 * </code>
 * <p>
 * Alternatively, an instantiation of this appender class may be set directly on
 * <code>org.apache.log4j.Logger</code> instances by using the
 * <code>addAppender()</code> API.
 * <p>
 * This appender is implicitly configured to use the 
 * <code>org.eclipse.hyades.logging.log4j.XmlLayout</code> formatter. 
 * <p>
 * NOTE: The Log4J classes must be on the CLASSPATH at run-time to
 * utilize this appender class.
 * <p>
 *  
 * 
 * @author    Paul E. Slauenwhite
 * @version   November 24, 2004
 * @since     July 20, 2004
 * @see       org.apache.log4j.AppenderSkeleton
 * @see       org.apache.log4j.Logger
 * @see       org.eclipse.hyades.logging.log4j.XmlLayout
 */
public class LoggingAgentAppender extends AppenderSkeleton {   

    /**
     * The default Logging Agent.
     */
    private static LoggingAgent defaultLoggingAgent = null;

    /**
     * Instance count of this class.
     */
    private static int instanceCount = 0;

    /**
     * All existing Logging Agents in the current JVM persisted for future
     * recovery.
     */
    private static HashMap loggingAgents = new HashMap();

    /**
     * Current thread lock for synchronized operations.
     */
    private static final Object CURRENT_THREAD_LOCK = new Object();

    /**
     * Default Logging Agent name.
     */
    public static final String DEFAULT_LOGGER_NAME = "Default Logging IRemoteEntity";

    /**
     * Constructor to create a Logging Agent Appender.
     * <p>
     * The layout for this type of appender is defaulted to 
     * a <code>org.eclipse.hyades.logging.log4j.XMLLayout</code> 
     * instance.
     */
    public LoggingAgentAppender() {
        
        super.setLayout(new XmlLayout());
        
        synchronized (CURRENT_THREAD_LOCK) {
            instanceCount++;
        }
    }

    /**
     * Publishing a <code>org.apache.log4j.spi.LoggingEvent</code> instance to a 
     * logger's Logging Agent.
     * <p>
     * The <code>org.apache.log4j.spi.LoggingEvent</code> is converted using
     * the appender's implicit layout (e.g.
     * <code>org.eclipse.hyades.logging.log4j.XmlLayout</code>) and
     * propagated on to the logger's Logging Agent.
     * <p>
     * The Logging Agent is created (e.g. currently does not exist) or retrieved 
     * (e.g. currently exists) by using the <code>org.apache.log4j.spi.LoggingEvent</code>'s
     * source logger name or a default Logging Agent name, if one does not exist.
     * <p>
     * This method quietly returns if the appender has been closed 
     * (e.g. see <code>close()</code>) and/or the parameter 
     * <code>org.apache.log4j.spi.LoggingEvent</code> is <code>null</code>.
     * <p>
     * 
     * @param loggingEvent
     *            The <code>org.apache.log4j.spi.LoggingEvent</code> to be published
     *            to a Logging Agent.
     */
    protected void append(LoggingEvent loggingEvent) {
        
        //Checks if the appender has not been closed and if the LoggingEvent is not 
        //null:
        if ((!closed) && (loggingEvent != null)) {
           
            String formattedLoggingEvent = null;
 
            try {
                formattedLoggingEvent = getLayout().format(loggingEvent);
            } 
            catch (Exception e) {
            
                getErrorHandler().error(null, e, 1,loggingEvent);
                
                return;
            }

            //Find the logger's Logging Agent, creating one if
            // necessary:
            LoggingAgent logger = null;

            final String loggerName = loggingEvent.getLoggerName();

            if ((loggerName != null) && (loggerName.trim().length() != 0)) {

                synchronized (CURRENT_THREAD_LOCK) {

                    if (loggingAgents.containsKey(loggerName)) {
                        logger = ((LoggingAgent) (loggingAgents.get(loggerName)));
                    } 
                    else {

                        //If we have never seen this Logging Agent before lets
                        // register it in the run-time:
                        logger = new LoggingAgent(loggerName);

                        loggingAgents.put(loggerName, logger);
                    }
                    
                    logger.write(formattedLoggingEvent);
                }
            } 
            else {

                synchronized (CURRENT_THREAD_LOCK) {       
                    
                    if(defaultLoggingAgent == null){
                        defaultLoggingAgent = new LoggingAgent(DEFAULT_LOGGER_NAME);
                    }
                    
                    defaultLoggingAgent.write(formattedLoggingEvent);
                }
            }
        }
    }

    /**
     * Flush any waiting messages to the Logging Agent(s).
     * <p>
     * This method quietly returns if the appender has been closed 
     * (e.g. see <code>close()</code>).
     */
    public void flush() {

        if(!closed){

            synchronized (CURRENT_THREAD_LOCK) {
                
    	        Iterator cachedLoggingAgents = loggingAgents.values().iterator();
    	
    	        while (cachedLoggingAgents.hasNext()) {
    	            ((LoggingAgent) (cachedLoggingAgents.next())).flush();
    	        }
    	        
                if(defaultLoggingAgent != null){
                    defaultLoggingAgent.flush();
                }
            }
        }
    }
    
    /**
     * Flush any waiting messages and de-registering the Logging Agent(s).
     * <p>
     * This method quietly returns if the appender has been closed 
     * (e.g. see <code>close()</code>).
     */
    public void close() {

        if(!closed){
            
            flush();
            
            synchronized (CURRENT_THREAD_LOCK) {

                if(--instanceCount == 0){

                    Iterator cachedLoggingAgentNames = loggingAgents.keySet().iterator();
        	    	
        	    	while (cachedLoggingAgentNames.hasNext()) {
        	    		((LoggingAgent)(loggingAgents.remove(cachedLoggingAgentNames.next()))).deregister();
        	    	}

        		    if(defaultLoggingAgent != null){
        		        defaultLoggingAgent.deregister();
        		    }
                }
            }
            
            closed = true;
        }
    }
   
    /**
     * @see org.apache.log4j.Appender#requiresLayout()
     */
    public boolean requiresLayout() {
        return false;
    }

    /**
     * This method quietly returns since the appender's layout is implicitly configured.
     * <p>
     *  
     * @see org.apache.log4j.Appender#setLayout(org.apache.log4j.Layout)
     */
    public void setLayout(Layout layout) {
        //No-operation since the appender's layout is implicitly configured.
    }
}
