/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Generator.java,v 1.5 2005/05/03 22:15:11 bjiang Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.internal.codegen;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;

import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.ui.internal.util.ResourceUtil;

/**
 * @author marcelop
 * @since 1.0.1
 */
abstract public class Generator
{
	public static final String CHARSET_UTF8 = "UTF-8"; 
	
	/**
	 * Generates the code for the specified test suite, creating all the required
	 * folders, packages and adjusting the project's classpath.
	 * 
	 * <p>The location of the test suite's implementor must be equals to the source
	 * container (ie, project and source folder).  The implementor's resource must
	 * be equals to the full class name (ie, package and class name).
	 *  
	 * @param testSuite
	 * @param monitor
	 * @throws Exception
	 */
	public void generate(ITestSuite testSuite, IProgressMonitor monitor) throws Exception
	{
		IFile file = getFileHandle(testSuite);
		IContainer sourceContainer = getSourceContainerHandle(testSuite);
		
		boolean mustGenerate = true;
		if(file.exists()) {
			try {
				doUpdateFile(testSuite, sourceContainer, file, monitor);
				mustGenerate = false;
			} catch (UnsupportedOperationException e) {
				file.delete(/*force*/true, null);
			}
		}
		
		if (mustGenerate) {
			ResourceUtil.createContainer(sourceContainer, monitor);
			doGenerateFile(testSuite, sourceContainer, file, monitor);
		}
	}
	
	protected void doGenerateFile(ITestSuite testSuite, IContainer sourceContainer, IFile file, IProgressMonitor monitor)
	throws Exception
	{
		generateFile(testSuite, file, monitor);
	}
		
	protected void doUpdateFile(ITestSuite testSuite, IContainer sourceContainer, IFile file, IProgressMonitor monitor) throws Exception {
		updateFile(testSuite, file, monitor);		
	}
		
	public IFile getFileHandle(ITestSuite testSuite)
	{
		IContainer sourceContainer = getSourceContainerHandle(testSuite);
		return sourceContainer.getFile(new Path(testSuite.getImplementor().getResource()));
	}
	
	protected IContainer getSourceContainerHandle(ITestSuite testSuite)
	{
		IContainer container = null;
		IPath path = new Path(testSuite.getImplementor().getLocation());
		if(path.segmentCount() == 1)
		{
			container = ResourcesPlugin.getWorkspace().getRoot().getProject(path.toString());
		}
		else
		{
			container = ResourcesPlugin.getWorkspace().getRoot().getFolder(path);
		}
		
		return container;
	}
	
	abstract protected void generateFile(ITestSuite testSuite, IFile file, IProgressMonitor monitor)
	throws Exception;	
	
	protected void updateFile(ITestSuite testSuite, IFile file, IProgressMonitor monitor) throws Exception {
		throw new UnsupportedOperationException();
	}
	
}
