/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Helper.java,v 1.7 2005/03/22 15:52:49 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.internal.codegen;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.facades.behavioral.IDeployableComponent;
import org.eclipse.hyades.models.common.facades.behavioral.INamedElement;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.common.TestCommonPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.ui.internal.util.JavaUtil;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.ToolFactory;
import org.eclipse.jdt.core.formatter.CodeFormatter;
import org.eclipse.jface.text.Document;
import org.eclipse.text.edits.TextEdit;
import org.eclipse.ui.model.IWorkbenchAdapter;

/**
 * @author marcelop
 * @since 1.0.1
 */
public class Helper implements IDisposable
{
	private static final String DEFAULT_NAMESPACE = "DEFNS#";
	private Set namesInWorkspace;
	private Map nameByObjectKey;

	private ImportManager importManager;
	private StringBuffer importStringBuffer;
	private int importInsertionPoint;

	public static String formatContent(String content) {
		return formatContent(content, 0);
	}
	
	public static String formatContent(String content, int offset)
	{
		if(content == null)
			return null;
		String code = "";
		Document doc = new Document(content);
		TextEdit tEdit = ToolFactory.createCodeFormatter(null).format(CodeFormatter.K_UNKNOWN, doc.get(), 0, doc.get().length(), offset, null);
        try {
            tEdit.apply(doc);
        } catch (Exception e) {
            TestCommonPlugin.logError(e);
        }
        code = doc.get();
		return code;
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		if (namesInWorkspace != null)
			namesInWorkspace.clear();

		if (nameByObjectKey != null)
			nameByObjectKey.clear();
	}

	public String getPackageName(Object object)
	{
		String cls = null;
		if (object instanceof ITest)
			object = ((ITest)object).getImplementor();

		if (object instanceof IDeployableComponent)
			cls = ((IDeployableComponent)object).getResource();

		if (cls != null)
		{
			int index = cls.lastIndexOf('.');
			if (index >= 0)
				return cls.substring(0, index);
		}

		return "";
	}

	public String getLabel(Object object)
	{
		if (object instanceof IAdaptable)
		{
			IWorkbenchAdapter workbenchAdapter = (IWorkbenchAdapter) ((IAdaptable)object).getAdapter(IWorkbenchAdapter.class);
			if (workbenchAdapter != null)
				return workbenchAdapter.getLabel(object);
		}

		if (object instanceof INamedElement)
			return ((INamedElement)object).getName();

		return object.toString();
	}
	
	public String getFilePath(Object object)
	{
		if(object instanceof EObject)
			return EMFUtil.getWorkspaceFilePath(((EObject)object).eResource()); 
		
		return null;		
	}

	public String retrieveClassName(INamedElement namedElement)
	{
		String name = getUniqueName("CLASS#", namedElement);
		if (name != null)
			return name;

		IDeployableComponent deployableComponent = null;
		if (namedElement instanceof ITestSuite)
			deployableComponent = ((ITestSuite)namedElement).getImplementor();
		else if (namedElement instanceof IDeployableComponent)
			deployableComponent = (IDeployableComponent)namedElement;

		if(deployableComponent != null)
		{
			name = deployableComponent.getResource();
			name = name.substring(name.lastIndexOf('.')+1);
		}
		else
			name = namedElement.getName();
			 

		name = JavaUtil.getValidClassName(name, false);
		if (name == null)
		{
			name = name.getClass().getName();
			int index = name.lastIndexOf('.');
			if (index >= 0)
				name = name.substring(index + 1);
		}

		name = retrieveUniqueName("CLASS#", namedElement, name, true);

		if(deployableComponent != null)
		{
			String newName = getPackageName(deployableComponent) + "." + name;
			if(!newName.equals(deployableComponent.getResource()))
				deployableComponent.setResource(newName);
		}
		return name;
	}

	public String retrieveIdentifierName(INamedElement namedElement, Object namespace)
	{
		String name = getUniqueName("IDENTIFIER#" + namespace, namedElement);
		if (name != null)
			return name;

		name = JavaUtil.getValidIdentifierName(namedElement.getName());
		if (name == null)
		{
			name = name.getClass().getName();
			int index = name.lastIndexOf('.');
			if (index >= 0)
				name = name.substring(index + 1);

			String auxName = name.substring(0, 1).toLowerCase();
			if (name.length() > 1)
				auxName = auxName + name.substring(1);
			name = auxName;
		}

		name = retrieveUniqueName("IDENTIFIER#" + namespace, namedElement, name, false);
		if(namedElement instanceof ITestCase)
		{
			ITestCase testCase = (ITestCase)namedElement;
			if(!name.equals(testCase.getImplementor().getResource()))
				testCase.getImplementor().setResource(name);
		}
		
		return name;
	}

	/**
	 * Returns a unique name for a given object derived from the the base name
	 * argument.  The namespace argument defines the context in which the returned
	 * name must be unique.  If <code>null</code> then a default namespace is default. 
	 *
	 * <p>If a name has been already created for a given object in the same	namespace 
	 * this name is returned.
	 * 
	 * @param namespace
	 * @param object
	 * @param name
	 * @return a not <code>null</code> String
	 */
	protected synchronized String retrieveUniqueName(String namespace, Object object, String baseName, boolean caseSensitive)
	{
		if ((object == null) || (baseName == null))
			return "";

		if(!caseSensitive)
			baseName = baseName.toLowerCase();

		if (namesInWorkspace == null)
			namesInWorkspace = new HashSet();
		if (nameByObjectKey == null)
			nameByObjectKey = new HashMap();

		if (namespace == null)
			namespace = DEFAULT_NAMESPACE;

		String key = namespace + ">";
		String objectKey = key + object.hashCode();
		String uniqueName = (String)nameByObjectKey.get(objectKey);
		if (uniqueName == null)
		{
			int count = 0;
			uniqueName = baseName;
			while (namesInWorkspace.contains(key + uniqueName))
				uniqueName = baseName + (++count);

			namesInWorkspace.add(key + uniqueName);
			nameByObjectKey.put(objectKey, uniqueName);
		}

		return uniqueName;
	}

	/**
	 * Returns a created unique name for a given object derived from the the base name
	 * argument.  The namespace argument defines the context in which the returned
	 * name must be unique.  If <code>null</code> then a default namespace is default. 
	 *
	 * @param namespace
	 * @param object
	 * @return a unique name or null if none was created by the 
	 * {@link #retrieveUniqueName(String, Object, String).
	 */
	protected String getUniqueName(String namespace, Object object)
	{
		if ((object == null) || (nameByObjectKey == null))
			return null;

		if (namespace == null)
			namespace = DEFAULT_NAMESPACE;

		String key = namespace + ">";
		String objectKey = key + object.hashCode();
		return (String)nameByObjectKey.get(objectKey);
	}

	public void markImportLocation(StringBuffer stringBuffer)
	{
		importStringBuffer = stringBuffer;
		importInsertionPoint = stringBuffer.length();
		importManager.addCompilationUnitImports(stringBuffer.toString());
	}

	public void emitSortedImports()
	{
		String NL = System.getProperties().getProperty("line.separator");
		StringBuffer imports = new StringBuffer();

		String previousPackageName = null;
		for (Iterator iter = importManager.getImports().iterator(); iter.hasNext();)
		{
			String importName = (String)iter.next();
			int index = importName.lastIndexOf(".");
			if (index != -1)
			{
				String packageName = importName.substring(0, index);
				if (previousPackageName != null && !previousPackageName.equals(packageName))
				{
					imports.append(NL);
				}
				previousPackageName = packageName;
			}
			imports.append(NL + "import " + importName + ";");
		}

		importStringBuffer.insert(importInsertionPoint, imports);
	}

	/**
	 * Update the specified compilation unit with the list of imports that were
	 * recorded by the helper using addImport.
	 * @throws JavaModelException 
	 * @author jcanches
	 */
	public void emitSortedImports(ICompilationUnit cu) throws JavaModelException {
		Iterator it = importManager.getImports().iterator();
		while (it.hasNext()) {
			String importName = (String) it.next();
			cu.createImport(importName, /*sibling*/null, /*progressMonitor*/null);
		}
	}

	public String getImportedName(String qualifiedName)
	{
		int index = qualifiedName.indexOf("$");
		importManager.addImport(index == -1 ? qualifiedName : qualifiedName.substring(0, index));
		return importManager.getImportedName(qualifiedName);
	}

	public void addImport(String qualifiedName)
	{
		importManager.addImport(qualifiedName);
	}

	protected ImportManager getImportManager()
	{
		return importManager;
	}

	public void setImportManager(ImportManager importManager)
	{
		this.importManager = importManager;
	}
	
	public String getTestMethodName(ITestCase testCase) {
		String identifier = testCase.getImplementor().getResource();
		if(identifier == null)
			identifier = retrieveIdentifierName(testCase, "METHOD");
		return identifier;
	}

}
