/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionHarnessDataProcessorFactory.java,v 1.5 2005/06/03 03:18:59 sschneid Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.harness;

import java.util.ArrayList;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.execution.harness.util.ISystemUtility;

/**
 * @author ejessee
 */
public class ExecutionHarnessDataProcessorFactory 
{
	private static ArrayList standAloneClassObjects = null;
	private static IConfigurationElement[] dpConfigElements=null;
	
	private boolean bStandAlone = false;
	
	
	public ExecutionHarnessDataProcessorFactory(boolean bStandAlone)
	{
		if(!bStandAlone &&dpConfigElements == null)
			loadDataProcessors();
		
		this.bStandAlone = bStandAlone;
	}
	
	
	/**
	 * loads dataprocessors if we are not running standalone
	 * @return
	 */
	private void loadDataProcessors()
	{
		
		dpConfigElements = Platform.getExtensionRegistry().getConfigurationElementsFor("org.eclipse.hyades.execution.harness.Dataprocessor"); //$NON-NLS-1$
		
			
	}	

	/**
	 * called to set the ArrayList of dataProcessor classes
	 * this array list should contained non-instantiated class objects which implement the
	 * IExecutionHarnessDataProcessor interface
	 * This api should only be called in "standalone" mode.
	 * @param standAloneClassObjects
	 */
	public static void setStandAloneClassObjects(ArrayList standAloneClassObjects) 
	{		
		ExecutionHarnessDataProcessorFactory.standAloneClassObjects = standAloneClassObjects;
		
	}
	
	

	/**
	 * @return Returns the dataProcessors collection.  Each object in the collection is an instantiated but not
	 * initialized dataprocessor.
	 * @deprecated use the get data processors method that takes a data process list instead
	 */
	public ArrayList getDataProcessors() 
	{
		ArrayList dataProcessors = new ArrayList();
		if(this.bStandAlone)
		{
			for(int i=0; i<standAloneClassObjects.size(); i++)
			{
				try 
				{
					dataProcessors.add(((Class)standAloneClassObjects.get(i)).newInstance());
				} 
				catch (InstantiationException e) {
					
					((ISystemUtility)ExecutionHarnessPlugin.getDefault()).logError(e);
				} 
				catch (IllegalAccessException e) {
					
					((ISystemUtility)ExecutionHarnessPlugin.getDefault()).logError(e);
				}
			}
		}
		
		else
		{
			for ( int i=0; i<dpConfigElements.length; i++)
			{
				try
				{
					IExecutionHarnessDataProcessor dataProcessor = (IExecutionHarnessDataProcessor)dpConfigElements[i].createExecutableExtension("class"); //$NON-NLS-1$
					dataProcessors.add(dataProcessor);
				}
				catch(CoreException e)
				{
					((ISystemUtility)ExecutionHarnessPlugin.getDefault()).logError(e);
				}
			}
		}
		
		return dataProcessors;
	}
	
	/**
	 * Return a list of data processors that are valid based on the list specified.  Any data processor that is
	 * on the list and is also found and valid is included in the return array list.  The data processors that are
	 * returned are already instantiated but not initialized (any valid data processor on the list that is
     * available will be instantiated and in most cases this causes some additional side-effects of resource creation
	 * such as additional threads being spawned)
	 * 
	 * @param dataProcessors the valid data processors to include in the return list if available
	 * @return the array list of data processors
	 */
	public ArrayList getDataProcessors(ArrayList validDataProcessorIDs) {
	    
		ArrayList dataProcessors = new ArrayList();
		
		/**
		 * For the standalone case, we'll return all data processors and ignore ID list, there is a way
		 * to get this before instantiating an instance although this requires reflection and conformance
		 * to a naming pattern since there are no class-level interface concepts to force this.
		 */
		if(this.bStandAlone)
		{
			for(int i=0; i<standAloneClassObjects.size(); i++)
			{
				try 
				{
					dataProcessors.add(((Class)standAloneClassObjects.get(i)).newInstance());
				} 
				catch (InstantiationException e) {
					
					((ISystemUtility)ExecutionHarnessPlugin.getDefault()).logError(e);
				} 
				catch (IllegalAccessException e) {
					
					((ISystemUtility)ExecutionHarnessPlugin.getDefault()).logError(e);
				}
			}
		}
		
		else
		{
			for ( int i=0; i<dpConfigElements.length; i++)
			{
				try
				{
				    
				    // Restrict instantiation to valid data processors only; removes overhead of unused data processors
				    IConfigurationElement configurationElement = dpConfigElements[i];
				    if (validDataProcessorIDs.contains(configurationElement.getAttribute("id"))) {
				        IExecutionHarnessDataProcessor dataProcessor = (IExecutionHarnessDataProcessor) configurationElement.createExecutableExtension("class"); //$NON-NLS-1$
				        dataProcessors.add(dataProcessor);
				    }
				    
				}
				catch(CoreException e)
				{
					((ISystemUtility)ExecutionHarnessPlugin.getDefault()).logError(e);
				}
			}
		}
		
		return dataProcessors;	    
	}

}
