/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ConfigurationUtil.java,v 1.10 2005/02/16 22:21:15 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.configuration.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.configuration.CFGArtifact;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGMachineConstraint;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofileFactory;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;


/**
 * Convenient utility class to handle configuration objects.
 * 
 * @author bjiang
 * @since  3.0
 */
public class ConfigurationUtil
{
    public static final String CLASSPATH = "CLASSPATH";
    public static final String ROOTDIR = "ROOTDIR";
    public static final String ATTRS_PROP_GROUP_ID_ARTIFACT = "org.eclipse.hyades.test.configuration.artifact.attributes";
	public static final String ATTRS_PROP_GROUP_ID_LOCATION = "org.eclipse.hyades.test.configuration.location.attributes";
	
	public static final String LOCALHOST = "localhost";
	public static final String DEFAULT_DEPLOYMENT_NAME = "local_deployment";

	/**
	 * Creates a new {@link org.eclipse.hyades.models.common.testprofile.TPFDeployment TPFDeployment} 
	 * setting the attributes based on the specified information.
	 * @return a none null EObject
	 */
	public static TPFDeployment createDeployment(String name, String description)
	{
		TPFDeployment deployment = Common_TestprofileFactory.eINSTANCE.createTPFDeployment();
		
		if(name != null)
			deployment.setName(name);	
		if((description != null) && (description.trim().length() > 0))
			deployment.setDescription(description);
		
		return deployment;
	}
	
	private static TPFDeployment defaultDeployment = null;
	
	/**
	 * Returns the default {@link org.eclipse.hyades.models.common.testprofile.TPFDeployment TPFDeployment}
	 * that references a location for <i>localhost</i>.
	 * @return a none null EObject
	 */
	public static TPFDeployment createDefaultDeployment()
	{
		if (defaultDeployment == null) {
			defaultDeployment = createDeployment(DEFAULT_DEPLOYMENT_NAME, "");
			defaultDeployment.getRefLocations().add(createLocation());
		}
		return defaultDeployment;
	}
	
	/**
	 * Returns whether the specified deployment is the default deployment.
	 * @param deployment
	 * @return
	 */
	public static boolean isDefaultDeployment(TPFDeployment deployment) {
		return deployment == defaultDeployment;
	}
	
	/**
	 * Creates the TPFDeployment setting the attributes based on the specified information.
	 * @return a not null EObject
	 */
	public static TPFDeployment createDeploymentWithPair(String name, String description, CFGClass testAsset, String hostname)
	{
		TPFDeployment deployment = createDeployment(name, description);
		
		CFGArtifact artifact = createArtifact(null, null);
		artifact.getDeployableInstances().add(testAsset);
		CFGLocation location = createLocation(null, null, hostname);
		CFGArtifactLocationPair pair = createArtifactLocationPair(artifact, location);
		deployment.getArtifactLocations().add(pair);
		
		return deployment;
	}
	
	/**
	 * Creates the CFGArtifact setting the attributes based on the specified information.
	 * @return a not null EObject
	 */
	public static CFGArtifact createArtifact(String name, String description)
	{
		CFGArtifact artifact = Common_ConfigurationFactory.eINSTANCE.createCFGArtifact();
		
		if(name != null)
			artifact.setName(name);	
		if((description != null) && (description.trim().length() > 0))
			artifact.setDescription(description);
		
		return artifact;
	}	

	/**
	 * Creates the CFGLocation setting the attributes based on the specified information.
	 * @return a not null EObject
	 */
	public static CFGLocation createLocation(String name, String description, String hostname)
	{
		CFGLocation location = createLocation();
		if(name != null)
			location.setName(name);					
		if((description != null) && (description.trim().length() > 0))
			location.setDescription(description);
		if(hostname != null && location instanceof CFGMachineConstraint)
			((CFGMachineConstraint)location).setHostname(hostname);
			
		return location;
	}
	
	/**
	 * Creates a CFGLocation object with default hostname <i>localhost</i> and a default root dir property,
	 * but this location has no name or description. Use {@link #createLocation(String, String, String)} is prefered.
	 * 
	 * @return CFGLocation object
	 */
	public static CFGLocation createLocation()
	{
		CFGMachineConstraint machine = Common_ConfigurationFactory.eINSTANCE.createCFGMachineConstraint();
		String hostname = LOCALHOST;
		machine.setHostname(hostname);
		
		return machine;
	}

	/**
	 * Returns the first pair of the <i>deployment</i> that contains reference to the specified <i>testAsset</i>.
	 * 
	 * @param testAsset
	 * @param deployment
	 * @return first pair contains <i>testAsset</i> or <code>null</code> if such a pair is not available from the <i>deployment</i>.
	 */
	public static CFGArtifactLocationPair searchPairWithTestAsset(CFGClass testAsset, TPFDeployment deployment)
	{
		if(testAsset == null || deployment == null)
			return null;
		
		CFGArtifactLocationPair pair = null;
		CFGArtifact artifact = null;
		for(Iterator i = deployment.getArtifactLocations().iterator(); i.hasNext();)
		{
			pair = (CFGArtifactLocationPair)i.next();
			artifact = pair.getArtifact();
			if(containsById(artifact.getDeployableInstances(), testAsset))
				return pair;
		}
		
		return null;
	}
	
	/**
	 * Returns the first location for the specified <i>testAsset</i> in this <i>deployment</i>.
	 * 
	 * @param testAsset
	 * @param deployment
	 * @return first location for <i>testAsset</i> or <code>null</code> if no such location available in the <i>deployment</i>.
	 */
	public static CFGLocation searchLocationWithTestAsset(CFGClass testAsset, TPFDeployment deployment)
	{
		CFGArtifactLocationPair pair = searchPairWithTestAsset(testAsset, deployment);
		if(pair != null)
			return pair.getLocation();
		return null;
	}
	
	public static CFGArtifact searchArtifactWithTestAsset(CFGClass testAsset, TPFDeployment deployment)
	{
		if(testAsset == null || deployment == null)
			return null;
		
		CFGArtifact artifact = null;
		for(Iterator i = deployment.getArtifacts().iterator(); i.hasNext();)
		{
			artifact = (CFGArtifact)i.next();
			if(containsById(artifact.getDeployableInstances(), testAsset))
				return artifact;
		}
		
		return null;
	}
	
	public static CFGArtifactLocationPair createArtifactLocationPair(CFGArtifact artifact, CFGLocation location)
	{
		CFGArtifactLocationPair pair = Common_ConfigurationFactory.eINSTANCE.createCFGArtifactLocationPair();
		pair.setArtifact(artifact);
		pair.setLocation(location);
		
		return pair;
	}
	
	public static BVRProperty[] searchPropertiesByName(List properties, String name, boolean caseSensitive)
	{
		if((properties != null) && (!properties.isEmpty()) && (name != null))
		{
			List ret = new ArrayList();
			for (Iterator i = properties.iterator(); i.hasNext();)
			{
				BVRProperty property = (BVRProperty)i.next();
				if(caseSensitive)
				{
					if(name.equals(property.getName()))
						ret.add(property);
				}
				else
				{
					if(name.equalsIgnoreCase(property.getName()))
						ret.add(property);						
				}
			}
			return (BVRProperty[])ret.toArray(new BVRProperty[ret.size()]);
		}
		
		return new BVRProperty[0];
	}
	
	/**
	 * Returns the {@link org.eclipse.hyades.models.common.configuration.CFGPropertyGroup} with the sepecified
	 * <i>propGroupId</i> from the given list.
	 * 
	 * @param propertyGroups
	 * @param propGroupId
	 * @return the target property group or <code>null</code> if such property group does not exist in the given list.
	 */
	public static CFGPropertyGroup searchPropertyGroupById(List propertyGroups, String propGroupId)
	{
	    if(propertyGroups == null || propertyGroups.isEmpty() || propGroupId == null)
	        return null;
	    
	    CFGPropertyGroup group = null;
	    for(Iterator it = propertyGroups.iterator(); it.hasNext(); )
	    {
	        Object next = it.next();
	        if(next instanceof CFGPropertyGroup && ((CFGPropertyGroup)next).getPropertyGroupID() != null && ((CFGPropertyGroup)next).getPropertyGroupID().equals(propGroupId))
	        {
	            group = (CFGPropertyGroup)next;
	            break;
	        }
	    }
	    
	    return group;
	}
	
	/**
	 * Checks if the any of the CMNNamedElement in the list has the same ID as the specified <i>element</i>.  
	 * @param namedElementList
	 * @param element
	 * @return
	 */
	public static boolean containsById(List namedElementList, CMNNamedElement element)
	{
	    if(namedElementList.isEmpty())	
	        return false;
	    
	    for(Iterator it = namedElementList.iterator(); it.hasNext(); )
	    {
	        Object obj = it.next();
	        if(obj instanceof CMNNamedElement && ((CMNNamedElement)obj).getId().equals(element.getId()))
	            return true;
	    }
	    
	    return false;    
	}
	
	public static String getHostNameForTestAsset(CFGClass rootResource, TPFDeployment deployment)
	{
		String machineName = null;
		if(!deployment.getArtifactLocations().isEmpty())
		{
			CFGArtifactLocationPair pair = searchPairWithTestAsset(rootResource, deployment);
			if(pair != null)
			{
				CFGLocation location = pair.getLocation();
				if(location instanceof CFGMachineConstraint)
					machineName = ((CFGMachineConstraint)location).getHostname();
				else if(location != null)
					machineName = location.getName();
			}
		}
		
		return machineName;
	}
	
	public static String getHostName(CFGLocation location)
	{
		if(location == null)
			return null;
		
		if(location instanceof CFGMachineConstraint)
			return ((CFGMachineConstraint)location).getHostname();
		else
			return location.getName();
	}
	
	/**
	 * This method returns the default location for this deployment, if such a
	 * default exists.  The default location for a deployment is defined to 
	 * exist only if the deployment has no artifact/location pairs and if the 
	 * deployment contains a single location (pre-Hyades 3.0 location) or a 
	 * reference to a single location (Hyades 3.0+).
	 * <p><b>This method will return <code>void</code> if any of the above conditions
	 * are not met.</b></p>
	 * @param deployment
	 * @return the default location for this deployment
	 */
	public static CFGLocation getDefaultLocation(TPFDeployment deployment)
	{
    	// bugzilla_62096
    	// special case support for a deployment that has no artifact/
    	// location pairs, but has a reference to a single location.
    	// This supports the old execution use case.
    	// In this case, we will use the single location specified by the 
		// deployment
		CFGLocation location = null;

		if (isDefaultLocation(deployment))
		{
			if ( deployment.getLocations().size() == 1)
	        {
	        	location = (CFGLocation)deployment.getLocations().get(0);
	        }
			else if ( deployment.getRefLocations().size() == 1)
			{
				location = (CFGLocation)deployment.getRefLocations().get(0);
			}
		}
		
		return location;
	}
	
	public static boolean isDefaultLocation(TPFDeployment deployment)
	{
		return ( deployment.getArtifactLocations().isEmpty() &&
			 (deployment.getLocations().size() == 1 ||
			  deployment.getRefLocations().size() == 1));
	}

}
