/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AntTaskAdapter.java,v 1.3 2005/05/20 21:38:57 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.client.adapters.ant;

import java.io.File;
import java.util.Properties;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.taskdefs.Property;
import org.apache.tools.ant.types.Path;
import org.eclipse.hyades.automation.client.adapters.java.JavaAdapter;

/**
 * @author Scott E. Schneider
 * 
 * These are the two included ant task definitions, the first one, named
 * "Automation" allows for the automation of any automatable published service
 * and the second, named "Test" allows for the automation of test-specific
 * services, such as test execution (for example, executing a JUnit test suite
 * defined by TPTP).
 * 
 * @provisional
 */
public class AntTaskAdapter {

    /**
     * @author Scott E. Schneider
     * 
     * This ant task is a general TPTP ant task used to request the execution of
     * published automatable TPTP services. A TPTP automatable service is
     * published via the implementation of specific automation-related plug-in
     * extension points and must implement automation-related Java interfaces as
     * well.
     */
    public static class Automation extends Task {

        /**
         * Stores the java adapter used for access the automation client
         */
        private JavaAdapter adapter;

        /**
         * The automation command to execute
         */
        private String command;

        /**
         * The automation service identifier, the name of the service to execute
         */
        private String identifier;

        /**
         * The number of iterations that this automation service should execute
         */
        private int iterations;

        /**
         * The service's state to be transmitted to the server
         */
        private Properties properties;

        /**
         * Add the configured property to the properties
         * 
         * @param property
         *            the property to add
         */
        public void addConfiguredProperty(Property property) {
            this.properties.setProperty(property.getName(), property.getValue());
        }

        /**
         * Add property to the list with the given name and value
         * 
         * @param name
         *            name of the property
         * @param value
         *            value of the property
         */
        public void addProperty(String name, boolean value) {
            this.addProperty(name, Boolean.toString(value));
        }

        /**
         * Add the property with the given name and associated file value
         * 
         * @param name
         *            the name of the property
         * @param file
         *            the value of the file, will be rendered to string
         */
        public void addProperty(String name, File file) {
            if (file != null) {
                this.addProperty(name, file.toString());
            }
        }

        /**
         * Add the given named value with the value as a path
         * 
         * @param name
         *            the name of the value
         * @param path
         *            the value as a path, to be used for things like system
         *            paths or classpaths, standard ant path
         */
        public void addProperty(String name, Path path) {
            if (path != null) {
                this.addProperty(name, path.toString());
            }
        }

        /**
         * Add the given named value property to the properties
         * 
         * @param name
         *            the name of the value
         * @param value
         *            the value
         */
        public void addProperty(String name, String value) {
            if (name != null && value != null) {
                Property property = new Property();
                property.setName(name);
                property.setValue(value);
                this.addConfiguredProperty(property);
            }
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.apache.tools.ant.Task#execute()
         */
        public void execute() throws BuildException {
            for (int i = 0; i < this.iterations; i++) {
                if (this.command.equalsIgnoreCase("execute")) {
                    adapter.execute(this.identifier, this.properties);
                }
            }
        }

        /**
         * Returns the root value as defined by eclipse.home currently
         * 
         * @return
         */
        private String getRoot() {
            String root = this.getProject().getProperty(AntTaskAdapter.ROOT_VARIABLE);
            return (root != null ? root : ".");
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.apache.tools.ant.Task#init()
         */
        public void init() {
            this.identifier = null;
            this.iterations = 1;
            this.properties = new Properties();
            this.adapter = new JavaAdapter(this.getRoot());
        }

        /**
         * Set the automation command to use
         * 
         * @param command
         *            the automation command, currently only execute is
         *            supported
         */
        public void setCommand(String command) {
            this.command = command;
        }

        /**
         * The number of iterations to execute this command
         * 
         * @param iterations
         *            the number of iterations from 1 to many
         */
        public void setIterations(int iterations) {
            this.iterations = iterations;
        }

        /**
         * Identifies the service to invoke
         * 
         * @param identifier
         *            the service identifier, which matches the extension point
         *            id in the implementor within Eclipse
         */
        public void setService(String identifier) {
            this.identifier = identifier;
        }

    }

    /**
     * @author Scott E. Schneider
     * 
     * There are more specific TPTP ant tasks that provide domain-specific
     * interfaces to the more general TPTP automatable services (for example
     * this is the TPTP test ant task that enables a more strongly-typed
     * execution of tests).
     */
    public static class Test extends AntTaskAdapter.Automation {

        /**
         * The test execution classpath to use
         */
        private Path classpath;

        /**
         * The deployment file to use that defines the deployment to use for
         * this test execution
         */
        private File deployment;

        /**
         * The location file to use that defines the location to use for this
         * test execution
         */
        private File location;

        /**
         * Whether or not the results should be overwritten or not
         */
        private boolean overwrite;

        /**
         * The Eclipse project that the test definition is stored in
         */
        private String project;

        /**
         * The file to hold the results in
         */
        private File results;

        /**
         * The file that defines the test suite itself
         */
        private File suite;

        /**
         * Path to where the test workspace is stored
         */
        private Path workspace;

        /**
         * Creates a classpath
         * 
         * @return the path of the classpath
         */
        public Path createClasspath() {
            Path classpath = new Path(this.getProject());
            this.classpath = classpath;
            return classpath;
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.apache.tools.ant.Task#execute()
         */
        public void execute() throws BuildException {
            this.addProperty("workspace", this.workspace);
            this.addProperty("project", this.project);
            this.addProperty("suite", this.suite);
            this.addProperty("deployment", this.deployment);
            this.addProperty("location", this.location);
            this.addProperty("results", this.results);
            this.addProperty("overwrite", this.overwrite);
            this.addProperty("classpath", this.classpath);
            this.setCommand("execute");
            super.execute();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.apache.tools.ant.Task#init()
         */
        public void init() {
            super.init();
            this.workspace = null;
            this.project = null;
            this.suite = null;
            this.deployment = null;
            this.location = null;
            this.results = null;
            this.overwrite = false;
            this.classpath = new Path(this.getProject(), "");
            this.setService("org.eclipse.hyades.execution.harness.TestExecutionHarness");
        }

        /**
         * Provides the ability to set the deployment
         * 
         * @param deployment
         *            the deployment file to use
         */
        public void setDeployment(File deployment) {
            this.deployment = deployment;
        }

        /**
         * Provides the ability to set the location
         * 
         * @param location
         *            the location file to use
         */
        public void setLocation(File location) {
            this.location = location;
        }

        /**
         * Provides the ability to set the overwrite value for the execution
         * results
         * 
         * @param overwrite
         *            whether of not the execution results will be overwritten
         */
        public void setOverwrite(boolean overwrite) {
            this.overwrite = overwrite;
        }

        /**
         * Provides the ability to set the project associated with the test
         * suite
         * 
         * @param project
         *            the Eclipse project
         */
        public void setProject(String project) {
            this.project = project;
        }

        /**
         * Provides the ability to set the execution results file
         * 
         * @param results
         *            the results of the execution
         */
        public void setResults(File results) {
            this.results = results;
        }

        /**
         * Provides the ability to set the test suite
         * 
         * @param suite
         *            the test suite definition file
         */
        public void setSuite(File suite) {
            this.suite = suite;
        }

        /**
         * Provides the ability to set the workspace value
         * 
         * @param workspace
         *            the workspace to use, a fully-qualified path
         */
        public void setWorkspace(Path workspace) {
            this.workspace = workspace;
        }

    }

    /**
     * The name of the variable that holds the eclipse home directory or root
     * directory
     */
    private static final String ROOT_VARIABLE = "eclipse.home";

}