/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractExecutionStrategy.java,v 1.2 2005/05/17 03:49:04 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.client.strategies;

import org.eclipse.hyades.automation.core.Service;

/**
 * The client execution strategy to use, this strategy specifies how the command
 * will be specifically transmitted to the automation server, examples of the
 * various strategies include in-process and out-of-process, out-of-process is
 * the default.
 * 
 * @author Scott E. Schneider
 */
public abstract class AbstractExecutionStrategy implements Service.Executable {

    /**
     * The execution strategy factory, creates the requested strategy as well as
     * providing a default
     * 
     * @author Scott E. Schneider
     */
    public final static class Factory {

        /**
         * The singleton factory to store
         */
        private static final Factory factory;

        static {
            factory = new Factory();
        }

        /**
         * Restrict construction of singleton
         */
        private Factory() {
        }

        /**
         * Singleton static access point
         * 
         * @return the singleton instance of the factory
         */
        public static final Factory getInstance() {
            return Factory.factory;
        }

        /**
         * Create the default strategy, this strategy will definitely work
         * whereas other strategies might be unsupported
         * 
         * @return the default strategy
         */
        public Service.Executable createDefaultStrategy() {
            return this.createOutOfProcessStrategy();
        }

        /**
         * Create an in-process strategy, to launch the automation command
         * within the same process as the requestor
         * 
         * @return the in-process strategy is returned
         */
        public Service.Executable createInProcessStrategy() {
            return new InProcessStrategy();
        }

        /**
         * Create an out-of-process strategy, to launch the automation command
         * within its own process
         * 
         * @return the out-of-process strategy is returned
         */
        public Service.Executable createOutOfProcessStrategy() {
            return new OutOfProcessStrategy();
        }

        /**
         * Creates a void strategy, basically does nothing but implements the
         * service executable interface
         * 
         * @return the void strategy is returned, good for testing
         */
        public Service.Executable createVoidStrategy() {
            return new VoidStrategy();
        }

    }

}