package org.eclipse.hyades.test.common.agent;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ComptestAgent.java,v 1.4 2005/02/16 22:21:34 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
 
import org.eclipse.hyades.internal.execution.local.common.BinaryCustomCommand;
import org.eclipse.hyades.internal.execution.local.control.AgentImpl;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;


/**
 * The Component Test Agent is a wrapper over the basic Logging Agent
 * that basically provides a synchronization mechanism used to ensure that
 * the communication pipe is ready to transmit the information.
 */
public class ComptestAgent
{
	private static final boolean TEST = false;
	
	/* Token to identify a message for the ComptestAgentListener */
	private static final String LISTENER_MSG = "LM##HP:";
	
	/* ContextID for the broadcast messages */
	public static final long BROADCAST_CONTEXT_ID = 500l;
	
	/* Sleep time at the end of the process */
	private static final long SHUTDOWN_SLEEP_TIME = 2000l;
	
	/* This is our logging mechanism */
	private RemoteComponentSkeleton loggingAgent;
	
	{
		Runtime.getRuntime().addShutdownHook(new Thread()
		{
			public void run()
			{
				try
				{
					sleep(SHUTDOWN_SLEEP_TIME);
				}
				catch (InterruptedException e)
				{
				}
			}
		});
	}

	/**
	 * Create a <code>ComptestAgent</code> that will write to...
	 *
	 *@param
	 */
	public ComptestAgent(RemoteComponentSkeleton remoteAgent)
	{
		loggingAgent = remoteAgent;
	}
	
	/**
	 * Sends a message to the the specified agent.
	 * 
	 * @param agent
	 * @param message
	 * @throws InactiveAgentException
	 */
	public static void sendData(AgentImpl agent, String message)
	throws InactiveAgentException
	{
		if(message == null)
			message = "";
			
		BinaryCustomCommand command=new BinaryCustomCommand();
		command.setData(ComptestAgent.LISTENER_MSG + message);
		agent.invokeCustomCommand(command);
	}
	
	/**
	 * Sends data to the the specified agent.
	 * 
	 * @param agent
	 * @param message
	 * @throws InactiveAgentException
	 */
	public static void sendData(AgentImpl agent, byte[] data)
	throws InactiveAgentException
	{
		if(data == null)
			data = new byte[0];
			
		BinaryCustomCommand command=new BinaryCustomCommand();
		command.setData(data);
		agent.invokeCustomCommand(command);
	}
	
	/**
	 * Returns true if the Component Test Agent is initialized
	 * 
	 * @return boolean
	 */	
	public boolean isInitialized()
	{
		return TEST || (loggingAgent != null);
	}

    /**
     * Returns whether the agent controller is currently active.
     */
    public boolean isAgentControllerActive()
    {
    	return (loggingAgent != null) && loggingAgent.isAgentControllerActive();
    }

    /**
     * Returns whether or not the agent is currently
     * being monitored.<br>
     * ie. Whether the log messages are being collected.<br>
     */
    public boolean isLogging()
    {
        return (loggingAgent != null) && loggingAgent.isLogging();
    }
	
	/**
	 * Writes the "toString()" value of the object to the agent.
	 * 
	 * @param object
	 */
	public void write(Object object)
	{
		if(isInitialized())
		{
			if(isAgentControllerActive() && isLogging())
				loggingAgent.logMessageUTF8(object.toString());
				
			if(TEST)
				System.out.println(object.toString()); 			
		}
		else
			System.out.println(object.toString());			
	}
	
	public boolean write(String fragment){
		boolean retValue = false;
		if(isInitialized()){
		   retValue = true;
		   if(isAgentControllerActive() && isLogging())
			  loggingAgent.logMessageUTF8(fragment);
				
		   if(TEST)
			  System.out.println(fragment); 			
	    }
	    else
		   System.out.println(fragment);
		return retValue;	
	}
	
	
	/*
	 * This is the code to receive a message at the 
	 * at the machine that has launched the process.
	 * 
		public void handleCommand(Agent agent, CommandElement command)
		{
			switch ((int)command.getTag())
			{
				case (int)Constants.RA_CUSTOM_COMMAND:
					CustomCommand customCommand = (CustomCommand)command;
					String data = customCommand.getData();
					break;
	
				default :
					break;
			}
		}
	 * 
	 */
	
	/**
	 * Sends a String message to any AgentListener listener attached to 
	 * the agent at the machine that has launched the process.
	 * 
	 * @param message
	 */
	public void sendMessage(String message)
	{
		if(!isInitialized())
			throw new RuntimeException("ComptestAgent not initialized.");

		/* The message will be received at the 
		 * handleCommand(Agent, CommandElement) method of the listener
		 */
		loggingAgent.broadcastMessage(message, BROADCAST_CONTEXT_ID);	
	}
	
	/**
	 * Sends a bynary message to any AgentListener listener attached to 
	 * the agent at the machine that has launched the process.
	 * 
	 * @param message
	 */
	public void sendMessage(byte[] message, int offset, int length)
	{
		if(!isInitialized())
			throw new RuntimeException("ComptestAgent not initialized.");

		/* The message will be received at the 
		 * handleCommand(Agent, CommandElement) method of the listener
		 */
		loggingAgent.broadcastMessage(message, offset, length, BROADCAST_CONTEXT_ID);	
	}	
	
}