/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: IFileManagerExtended.java,v 1.2 2005/05/03 22:19:24 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.core.file;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;

/**
 * New and improved file manager interface that supports easier feature
 * extension in the future. The get, put and delete operations have been
 * improved with plurality support, progress monitor support and increased
 * type-safety for method signatures. File manager options are now available to
 * further fine-tune the semantics of a given method such as get or put. Options
 * are described in more detail under the option type-safe enumeration in this
 * interface.
 * 
 * @author Scott E. Schneider
 */
public interface IFileManagerExtended extends IFileManager {

	/**
	 * Cookie class used for the sake of uniqueness when identifing a set of
	 * state on the file server. Might need to add algorithm in the future to
	 * create a unique identifier. The to string method is called before this is
	 * sent over the wire, using that as the unique identifier. Override the to
	 * string method in the future for greater uniqueness.
	 * 
	 * @author Scott E. Schneiderl
	 */
	public static class Cookie {
		public static final Cookie NONE = new Cookie();

		public static Cookie create() {
			return new Cookie();
		}

		private Cookie() {
		}
	}

	/**
	 * A lightweight list class to use when passing one or many file identifier
	 * names into the file manager extended interface. An already existing
	 * string array can be used to construct an instance of a name list -- for
	 * those without a pre- existing array, list operations to add to the list
	 * are available.
	 * 
	 * @author Scott E. Schneider
	 */
	public static class FileIdentifierList {
		private static final long serialVersionUID = 3977859592212198201L;

		public static FileIdentifierList create() {
			return new FileIdentifierList();
		}

		public static FileIdentifierList create(String name) {
			return new FileIdentifierList(new String[] { name });
		}

		public static FileIdentifierList create(String[] names) {
			return new FileIdentifierList(names);
		}

		private List list;

		private FileIdentifierList() {
			this.list = new ArrayList();
		}

		private FileIdentifierList(String[] names) {
			this.list = Arrays.asList(names);
		}

		public void add(File file) {
			this.list.add(file.getAbsolutePath());
		}

		public void add(String identifier) {
			this.list.add(identifier);
		}

		public Iterator iterator() {
			return this.list.iterator();
		}

		public int size() {
			return this.list.size();
		}

		public String[] getArray() {
			String[] identifiers = new String[this.size()];
			return (String[]) this.list.toArray(identifiers);
		}

		public String toString() {
			return this.list.toString();
		}
	}

	/**
	 * Type-safe enumeration class used for file manager operations, not final
	 * since enumeration values can be added in the future by classes in the
	 * local package-only. This allows options to evolve in the future without
	 * the main file manager interface breaking compatability. Another class or
	 * interface in this package would be created that can extend this Option
	 * class in order to add more enumeration values and hence options.
	 * 
	 * @author Scott E. Schneider
	 */
	public static class Option {
		public static final Option ENABLE_VERIFICATION = new Option();

		public static final Option[] NONE = new Option[0];

		public static final Option OPERATE_WITH_ATOMICITY = new Option();

		public static final Option OVERWRITE_ALL_EXISTING = new Option();

		public static final Option OVERWRITE_PARTIAL_EXISTING = new Option();

		public static final Option USE_TRANSFER_PACKAGING = new Option();

		Option() {
		} // can be instantiated/subclassed from this package
	}

	/**
	 * Delete files identified by the state on the remote file server specified
	 * by the given state identifier (cookie)
	 * 
	 * @param cookie
	 *            identifies the state on the server for this command, for
	 *            example, when a put file command is executed, optionally a
	 *            state identifier (cookie) may be given, when this is the case,
	 *            the file server stores state concerning the particular put
	 *            file command -- it will know for example which files have been
	 *            created and then it will know how to use that state to delete
	 *            them again
	 * @param monitor
	 *            the progress monitor to use for progress query and canceling
	 * @throws IOException
	 */
	public void deleteFile(Cookie cookie, IProgressMonitor monitor)
			throws IOException;

	/**
	 * Delete files identified by the remote names specified.
	 * 
	 * @param remoteIdentifiers
	 *            the remote files to delete
	 * @throws IOException
	 */
	public void deleteFile(FileIdentifierList remoteIdentifiers)
			throws IOException;

	/**
	 * Delete files identified by the remote names specified using the given
	 * progress monitor for progres tracking and cancelability.
	 * 
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be deleted
	 * @param monitor
	 *            the progress monitor to use for progress query and canceling
	 * @throws IOException
	 */
	public void deleteFile(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException;

	/**
	 * Delete files identified by the state on the remote file server specified
	 * by the given state identifier (cookie)
	 * 
	 * @param cookie
	 *            identifies the state on the server for this command, for
	 *            example, when a put file command is executed, optionally a
	 *            state identifier (cookie) may be given, when this is the case,
	 *            the file server stores state concerning the particular put
	 *            file command -- it will know for example which files have been
	 *            created and then it will know how to use that state to delete
	 *            them again
	 * @throws IOException
	 */
	public void deleteFile(IFileManagerExtended.Cookie cookie)
			throws IOException;

	/**
	 * Transfer files identified by remote names into files identified by local
	 * names, basically download files from remote to local.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied to
	 * @param remoteNames
	 *            the names identifying the remote files to be copied from
	 * @throws IOException
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException;

	/**
	 * Transfer files identified by remote names into files identified by local
	 * names, basically download files from remote to local. Use the given
	 * options to determine the exact semantics of the operation.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied to
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied from
	 * @param options
	 *            the options to use for the file manager operation
	 * @throws IOException
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options) throws IOException;

	/**
	 * Transfer files identified by remote names into files identified by local
	 * names, basically download files from remote to local. Use the given
	 * options to determine the exact semantics of the operation. Use the
	 * progress monitor to query status and support cancelability.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied to
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied from
	 * @param options
	 *            the options to use for the file manager operation
	 * @param monitor
	 *            the progress monitor to use for progress and execution control
	 * @throws IOException
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options, IProgressMonitor monitor)
			throws IOException;

	/**
	 * Transfer files identified by remote names into files identified by local
	 * names, basically download files from remote to local. Use the progress
	 * monitor to query status and support cancelability.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied to
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied from
	 * @param monitor
	 *            the progress monitor to use for progress and execution control
	 * @throws IOException
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException;

	/**
	 * Indicates whether the file server is up and running and available for
	 * commands to be sent
	 * 
	 * @return true if the file server is ready for commands to be sent to it,
	 *         it should always be available unless some exception condition
	 *         brought it down, it is initialized and brought online as the
	 *         agent controller starts up
	 */
	public boolean isServerAvailable() throws IOException;

	/**
	 * List the contents of the file (which is a directory) identified by the
	 * remote names given. The contents of the directories are returned in a
	 * file location list that can be iterated through to retrieve the
	 * individual strings stored in the list. Each string represents the textual
	 * identity of the file or file directory that is contained in the
	 * directories identified by the given remote names.
	 * 
	 * @param remoteIdentifiers
	 *            the names identifying the remote file directories to list
	 *            contents of
	 * @return the contents of the specified file directories are returned to
	 *         the caller
	 * @throws IOException
	 */
	public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers)
			throws IOException;

	/**
	 * List the contents of the file (which is a directory) identified by the
	 * remote names given. The contents of the directories are returned in a
	 * file location list that can be iterated through to retrieve the
	 * individual strings stored in the list. Each string represents the textual
	 * identity of the file or file directory that is contained in the
	 * directories identified by the given remote names.
	 * 
	 * @param remoteIdentifiers
	 *            the names identifying the remote file directories to list
	 *            contents of
	 * @param monitor
	 *            the progres monitor for querying progress and canceling
	 * @return the contents of the specified file directories are returned to
	 *         the caller
	 * @throws IOException
	 */
	public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException;

	/**
	 * Modify the permissions of a file or file directory given the names that
	 * identify the remote files and/or file directories and a permission
	 * directive in the form of a platform-specific string. The
	 * platform-specific string will be interpreted by the server-side and
	 * examples of usage will be included here when available.
	 * 
	 * @param remoteIdentifiers
	 *            the names identifying the remote files and file directories to
	 *            apply the given permission directive to
	 * @param permissionDirective
	 *            the permission directive to pass, unchanged, to the
	 *            server-side file manager implementation
	 * @throws IOException
	 */
	public void modifyPermission(FileIdentifierList remoteIdentifiers,
			String permissionDirective) throws IOException;

	/**
	 * Modify the permissions of a file or file directory given the names that
	 * identify the remote files and/or file directories and a permission
	 * directive in the form of a platform-specific string. The
	 * platform-specific string will be interpreted by the server-side and
	 * examples of usage will be included here when available.
	 * 
	 * @param remoteIdentifiers
	 *            the names identifying the remote files and file directories to
	 *            apply the given permission directive to
	 * @param permissionDirective
	 *            the permission directive to pass, unchanged, to the
	 *            server-side file manager implementation
	 * @param monitor
	 *            the progress monitor for querying progress and canceling
	 * @throws IOException
	 */
	public void modifyPermission(FileIdentifierList remoteIdentifiers,
			String permissionDirective, IProgressMonitor monitor)
			throws IOException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote.
	 * 
	 * @param cookie
	 *            the state identifier (cookie) to use for this one particular
	 *            command invocation, can be saved and then passed in to other
	 *            methods to access this stored server state (such as the delete
	 *            file method)
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to
	 * @throws IOException
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. Use the given options
	 * to determine the exact semantics of the operation.
	 * 
	 * @param cookie
	 *            the state identifier (cookie) to use for this one particular
	 *            command invocation, can be saved and then passed in to other
	 *            methods to access this stored server state (such as the delete
	 *            file method)
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to
	 * @param options
	 *            the options to use for the file manager operation
	 * @throws IOException
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options) throws IOException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. Use the given options
	 * to determine the exact semantics of the operation. The progress monitor
	 * is used for progress reporting and cancelability.
	 * 
	 * @param cookie
	 *            the state identifier (cookie) to use for this one particular
	 *            command invocation, can be saved and then passed in to other
	 *            methods to access this stored server state (such as the delete
	 *            file method)
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to
	 * @param options
	 *            the options to use for the file manager operation
	 * @param monitor
	 *            the progress monitor to use for reporting and canceling
	 * @throws IOException
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options, IProgressMonitor monitor)
			throws IOException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. The progress monitor
	 * is used for progress reporting and cancelability.
	 * 
	 * @param cookie
	 *            the state identifier (cookie) to use for this one particular
	 *            command invocation, can be saved and then passed in to other
	 *            methods to access this stored server state (such as the delete
	 *            file method)
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identiyfing the remote files to be copied to
	 * @param monitor
	 *            the progress monitor to use for reporting and canceling
	 * @throws IOException
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to
	 * @throws IOException
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. Use the given options
	 * to determine the exact semantics of the operation.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to
	 * @param options
	 *            the options to use for the file manager operation
	 * @throws IOException
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options) throws IOException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. Use the given options
	 * to determine the exact semantics of the operation. The progress monitor
	 * is used for progress reporting and cancelability.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identifying the remote files to be copied to
	 * @param options
	 *            the options to use for the file manager operation
	 * @param monitor
	 *            the progress monitor to use for reporting and canceling
	 * @throws IOException
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers,
			IFileManagerExtended.Option[] options, IProgressMonitor monitor)
			throws IOException;

	/**
	 * Transfer files identified by local names into files identified by remote
	 * names, basically upload files from local to remote. The progress monitor
	 * is used for progress reporting and cancelability.
	 * 
	 * @param localIdentifiers
	 *            the names identifying the local files to be copied from
	 * @param remoteIdentifiers
	 *            the names identiyfing the remote files to be copied to
	 * @param monitor
	 *            the progress monitor to use for reporting and canceling
	 * @throws IOException
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException;

}