/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ReusableThread.java,v 1.6 2005/02/16 22:20:08 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.execution.core.task;

/**
 * A thread wrapper that allows threads to be reused and not expire after the
 * run method has ended. The reusable thread class works in conjunction with the
 * the reuable thread pool class. A reusable thread wraps the runnable that is
 * attached to the reusable thread and keeps a run loop going while the wrapped
 * runnable is executed within. Stopping the thread does not render the reusable
 * thread useless; it can be started up again with the same runnable or a
 * different runnable. This allows for a thread to be reused across runnable
 * implementations.
 * 
 * To dispose of a reusable thread; the kill method should be invoked. The kill
 * method stops and then signals the thread to exit the wrapping run method
 * loop. If the kill method is not called, the run loop will wait for the next
 * invocation of start to begin again.
 * 
 * @author Scott E. Schneider
 */
class ReusableThread implements Runnable {

    /**
     * Indicates whether this reusuable thread has completed running its
     * associated runnable.
     */
    private boolean isComplete;

    /**
     * Indicates whether death has been requested for this reusable thread; once
     * a thread is dying it is no longer reusable by the pool.
     *  
     */
    private boolean isDeathRequested;

    /**
     * Indicates whether this reusable thread is startable or not; a reusable
     * thread is only startable after its runnable started and then completed.
     */
    private boolean isStartable;

    /**
     * The ruunable associated with this reusable thread; the runnable can be
     * swapped out via the attach method once a runnable associated with this
     * thread has completed.
     */
    private Runnable runnable;

    /**
     * The underlying Java thread that is the "real" thread that executes the
     * code within the runnable's run method. The reusable thread is a wrapper
     * around the "real" thread and has a run method that is kept alive until
     * thread death is requested via the kill method.
     */
    private final java.lang.Thread thread;

    /**
     * Constructs a reusable thread and initializes the reusable thread by
     * creating a new underlying "real" daemon thread and starting it.
     */
    ReusableThread() {
        this.thread = new java.lang.Thread(this, this.toString());
        this.thread.setDaemon(false);
        this.thread.start();
    }

    /**
     * Attaches a runnable to this reusable thread which will be executed once
     * the thread has been started. A new runnable can be attached only when the
     * reusable thread is in the startable state.
     * 
     * @param runnable
     */
    public void attach(Runnable runnable) {
        this.runnable = runnable;
        this.isComplete = false;
    }

    /**
     * Interrupts the underlying "real" thread
     */
    private void interrupt() {
        this.thread.interrupt();
    }

    /**
     * Indicates whether this reusable thread has completed executing its
     * runnable.
     * 
     * @return true if the runnable has completed executing
     */
    public boolean isComplete() {
        return this.isComplete;
    }

    /**
     * Kills the reusable thread rendering it useless for further executions.
     * The thread can be started and stopped multiple times using different
     * runnables but once it is killed it cannot be reused.
     */
    public void kill() {
        this.isDeathRequested = true;
        this.stop();
    }

    /**
     * The reusable thread's run method wraps the underlying runnable's run
     * method.
     */
    public void run() {
        while (!this.isDeathRequested) {
            try {
                synchronized (this) {
                    this.isStartable = true;
                    this.notify();
                    this.wait();
                    this.isStartable = false;
                }
                this.isComplete = false;
                this.runnable.run();
                this.isComplete = true;
                synchronized (this) {
                    this.notify();
                }
            } catch (InterruptedException e) {
                // Resets the thread, waiting to begin again
            }
        }
    }

    /**
     * Starts the reusable thread, allowing the runnable to execute.
     */
    public void start() {
        while (!this.isStartable) {
            try {
                synchronized (this) {
                    this.wait();
                }
            } catch (InterruptedException e) {
            }
        }
        synchronized (this) {
            this.notify();
        }
    }

    /**
     * Stops the reusable thread, sending an interrupt request to the underlying
     * "real" executing thread.
     */
    public void stop() {
        this.interrupt();
    }

}