/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Marshaller.java,v 1.6 2005/06/03 03:18:50 sschneid Exp $
 * 
 * Contributors: 
 * IBM Rational - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.invocation;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.core.runtime.Platform;

/**
 * This class knows how to marshal and unmarshal method calls.
 * 
 * No instances of this class can be created. All data members are class (<code>static</code> )members and all methods are class
 * methods. This class will be loaded in both the local and remote VMs, and all of its state will be properly initialized when the
 * class is loaded.
 */
public class Marshaller {

    private static final int RETURN_VALUE_IN_STREAM = 88;

    private static final int RETURN_VALUE_NOT_IN_STREAM = 188;
    
    /**
     * There is a maximum of three minutes to wait for return data or it times out
     */
    private static final int WAIT_FOR_RETURN_DATA_TIMEOUT = 180000;

    /**
     * This map is used to keep track of existing instances, on which methods are to be invoked.
     * 
     * key: unique id associated with objects on which methods will be invoked value: delegate objects on which methods will be
     * invoked
     */
    private static HashMap instanceMap = new HashMap();
    
    private static final boolean DEBUG = false;

    static {
        if (DEBUG) {
	        Thread monitor = 
		        new Thread() {
		            public void run() {
		                while (true) {
		                    try {
		                        synchronized (this) {
		                            this.wait(10000);
		                        }
		                    } catch (InterruptedException e) {
		                        // Handling not necessary
		                    }
		                    System.err.println("====================================================================================");
		                    System.err.println("[" + Marshaller.instanceMap.size() + " items in the instance map");
		                    System.err.println(Marshaller.instanceMap);
		                    System.err.println("====================================================================================");
		                }
		            }
		        };
		        monitor.setName("Marshaller Monitor");
		        monitor.setDaemon(true);
		        monitor.start();
        }
    }

    /**
     * This list is used as a queue for return values from remote method invocations.
     */
    private static ArrayList returnValueQueue = new ArrayList();

    // There should be no instances of this class.
    private Marshaller() throws IOException {
    }

    /**
     * Add an object to the instance map.
     * 
     * @param uniqueId
     * @param instance
     */
    public static void addInstanceToMap(Integer uniqueId, Object instance) {
        synchronized (instanceMap) {
            instanceMap.put(uniqueId, Marshaller.createReferenceProxy(instance));
        }
    }

    private static ReferenceProxy createReferenceProxy(Object referent) {
        return new ReferenceProxy(referent);
    }

    private static class ReferenceProxy {

        private static IFactory factory;

        private static interface IFactory {
            public Object createReference(Object referent);

            public Object resolve(Object reference);

            public String toString(Object reference);
        }

        private static class WeakReferenceFactory implements IFactory {
            public Object createReference(Object referent) {
                return new WeakReference(referent);
            }

            public Object resolve(Object reference) {
                return ((WeakReference) reference).get();
            }

            public String toString(Object reference) {
                if (reference != null) {
                    Object referent = ((WeakReference) reference).get();
                    if (referent != null) {
                        return referent.toString();
                    }
                }
                return "null";
            }
        }

        private static class PassThroughReferenceFactory implements IFactory {
            public Object createReference(Object referent) {
                return referent;
            }

            public Object resolve(Object reference) {
                return reference;
            }

            public String toString(Object reference) {
                if (reference != null) {
                    return reference.toString();
                }
                return "null";
            }
        }

        static {
            // Create appropriate reference factory (based on local or remote)
            try {
                if (Platform.getProduct() == null) {
                    factory = new PassThroughReferenceFactory();
                } else {
                    factory = new WeakReferenceFactory();
                }
            } catch (NoClassDefFoundError e) {
                factory = new PassThroughReferenceFactory();
            }
        }

        /**
         * The underlying reference being proxied
         */
        private final Object reference;

        /**
         * Construct a new reference proxy to the referent object
         * 
         * @param referent
         *            the object to be referenced
         */
        private ReferenceProxy(Object referent) {
            this.reference = ReferenceProxy.factory.createReference(referent);
        }

        /**
         * Resolve the reference proxy into the object being contained/proxied by this instance
         * 
         * @return return the reference being contained by this instance
         */
        private Object resolve() {
            return ReferenceProxy.factory.resolve(this.reference);
        }

        public String toString() {
            return ReferenceProxy.factory.toString(this.reference);
        }

    }

    /**
     * Get an object from the instance map.
     * 
     * @param uniqueId
     * @return
     */
    public static Object getInstanceFromMap(Integer uniqueId) {
        Object instance = null;
        synchronized (instanceMap) {
			ReferenceProxy referenceProxy = (ReferenceProxy) instanceMap.get(uniqueId);
			if (referenceProxy != null) {
	            instance = referenceProxy.resolve();
			}
        }
        return instance;
    }

    /**
     * Remove an object from the instance map.
     * 
     * @param uniqueId
     */
    public static void removeInstanceFromMap(Integer uniqueId) {
        synchronized (instanceMap) {
            if (instanceMap.containsKey(uniqueId)) {
                instanceMap.remove(uniqueId);
            }
        }
    }

    /**
     * Marshals a method call into a <code>byte[]</code>. The data is serialized as follows:
     * 
     * <ol>
     * <li>a unique id that is associated with the target object. This value is an <code>int</code>.
     * <li>an array of <code>Class</code> objects representing the method argument types. For methods that do not accept
     * arguments, this is a zero-length array.
     * <li>an array of <code>Object</code> s that are the arguments of the method to be invoked. For methods that do not accept
     * arguments, this is a zero-length array.
     * <li>the name of the method to be invoked. This is a <code>String</code>.
     * </ol>
     * 
     * @param callData
     * @return
     */
    public static byte[] marshalMethodCall(CallData callData) throws IOException {

        // Use a custom output stream that provides pass-by-ref for remote
        // objects.
        ByteArrayOutputStream outByteStream = new ByteArrayOutputStream();
        RemoteReferenceOutputStream outStream = new RemoteReferenceOutputStream(outByteStream);

        outStream.writeInt(callData.getTargetId().intValue());
        outStream.writeObject(callData.getArgTypes());
        outStream.writeObject(callData.getCallArgs());
        outStream.writeObject(callData.getCall());
        outStream.flush();
        
        byte[] bytes = outByteStream.toByteArray();
        outStream.close();
        outByteStream.close();
        
        return bytes;
    }

    /**
     * Unmarshals a method call into a <code>CallData</code> instance. The order of objects expected in the input byte array are
     * specified in the documentation for <code>marshalMethodCall</code>.
     * 
     * @param callData
     * @return
     * 
     * @see #marshalMethodCall(CallData)
     */
    public static CallData unmarshalMethodCall(byte[] callData) throws IOException, ClassNotFoundException {

        ByteArrayInputStream inByteStream = new ByteArrayInputStream(callData);
        RemoteReferenceInputStream in = new RemoteReferenceInputStream(inByteStream);
        Object tmp = null;

        // the targetid
        int targetId = in.readInt();

        // the method arg types
        tmp = in.readObject();
        if (!(tmp instanceof Class[]))
            throw new ClassCastException("Expected an Class[] but got \"" + tmp.getClass().getName() + "\"");
        Class[] argTypes = (Class[]) tmp;

        // the method call args
        tmp = in.readObject();
        if (!(tmp instanceof Object[]))
            throw new ClassCastException("Expected an Object[] but got \"" + tmp.getClass().getName() + "\"");
        Object[] callArgs = (Object[]) tmp;

        // the method to call
        tmp = in.readObject();
        if (!(tmp instanceof String))
            throw new ClassCastException("Expected a String but got \"" + tmp.getClass().getName() + "\"");
        String call = (String) tmp;

        in.close();
        inByteStream.close();
        
        return new CallData(new Integer(targetId), argTypes, callArgs, call);
    }

    /**
     * Marshals the return value from a method call into a <code>byte[]</code>. The data is serialized as follows:
     * 
     * <ol>
     * <li>a unique id that is associated with the target object -- the object on which the method was invoked. This value is an
     * <code>int</code>.
     * <li>an array of <code>Class</code> objects representing the method argument types. For methods that do not accept
     * arguments, this is a zero-length array.
     * <li>the name of the method to be invoked. This is a <code>String</code>.
     * <li>an indicator signifying the presence (or lack thereof) of a return value. This value is one of the <code>int</code>
     * constants defined in this class.
     * <li><b><em>CONDITIONAL</em</b>an <code>Object</code> that is the value
     *     returned as a result of the method invocation.
     * </ol>
     * 
     * @param value
     * @return
     * @throws IOException
     */
    public static byte[] marshalReturnValue(ReturnData rtnData) throws IOException {
        ByteArrayOutputStream outByteStream = new ByteArrayOutputStream();
        RemoteReferenceOutputStream outStream = new RemoteReferenceOutputStream(outByteStream);

        outStream.writeInt(rtnData.getTargetId().intValue());
        outStream.writeObject(rtnData.getArgTypes());
        outStream.writeObject(rtnData.getCall());
        if (rtnData.getReturnValue() != null) {
            outStream.writeInt(RETURN_VALUE_IN_STREAM);
            outStream.writeObject(rtnData.getReturnValue());
        } else {
            outStream.writeInt(RETURN_VALUE_NOT_IN_STREAM);
        }
        outStream.flush();
        
        byte[] bytes = outByteStream.toByteArray();
        outStream.close();
        outByteStream.close();
        
        return bytes;
    }

    /**
     * Unmarshals the result of a method call into a <code>ReturnData</code> instance. The order of objects expected in the input
     * byte array are specified in the documentation for <code>marshalReturnValue</code>.
     * 
     * @param data
     * @return
     * @throws IOException
     * @throws ClassNotFoundException
     * 
     * @see marshalReturnValue(ReturnData)
     */
    public static ReturnData unmarshalReturnValue(byte[] data) throws IOException, ClassNotFoundException {

        // Use a custom input stream that provides reference resolution for
        // remote object references.
        ByteArrayInputStream inByteStream = new ByteArrayInputStream(data);
        RemoteReferenceInputStream in = new RemoteReferenceInputStream(inByteStream);

        Object tmp = null;

        // the id associated with the object on which a method was invoked
        int targetId = in.readInt();

        // the arg types of the method that was invoked
        tmp = in.readObject();
        if (!(tmp instanceof Class[]))
            throw new ClassCastException("Expected an Class[] but got \"" + tmp.getClass().getName() + "\"");
        Class[] argTypes = (Class[]) tmp;

        // the method that was called
        tmp = in.readObject();
        if (!(tmp instanceof String))
            throw new ClassCastException("Expected a String but got \"" + tmp.getClass().getName() + "\"");
        String call = (String) tmp;

        // the return value presence indicator and the return value
        int rtnValPresence = in.readInt();
        Object rtnVal = null;
        if (rtnValPresence == RETURN_VALUE_IN_STREAM)
            rtnVal = in.readObject();
        
        in.close();
        inByteStream.close();

        return new ReturnData(new Integer(targetId), argTypes, call, rtnVal);
    }

    /**
     * Add the result of a method invocation to the return data queue.
     * 
     * @param value
     */
    public static void queueReturnValue(ReturnData value) {
        synchronized (returnValueQueue) {
            returnValueQueue.add(value);
            returnValueQueue.notifyAll();
        }
    }

    /**
     * Pull the result of a method invocation off of the return data queue.
     * 
     * @return
     */
    public static ReturnData unqueueReturnValue() {
        ReturnData rtnVal = null;
        synchronized (returnValueQueue) {
            if (!returnValueQueue.isEmpty())
                rtnVal = (ReturnData) returnValueQueue.remove(0);
        }
        return rtnVal;
    }

    /**
     * Get a copy of the return value at the front of the queue. This method is used by remote stubs as a part of the check to
     * ensure that a return value is the result of a given method invocation. This helps to insure that a remote stub doesn't pull
     * the wrong value off of the queue.
     * 
     * @return
     */
    public static ReturnData peekReturnValue() {
        ReturnData rtnVal = null;
        synchronized (returnValueQueue) {
            if (!returnValueQueue.isEmpty())
                rtnVal = (ReturnData) returnValueQueue.get(0);
        }
        return rtnVal;
    }

    /**
     * Are there any return values in the queue?
     * 
     * @return
     */
    public static boolean isReturnDataAvailable() {
        boolean rtnVal = false;
        synchronized (returnValueQueue) {
            rtnVal = !returnValueQueue.isEmpty();
        }
        return rtnVal;
    }

    /**
     * Wait until there is data in the return queue. This method blocks the calling thread until return data is present in the
     * queue or the timeout occurs.
     */
    public static void waitForReturnData() {
        while (!isReturnDataAvailable()) {
            try {
                synchronized (returnValueQueue) {
                    returnValueQueue.wait(Marshaller.WAIT_FOR_RETURN_DATA_TIMEOUT);
                }
            } catch (InterruptedException e) {
            }
        }
    }

}