/**********************************************************************
Copyright (c) 2005 IBM Corporation and others.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Eclipse Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/epl-v10.html
$Id: UpdateablePathClassLoader.java,v 1.4 2005/02/16 22:20:15 qiyanli Exp $

Contributors:
 IBM Corporation - initial implementation
**********************************************************************/
package org.eclipse.hyades.execution.invocation;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * This classloader re-searches the system classpath (which is specified via
 * the system property, <code>java.class.path</code>) every time it fields
 * a request to find a class or a resource. The effect is that the classpath
 * that we use is updateable at runtime.
 */
public class UpdateablePathClassLoader extends ClassLoader {

	/**
	 * @param parent
	 */
	public UpdateablePathClassLoader(ClassLoader parent) {
		super(parent);
	}

	/**
	 * 
	 */
	public UpdateablePathClassLoader() {
		super();
	}

	/**
	 * This version searches the system classpath every time this method
	 * is called, so that runtime updates of that property are utilized.
	 * 
	 * The property used for the system classpath value is
	 * <code>java.class.path</code>
	 * 
	 * @see java.lang.ClassLoader#findClass(java.lang.String)
	 * @see java.lang.System.getProperty(String, String)
	 * @see java.lang.System.setProperty(String, String)
	 */
	protected Class findClass(String name) throws ClassNotFoundException {
		StringTokenizer tokenizer = getSearchPathTokens();
		String searchName = name.replace('.', File.separatorChar);
		File classFile = null;
		while ( tokenizer.hasMoreTokens() ) {
			StringBuffer buf = new StringBuffer(tokenizer.nextToken());
			buf.append(File.separatorChar);
			buf.append(searchName);
			buf.append(".class");
			classFile = new File(buf.toString());
			if ( classFile.exists() )
				break;
		}
		if ( !classFile.exists() ) {
			// Couldn't find it. Give any other classloaders (that haven't already tried)
			// a chance to find it.
			throw new ClassNotFoundException(name);
		}
		
		
		FileInputStream fis = null;
		byte[] classBytes = null;
		try {
			fis = new FileInputStream(classFile);
			classBytes = new byte[fis.available()];
			fis.read(classBytes);
		}
		catch ( FileNotFoundException e ) {
			// this shouldn't happen
			throw new ClassNotFoundException(
				"Could not open input file: \""+classFile.getPath()+"\"");
		}
		catch ( IOException e ) {
			throw new ClassNotFoundException(
				"Could not read '.class' file: \""+classFile.getPath()+"\"");
		}
		
		return defineClass(name, classBytes, 0, classBytes.length);
	}

	/**
	 * This version searches the system classpath every time this method
	 * is called, so that runtime updates of that property are utilized.
	 * 
	 * The property used for the system classpath value is
	 * <code>java.class.path</code>
	 * 
	 * @see java.lang.ClassLoader#findResource(java.lang.String)
	 * @see java.lang.System.getProperty(String, String)
	 * @see java.lang.System.setProperty(String, String)
	 */
	protected URL findResource(String name) {
		URL resourceURL = null;
		StringTokenizer tokenizer = getSearchPathTokens();
		File resourceFile = null;
		while ( tokenizer.hasMoreTokens() ) {
			StringBuffer buf = new StringBuffer(tokenizer.nextToken());
			buf.append(File.separatorChar);
			buf.append(name);
			resourceFile = new File(buf.toString());
			if ( resourceFile.exists() )
				break;
		}
		try {
			if ( resourceFile.exists() )
				resourceURL = resourceFile.toURL();
		}
		catch ( MalformedURLException e){
			System.err.println("Could not find resource \""+name+"\"\n"+e);
		}

		return resourceURL;
	}

	/**
	 * This version searches the system classpath every time this method
	 * is called, so that runtime updates of that property are utilized.
	 * 
	 * The property used for the system classpath value is
	 * <code>java.class.path</code>
	 * 
	 * @see java.lang.ClassLoader#findResources(java.lang.String)
	 * @see java.lang.System.getProperty(String, String)
	 * @see java.lang.System.setProperty(String, String)
	 */
	protected Enumeration findResources(String name) throws IOException {
		Vector resourceURLs = new Vector();
		StringTokenizer tokenizer = getSearchPathTokens();
		File resourceFile = null;
		while ( tokenizer.hasMoreTokens() ) {
			StringBuffer buf = new StringBuffer(tokenizer.nextToken());
			buf.append(File.separatorChar);
			buf.append(name);
			resourceFile = new File(buf.toString());
			if ( resourceFile.exists() ) {
				try {
					resourceURLs.add(resourceFile.toURL());
				}
				catch ( MalformedURLException e){
					System.err.println("Could not find resource \""+name+"\"\n"+e);
				}
			}
		}
		return resourceURLs.elements();
	}
	
	private StringTokenizer getSearchPathTokens(){
		String classpath = System.getProperty("java.class.path", ".");
		return new StringTokenizer(classpath, File.pathSeparator);
	}

}
