/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BaseEditorExtension.java,v 1.8 2005/03/02 23:41:34 bjiang Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.editor.extension;

import java.lang.reflect.InvocationTargetException;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.editor.extension.IWriteAccessChangeListener;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.navigator.EObjectProxyNode;
import org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter;
import org.eclipse.hyades.ui.editor.EditorExtension;
import org.eclipse.hyades.ui.editor.IHyadesEditorExtension;
import org.eclipse.hyades.ui.editor.IHyadesEditorPart;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.operation.ModalContext;
import org.eclipse.jface.util.ListenerList;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

/**
 * @author marcelop, bcormier
 * @since 3.0.0
 */
public abstract class BaseEditorExtension 
extends EditorExtension implements ISynchronizedEditorAdapter, IHyadesEditorExtension
{
	private ListenerList writeAccessChangeListeners;
	
	private boolean dirty = false;
	private Clipboard clipboard;
	 
	/**
	 * The constructor for the class.
	 */
	public BaseEditorExtension()
	{
		writeAccessChangeListeners = new ListenerList();
	}
	
	/**
	 * Disposes any resource used by this object. There are no guarantees that 
	 * this method is invoked by this object's container or by the Hyades framework.
	 */
	public void dispose()
	{
		if(clipboard != null)
		{
			clipboard.dispose();
			clipboard = null;
		}
		writeAccessChangeListeners.clear();
		
		super.dispose();
	}
	
	/**
	 * Returns an object which is an instance of the given class
	 * associated with this object. Returns <code>null</code> if
	 * no such object can be found.
	 *
	 * @param adapter the adapter class to look up
	 * @return a object castable to the given class, 
	 *    or <code>null</code> if this object does not
	 *    have an adapter for the given class
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter)
	{
		if(ISynchronizedEditorAdapter.class == adapter)
			return this;
			
		return super.getAdapter(adapter);
	}

	/**
	 * Returns this editor extension's resource set.
	 * @return ResourceSet
	 */
	protected ResourceSet getResourceSet()
	{
		return ((EObject)getHyadesEditorPart().getEditorObject()).eResource().getResourceSet();	
	}
	
	/**
	 * Returns the extension editor's clipboard.
	 * @return Clipboard
	 */
	public Clipboard getClipboard()
	{
		if(clipboard == null)
			clipboard = new Clipboard(getHyadesEditorPart().getEditorPart().getEditorSite().getShell().getDisplay());
		return clipboard;
	}
	
	/**
	 * Adds a write access change listener to the list of registered listeners.
	 * @param listener the write access change listener to add
	 */
	public void addListener(IWriteAccessChangeListener listener)
	{
		writeAccessChangeListeners.add(listener);
	}
	
	/**
	 * Removes a write access change listener from the list of registered listeners.
	 * @param listener the write access change listener to remove
	 */
	public void removeListener(IWriteAccessChangeListener listener)
	{
		writeAccessChangeListeners.remove(listener);
	}
	
	/**
	 * Notifies a write access change to the registered listeners.
	 * @param file the resource from which the write access change has been raised
	 * @param isReadOnly the write access change that has been raised
	 */
	protected void notifyWriteAccessChangeListeners(IFile file, boolean isReadOnly)
	{
		if(file == null)
			return;
			
		Object[] listeners = writeAccessChangeListeners.getListeners();
		for(int i = 0, maxi = listeners.length; i < maxi; i++)
			((IWriteAccessChangeListener)listeners[i]).writeAccessChanged(file, isReadOnly);
	}
	
	/**
	 * Marks the editor as dirty.
	 */
	public synchronized void markDirty()
	{
		if(!dirty)
		{
			dirty = true;
			getHyadesEditorPart().firePropertyChange(IHyadesEditorPart.PROP_DIRTY);
		}
	}
	
	/**
	 * Clears the dirty flag without refreshing the editor.
	 */
	protected void clearDirty()
	{
		dirty = false;
	}
	
	/**
	 * Returns whether the contents of this editor extension have changed since the 
	 * last save operation.
	 * @return <code>true</code> if the contents have been modified and need
	 * saving, and <code>false</code> otherwise.
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#isDirty()
	 */
	public boolean isDirty()
	{
		return dirty;
	}

	/**
	* called by doSave() to allow the subclass to verify the editor contents before saving the
	* file.
	* 
	* @return true to continue the save operation, false to abort the save
	*/
	protected boolean isOkToSave()
	{
		/*
		 * Always return true here.  A subclass could do some processing of its own
		 * and return false to abort the save if it needs to...
		 */
		return true;
	}
	
	/**
	 * Saves the contents of this editor extension.
	 * @param monitor the progress monitor
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public final void doSave(IProgressMonitor monitor) {
		if (this.isOkToSave()) {
			WorkspaceModifyOperation operation = new WorkspaceModifyOperation() {
				protected void execute(IProgressMonitor progressMonitor)
						throws CoreException, InvocationTargetException,
						InterruptedException {
					try {
						save(progressMonitor);
						clearDirty();
					} catch (Exception e) {
						handleException(e);
					}
				}
			};
			IActionBars actionBars = getHyadesEditorPart().getEditorPart()
					.getEditorSite().getActionBars();
			if ((actionBars != null)
					&& (actionBars.getStatusLineManager() != null)) {
				actionBars.getStatusLineManager().setErrorMessage(null);
				actionBars.getStatusLineManager().setMessage(null);
				monitor = actionBars.getStatusLineManager()
						.getProgressMonitor();
				actionBars.getStatusLineManager().setCancelEnabled(false);
			} else if (monitor == null) {
				monitor = new NullProgressMonitor();
			}
			try {
				ModalContext.run(operation, false, monitor, Display
						.getCurrent());
			} catch (Exception e) {
				handleException(e);
			}
		}
	}
	
	/**
	 * Saves the contents of this editor extension.
	 * @param monitor the progress monitor to report the work
	 * @throws Exception this method is supposed to throw any exception.
	 */
	public void save(IProgressMonitor monitor)
	throws Exception
	{
		Resource[] resources = (Resource[])getResourceSet().getResources().toArray(new Resource[getResourceSet().getResources().size()]);
		int length = resources.length;
		int counter = 0;
		monitor.beginTask("", length); //$NON-NLS-1$
		
		for(int i = 0, maxi = resources.length; i < maxi; i++)
		{
			if(resources[i].isModified())
			{
				logSaveMessage(monitor, resources[i], length, counter++);
				EMFUtil.save(resources[i]);
			}
			monitor.worked(1);
		}
		monitor.done();
	}
	
	/**
	 * Displays a message when saving a resource.
	 * e.g : (3 of 5) myResource.res
	 * @param monitor the progress monitor
	 * @param resource the current resource to save
	 * @param total the total number of resources to save
	 * @param current the number of the current resource
	 */
	protected void logSaveMessage(IProgressMonitor monitor, Resource resource, int total, int current)
	{
		String resourceName = EMFUtil.getFilePath(resource);
		monitor.subTask(TestUIPlugin.getString("PRG_SAVE_RSRS", new String[]{Integer.toString(current), Integer.toString(total), resourceName})); //$NON-NLS-1$
	}
	
	/**
	 * Handles any exception thrown during the 
	 * {@link #save(IProgressMonitor)} execution.
	 * @param exception
	 */
	protected void handleException(Exception exception)
	{
		TestUIPlugin.logError(exception);
		String filePath = EMFUtil.getFilePath((EObject)getHyadesEditorPart().getEditorObject());
 		UIUtil.openSaveFileErrorDialog(getHyadesEditorPart().getEditorPart().getSite().getShell(), filePath, exception);
	}
	
	/**
	 * Reloades the editor's object.  This method is invoked by the synchronizer
	 * adapter and should return <code>true</code> if the synchronizer is not
	 * supposed to continue with the default treatment.
	 * @return boolean
	 */
	protected boolean reloadEditorObject()
	{
		Resource resource = ((EObject)getHyadesEditorPart().getEditorObject()).eResource(); 
		
		resource.unload();
		try
		{
			resource.load(EMFUtil.RESOURCE_OPTIONS);
		}
		catch(Exception e)
		{
			TestUIPlugin.logError(e);
		}
		
		if(!resource.getContents().isEmpty())
			getHyadesEditorPart().setEditorObject(resource.getContents().get(0));
		else
			getHyadesEditorPart().setEditorObject(null);

		return false;
	}

	/**
	 * Notifies this listener that the selection has changed.
	 * @param part the workbench part containing the selection
	 * @param selection the current selection
	 * @see org.eclipse.ui.ISelectionListener#selectionChanged(org.eclipse.ui.IWorkbenchPart, org.eclipse.jface.viewers.ISelection)
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection)
	{
		if(part instanceof TestNavigator)
		{
			if(!(selection instanceof IStructuredSelection))
				return;

			IStructuredSelection structuredSelection = (IStructuredSelection)selection;
			if(structuredSelection.size() != 1) 
				return;
			
			EObject obj2Select = null;
			
			Object first = structuredSelection.getFirstElement();
			if (first instanceof TPFExecutionResult) {
				URI uri = getObjectURI(first);
				if(uri == null)
					return;
				ResourceSet resSet = getResourceSet();
			    Resource res = resSet.getResource(uri.trimFragment(), false);
			    if (res != null)
			    {
					String id = uri.fragment();
				    for (Iterator i = res.getAllContents(); i.hasNext(); )
				    {
				      EObject eObject = (EObject)i.next();
				      if (id.equals(EcoreUtil.getID(eObject)) && eObject instanceof TPFExecutionResult)
				      {
				      	obj2Select = eObject;
				      }
				    }
			    }
			} else if (first instanceof EObjectProxyNode) {
				EObjectProxyNode proxy = (EObjectProxyNode) first;
				URI uri = proxy.getOriginatorURI();
				if(uri != null) {
					obj2Select = getResourceSet().getEObject(uri, false);
				}
			}
	
			if(obj2Select == null)
				return;
			
			setSelection(new StructuredSelection(obj2Select));
		}
	}
	
	/**
	 * Returns the complete EMF URI which includes the resource information.
	 * @param object
	 * @return
	 */
	protected URI getObjectURI(Object object)
	{
		if(object != null)
		{
			if(object instanceof EObject)
			{
				EObject eObject = (EObject)object;
				if(eObject.eResource() != null)
					return eObject.eResource().getURI().appendFragment(eObject.eResource().getURIFragment(eObject));
			}
		}
		
		return null;
	}
	
	/**
	 * This method is invoked when the editor file has been changed.
	 * @return <code>true</code> if the editor synchronizer should not
	 * perform the default treatment.
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileChanged()
	 */
	public boolean editorFileChanged()
	{
		return false;
	}

	/**
	 * This method is invoked when the editor file has been deleted.
	 * @return <code>true</code> if the editor synchronizer should not
	 * perform the default treatment.
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileDeleted()
	 */
	public boolean editorFileDeleted()
	{
		return false;
	}

	/**
	 * This method is invoked when the editor file has been changed.
	 * @return <code>true</code> if the editor synchronizer should not
	 * perform the default treatment.
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#editorFileWriteAccessChanged(boolean)
	 */
	public boolean editorFileWriteAccessChanged(boolean isReadOnly)
	{
		notifyWriteAccessChangeListeners(EMFUtil.getWorkspaceFile((EObject)getHyadesEditorPart().getEditorObject()), isReadOnly);
		return false;
	}

	/**
	 * Handles modifications in one or more monitored files.  The
	 * lists contains the modified IFiles.
	 * @param changedFiles
	 * @param removedFiles
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#handle(java.util.List, java.util.List)
	 */
	public void handle(List changedFiles, List removedFiles)
	{

	}

	/**
	 * Triggers a reload in the editor.
	 * @return <code>true</code> if the editor was reloaded
	 * or <code>false</code> otherwise. 
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#reload()
	 */
	public boolean reload()
	{
		String fragment = null;
		IStructuredSelection structuredSelection = getSelection();
		if(structuredSelection.size() == 1)
		{
			Object selection = structuredSelection.getFirstElement();
			if(selection instanceof EObject)
				fragment = ((EObject)selection).eResource().getURIFragment((EObject)selection);
		}
		
		if(!reloadEditorObject())
		{
			Object editorObject = getHyadesEditorPart().getEditorObject();
			refreshContent(editorObject);
			
			if((editorObject instanceof EObject) && (fragment != null))
			{
				Object selection = ((EObject)(editorObject)).eResource().getEObject(fragment);
				if(selection != null)
					structuredSelection = new StructuredSelection(selection);
				else
					structuredSelection = StructuredSelection.EMPTY;
					
				setSelection(structuredSelection);
			}
			
			dirty = false;
			getHyadesEditorPart().firePropertyChange(IHyadesEditorPart.PROP_DIRTY);			
		}
		
		return true;
	}
	
	/**
	 * This method is invoked when the editor content should be saved.
	 * @param wasDeleted indicates whether the editor file has been deleted.
	 * @return <code>true</code> if the editor synchronizer should not
	 * perform the default treatment.
	 * @see org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter#doSaveEditorFile(boolean)
	 */
	public boolean doSaveEditorFile(boolean wasDeleted)
	{
		getHyadesEditorPart().getEditorPart().doSave(new NullProgressMonitor());
		return false;
	}
}
