/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolForm.java,v 1.11 2005/04/01 19:24:06 bjiang Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.editor.form;

import java.util.ArrayList;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.edit.datapool.IDatapoolListener;
import org.eclipse.hyades.edit.datapool.IDatapoolVariable;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.adapter.DatapoolVariablePropertySource;
import org.eclipse.hyades.test.ui.adapter.NamedElementPropertySource;
import org.eclipse.hyades.test.ui.editor.form.util.DatapoolChildrenTreeSection;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.NamedElementSection;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.action.AddEquivalenceClassAction;
import org.eclipse.hyades.test.ui.internal.action.AddVariableAction;
import org.eclipse.hyades.test.ui.internal.action.MoveEquivalenceClassDown;
import org.eclipse.hyades.test.ui.internal.action.MoveEquivalenceClassUp;
import org.eclipse.hyades.test.ui.internal.action.MoveVariableDown;
import org.eclipse.hyades.test.ui.internal.action.MoveVariableUp;
import org.eclipse.hyades.test.ui.internal.action.OpenEquivalenceClassAction;
import org.eclipse.hyades.test.ui.internal.action.RemoveEquivalenceClassAction;
import org.eclipse.hyades.test.ui.internal.action.RemoveVariableAction;
import org.eclipse.hyades.test.ui.internal.editor.extension.DatapoolEditorExtension;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.util.ListenerList;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.model.IWorkbenchAdapter;
import org.eclipse.ui.views.properties.IPropertySource;

/**
 * This is the <i>Overview</i> tab of datapool editor.
 * 
 * @author psun
 * @author bjiang
 */
public class DatapoolForm extends EditorForm implements ISelectionProvider, ISelectionChangedListener {

	private NamedElementSection generalInfoSection;	
	private DatapoolChildrenTreeSection variablesSection;
	private ArrayList variablePropertySourceItems = new ArrayList();
	private DatapoolChildrenTreeSection equivalenceClassSection;
	private ArrayList equivalenceClassPropertySourceItems = new ArrayList();
	private IDatapoolListener datapoolListener;
	private ListenerList selectionChangeListenerList = new ListenerList();
	private TreeViewer focusedViewer;
	
	class DatapoolListener implements IDatapoolListener
	{
		
		public void cellChanged(IDatapool datapool, int equivalenceClassIndex,
				int recordIndex, int variableIndex) {}
		
		public void equivalenceClassAdded(IDatapool datapool,
				int newEquivalenceClassIndex) {
			((DatapoolEditorExtension)getBaseEditorExtension()).markDirty();
			IDatapoolEquivalenceClass eqClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(newEquivalenceClassIndex);
			IPropertySource eqClsPropSource = new NamedElementPropertySource(eqClass);
			equivalenceClassSection.getTreeViewer().add(datapool, eqClsPropSource);
			if(newEquivalenceClassIndex < equivalenceClassPropertySourceItems.size())
			    equivalenceClassPropertySourceItems.add(newEquivalenceClassIndex, eqClsPropSource);
			else if(newEquivalenceClassIndex == equivalenceClassPropertySourceItems.size())
			    equivalenceClassPropertySourceItems.add(eqClsPropSource);
			else 
			    return;
			equivalenceClassSection.getTreeViewer().refresh();
			equivalenceClassSection.getTreeViewer().setSelection(new StructuredSelection(eqClsPropSource));
		}
		public void equivalenceClassChanged(IDatapool datapool,
				int equivalenceClassIndex, String oldName) {
		    equivalenceClassSection.getTreeViewer().refresh();
		}
		public void equivalenceClassChanged(IDatapool datapool,
				int equivalenceClassIndex) {
			((DatapoolEditorExtension)getBaseEditorExtension()).markDirty();
			IPropertySource eqClsPropSource = (IPropertySource)equivalenceClassPropertySourceItems.get(equivalenceClassIndex);
		    equivalenceClassSection.getTreeViewer().update(eqClsPropSource, null);
		    equivalenceClassSection.getTreeViewer().setSelection(new StructuredSelection(eqClsPropSource));
		}
		public void equivalenceClassMoved(IDatapool datapool,
				int sourceEquivalenceClassIndex, int targetEquivalenceClassIndex) {
			((DatapoolEditorExtension)getBaseEditorExtension()).markDirty();
		    Object obj = equivalenceClassPropertySourceItems.remove(sourceEquivalenceClassIndex);
		    equivalenceClassPropertySourceItems.add(targetEquivalenceClassIndex, obj);
		    equivalenceClassSection.getTreeViewer().refresh();
		    equivalenceClassSection.getTreeViewer().setSelection(new StructuredSelection(equivalenceClassPropertySourceItems.get(targetEquivalenceClassIndex)));
		}
		public void equivalenceClassRemoved(IDatapool datapool,
				int equivalenceClassIndex) {
			((DatapoolEditorExtension)getBaseEditorExtension()).markDirty();
			Object eqClassPropSource = equivalenceClassPropertySourceItems.get(equivalenceClassIndex);
			if(eqClassPropSource != null)
			{
			    // default equivalence class is not in data model thus does not get persisted. 
				// Add this functionality when this is resolved.
				/*int defEC = datapool.getDefaultEquivalenceClassIndex();
				TreeItem[] items = equivalenceClassSection.getTreeViewer().getTree().getItems();
				if(defEC > -1 && defEC < items.length)
					items[defEC].setChecked(false);*/
				
				equivalenceClassSection.getTreeViewer().remove(eqClassPropSource);
				equivalenceClassPropertySourceItems.remove(equivalenceClassIndex);
				
				/*items = equivalenceClassSection.getTreeViewer().getTree().getItems();
				if(defEC > -1 && defEC < items.length)
					items[defEC].setChecked(true);*/
					
			    equivalenceClassSection.getTreeViewer().refresh();
				int selection = (equivalenceClassIndex < datapool.getEquivalenceClassCount()) ? equivalenceClassIndex : (equivalenceClassIndex - 1);
				equivalenceClassSection.getTreeViewer().setSelection(new StructuredSelection(equivalenceClassPropertySourceItems.get(selection)));
			}
		}
		public void equivalenceClassReordered(IDatapool datapool,
				int equivalenceClassIndex) {
		    equivalenceClassSection.getTreeViewer().refresh();
		}
		public void recordAdded(IDatapool datapool, int EquivClassIndex,
				int newRecordIndex) {}
		public void recordChanged(IDatapool datapool, int EquivClassIndex,
				int recordIndex) {}
		public void recordMoved(IDatapool datapool, int EquivClassIndex,
				int sourceRecordIndex, int targetRecordIndex) {}
		public void recordRemoved(IDatapool datapool, int EquivClassIndex,
				int recordIndex) {
		}
		public void save(IDatapool datapool) {}
		public void variableAdded(IDatapool datapool, int newVariableIndex) {
		    IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(newVariableIndex);
			IPropertySource variablePropSource = new DatapoolVariablePropertySource(variable);
			variablesSection.getTreeViewer().add(datapool, variablePropSource);
			if(newVariableIndex < variablePropertySourceItems.size())
			    variablePropertySourceItems.add(newVariableIndex, variablePropSource);
			else if(newVariableIndex == variablePropertySourceItems.size())
			    variablePropertySourceItems.add(variablePropSource);
			else
			    return;
			variablesSection.getTreeViewer().refresh();
			variablesSection.getTreeViewer().setSelection(new StructuredSelection(variablePropSource));
		}
		public void variableChanged(IDatapool datapool, int variableIndex,
				String oldName) {
			variableChanged(datapool, variableIndex);
		}
		public void variableChanged(IDatapool datapool, int variableIndex) {
			IPropertySource variablePropSource = (IPropertySource)variablePropertySourceItems.get(variableIndex);
		    variablesSection.getTreeViewer().update(variablePropSource, null);
		    variablesSection.getTreeViewer().setSelection(new StructuredSelection(variablePropSource));
		}
		public void variableMoved(IDatapool datapool, int sourceVariableIndex,
				int targetVariableIndex) {
		    Object obj = variablePropertySourceItems.remove(sourceVariableIndex);
		    variablePropertySourceItems.add(targetVariableIndex, obj);
		    variablesSection.getTreeViewer().refresh();
		    variablesSection.getTreeViewer().setSelection(new StructuredSelection(variablePropertySourceItems.get(targetVariableIndex)));
		}
		public void variableRemoved(IDatapool datapool, int variableIndex) {
		    Object variablePropSource = variablePropertySourceItems.get(variableIndex);
			if(variablePropSource != null)
			{
				variablesSection.getTreeViewer().remove(variablePropSource);
				variablePropertySourceItems.remove(variableIndex);
				int selection = (variableIndex < datapool.getVariableCount()) ? variableIndex : (variableIndex - 1);
				//bugzillar#85035: array index exception
				if ( selection > -1 )
					variablesSection.getTreeViewer().setSelection(new StructuredSelection(variablePropertySourceItems.get(selection)));
			}
		}
	}

	/**
	 * Constructor for DatapoolForm
	 * @param datapoolEditorExtension
	 * @param widgetFactory
	 */
	public DatapoolForm(DatapoolEditorExtension datapoolEditorExtension, WidgetFactory widgetFactory)
	{
		super(datapoolEditorExtension, widgetFactory);
		setHeadingText(TestUIPlugin.getString("W_DATAPOOL")); //$NON-NLS-1$
	}

	/**
	 * @see org.eclipse.update.ui.forms.internal.IForm#dispose()
	 */
	public void dispose()
	{
	    if(generalInfoSection != null)
	    {
			generalInfoSection.dispose();
			generalInfoSection = null;
	    }
	    
	    if(equivalenceClassSection != null)
	    {
	        equivalenceClassSection.dispose();
	        equivalenceClassSection = null;
	        equivalenceClassPropertySourceItems.clear();
	        equivalenceClassPropertySourceItems = null;
	    }
	    
	    if(variablesSection != null)
	    {
	        variablesSection.dispose();
	        variablesSection = null;
	        variablePropertySourceItems.clear();
	        variablePropertySourceItems = null;
	    }
	    
	    if(selectionChangeListenerList != null)
	    {
	        selectionChangeListenerList.clear();
	        selectionChangeListenerList = null;
	    }

		super.dispose();
	}

	/**
	 * Returns the datapool manipulated by this form.
	 * @return DPLDatapool
	 */
	protected DPLDatapool getDatapool()
	{
		return ((DatapoolEditorExtension)getBaseEditorExtension()).getDatapool();		
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorForm#createControl()
	 */
	public Control createControl()
	{
		Control control = super.createControl();
		WorkbenchHelp.setHelp(getControl(), TestUIPlugin.getID() + ContextIds.DATA_FORM);
		return control;
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#createEditorFormContents(org.eclipse.swt.widgets.Composite)
	 */
	protected void createEditorFormContents(Composite parent)
	{
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.makeColumnsEqualWidth = true;
		gridLayout.marginWidth = 10;
		gridLayout.horizontalSpacing = 15;
		parent.setLayout(gridLayout);

		Composite leftColumn = createColumn(parent);
		Composite rightColumn = createColumn(parent);

		Control control = createGenericInformation(leftColumn);
		control.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING));
		WorkbenchHelp.setHelp(control, TestUIPlugin.getID() + ContextIds.DATA_NAME_FORM);								

		createVariableSection(rightColumn);
		variablesSection.getTreeViewer().addSelectionChangedListener(this);
		createEquivalenceClassSection(rightColumn);
		equivalenceClassSection.getTreeViewer().addSelectionChangedListener(this);
		getBaseEditorExtension().getHyadesEditorPart().getEditorPart().getEditorSite().setSelectionProvider(this);
	}
	
	/**
	 * Creates the generic information form section.
	 * @return FormSection
	 */
	protected Control createGenericInformation(Composite parent)
	{
		generalInfoSection = new NamedElementSection(this)
		{
			private StyledText fileText;
			
			protected void addSouthControls(Composite parent, FormWidgetFactory formWidgetFactory)
			{
				Composite detailComposite = getWidgetFactory().createComposite(parent);
				GridLayout gridLayout = new GridLayout(2, false);
				gridLayout.marginHeight = 0;
				gridLayout.marginWidth = 0;
				detailComposite.setLayout(gridLayout);
				detailComposite.setLayoutData(GridDataUtil.createHorizontalFill());

				getWidgetFactory().createLabel(detailComposite, TestUIPlugin.getString("L_FILE")); //$NON-NLS-1$
				fileText = getWidgetFactory().createStyledText(detailComposite, SWT.FULL_SELECTION | SWT.SINGLE);
				fileText.setEnabled(false);
				fileText.setLayoutData(GridDataUtil.createHorizontalFill());
				WorkbenchHelp.setHelp(fileText, TestUIPlugin.getID() + ContextIds.DATA_TEXT_FORM);								
				
			}
			
			public void setInput(Object object)
			{
				super.setInput(object);				
				fileText.setText(EMFUtil.getFilePath(getDatapool()));					
			}
		};
		registerSection(generalInfoSection);		
		generalInfoSection.setHeaderText(TestUIPlugin.getString("EDT_GENERAL_INFO")); //$NON-NLS-1$
		generalInfoSection.setDescription(TestUIPlugin.getString("EDT_GENERAL_DSC")); //$NON-NLS-1$
		return generalInfoSection.createControl(parent, getWidgetFactory());
	}
	
	protected Control createEquivalenceClassSection(Composite parent)
	{
		IAction[] equivalenceClassActions = createEquivalenceClassActions();
		// default equivalence class is not in data model thus does not get persisted. 
		// Add this functionality when this is resolved.
		equivalenceClassSection = new DatapoolChildrenTreeSection(this, equivalenceClassActions, SWT.SINGLE) { // | SWT.CHECK
			public Object[] getChildren(IDatapool datapool) {
				if(datapool == null)
					return new Object[0];

				// if the current tree items do not match the datapool object, recreate them.
				if(equivalenceClassPropertySourceItems.size() != datapool.getEquivalenceClassCount())
				{
				    equivalenceClassPropertySourceItems.clear();
					IPropertySource[] eqClsPropSources = new IPropertySource[datapool.getEquivalenceClassCount()];				
					for(int i = 0; i < eqClsPropSources.length; i++)
					{
						eqClsPropSources[i] = new NamedElementPropertySource((org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass)datapool.getEquivalenceClass(i));
						equivalenceClassPropertySourceItems.add(eqClsPropSources[i]);
					}
				}
				return equivalenceClassPropertySourceItems.toArray();
			}

			public Object[] getElements(IDatapool inputElement) {				
				return getChildren(inputElement);
			}

			public Object getParent(Object element) {
				if(element instanceof IDatapoolEquivalenceClass)
					return ((IDatapoolEquivalenceClass)element).getDatapool();
				return null;
			}
			public boolean hasChildren(IDatapool element) {
				if(element != null && element.getEquivalenceClassCount() > 0)
					return true;
				return false;
			}
			/* (non-Javadoc)
			 * @see org.eclipse.hyades.test.ui.editor.form.util.DatapoolChildrenTreeSection#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			    // default equivalence class is not in data model thus does not get persisted. 
				// Add this functionality when this is resolved.
				/*if(newInput != null && newInput instanceof IDatapool)
				{
					int defIndex = ((IDatapool)newInput).getDefaultEquivalenceClassIndex();
					if(defIndex > -1 && getTreeViewer().getTree().getItemCount() > defIndex)
					{
						TreeItem defItem = getTreeViewer().getTree().getItems()[defIndex];
						defItem.setChecked(true);
					}
				}*/
			}
			
			protected void adjustClient(Composite parent)
			{
				super.adjustClient(parent);
				// default equivalence class is not in data model thus does not get persisted. 
				// Add this functionality when this is resolved.
				/*getTreeViewer().getTree().addSelectionListener(new SelectionListener() {
					public void widgetSelected(SelectionEvent e) {
						if(e.detail == SWT.CHECK)
						{
							int oldDefIndex = getDatapool().getDefaultEquivalenceClassIndex();
							TreeItem[] items = getTreeViewer().getTree().getItems();
							int newDefIndex = Arrays.asList(items).indexOf(e.item);
							items[newDefIndex].setChecked(true); // keep it checked
							
							if(newDefIndex != oldDefIndex)
							{
								if(oldDefIndex > -1 && oldDefIndex < items.length)
								    items[oldDefIndex].setChecked(false);							
						
								getDatapool().setDefaultEquivalenceClassIndex(newDefIndex);
								((DatapoolEditorExtension)getBaseEditorExtension()).markDirty();
								
								getTreeViewer().refresh();
							}
						}
					}

					public void widgetDefaultSelected(SelectionEvent e) {
						return; // double click
					}
				});*/
			}
		};

		registerSection(equivalenceClassSection);		
		equivalenceClassSection.setHeaderText(TestUIPlugin.getString("DatapoolEditor.EquivalenceClasses")); //$NON-NLS-1$
		equivalenceClassSection.setDescription(TestUIPlugin.getString("DatapoolEditor.EquivalenceClassesSectionDescription")); //$NON-NLS-1$
		equivalenceClassSection.setCollapsable(true);
		Control control = equivalenceClassSection.createControl(parent, getWidgetFactory());
		control.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		// default equivalence class is not in data model thus does not get persisted. 
		// Add this functionality back when this is resolved.
		/*equivalenceClassSection.getTreeViewer().setLabelProvider(new LabelProvider(){
		    
			 (non-Javadoc)
	         * @see org.eclipse.jface.viewers.LabelProvider#getText(java.lang.Object)
	         
	        public String getText(Object element)
	        {
	            if(element instanceof NamedElementPropertySource)
	            {
	                IDatapoolEquivalenceClass eqClass = (IDatapoolEquivalenceClass)((NamedElementPropertySource)element).getNamedElement();
	                IDatapool dp = (IDatapool) eqClass.getDatapool();
	                if(dp.getEquivalenceClassIndexById(eqClass.getId()) == dp.getDefaultEquivalenceClassIndex())
	                    return eqClass.getName() + TestUIPlugin.getString("DatapoolEditor.(default)"); //$NON-NLS-1$
	                else
	                    return eqClass.getName();
	            }
	            return super.getText(element);
	        }
        });*/
		return control;
	}

	protected Control createVariableSection(Composite parent)
	{
		IAction[] variableActions = createVariableActions();
		variablesSection = new DatapoolChildrenTreeSection(this, variableActions, SWT.SINGLE){
			public Object[] getChildren(IDatapool datapool) {
				if(datapool == null)
					return new Object[0];
				
				// if the current tree items do not match the datapool object, recreate them.
				if(variablePropertySourceItems.size() != datapool.getVariableCount())
				{
				    variablePropertySourceItems.clear();
					IPropertySource[] variablePropSources = new IPropertySource[datapool.getVariableCount()];				
					for(int i = 0; i < variablePropSources.length; i++)
					{
						variablePropSources[i] = new DatapoolVariablePropertySource((org.eclipse.hyades.edit.datapool.IDatapoolVariable)datapool.getVariable(i));
						variablePropertySourceItems.add(variablePropSources[i]);
					}
				}
				return variablePropertySourceItems.toArray();
			}

			public Object[] getElements(IDatapool inputElement) {				
				return getChildren(inputElement);
			}

			public Object getParent(Object element) {
				//if(element instanceof IDatapoolVariable)
					//return ((IDatapoolVariable)element).getDatapool();
				return null;
			}
			public boolean hasChildren(IDatapool element) {
				if(element != null && element.getVariableCount() > 0)
					return true;
				return false;
			}
			/* (non-Javadoc)
			 * @see org.eclipse.hyades.test.ui.editor.form.util.DatapoolChildrenTreeSection#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {}
		};
		
		registerSection(variablesSection);		
		variablesSection.setHeaderText(TestUIPlugin.getString("DatapoolEditor.Variables")); //$NON-NLS-1$
		variablesSection.setDescription(TestUIPlugin.getString("DatapoolEditor.VariablesSectionDescription")); //$NON-NLS-1$
		variablesSection.setCollapsable(true);
		Control control = variablesSection.createControl(parent, getWidgetFactory());
		control.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));		
		return control;
	}
	
	private IAction[] createEquivalenceClassActions()
	{
		IAction[] actions = new IAction[5];
		IDatapool datapool = getDatapool();
		actions[0] = new AddEquivalenceClassAction(datapool, TestUIPlugin.getString("DatapoolEditor.Add")); //$NON-NLS-1$
		actions[1] = new RemoveEquivalenceClassAction(getDatapool(), TestUIPlugin.getString("DatapoolEditor.Remove")); //$NON-NLS-1$
		actions[2] = new MoveEquivalenceClassUp(getDatapool(), TestUIPlugin.getString("DatapoolEditor.Up")); //$NON-NLS-1$
		actions[3] = new MoveEquivalenceClassDown(getDatapool(), TestUIPlugin.getString("DatapoolEditor.Down")); //$NON-NLS-1$
		actions[4] = new OpenEquivalenceClassAction((DatapoolEditorExtension)getBaseEditorExtension(), getDatapool(), TestUIPlugin.getString("DatapoolEditor.Open")); //$NON-NLS-1$
		
		return actions;
	}

	private IAction[] createVariableActions()
	{
		IAction[] actions = new IAction[4];
		IDatapool datapool = getDatapool();
		actions[0] = new AddVariableAction(datapool, TestUIPlugin.getString("DatapoolEditor.Add")); //$NON-NLS-1$
		actions[1] = new RemoveVariableAction(datapool, TestUIPlugin.getString("DatapoolEditor.Remove")); //$NON-NLS-1$
		actions[2] = new MoveVariableUp(datapool, TestUIPlugin.getString("DatapoolEditor.Up")); //$NON-NLS-1$
		actions[3] = new MoveVariableDown(datapool, TestUIPlugin.getString("DatapoolEditor.Down")); //$NON-NLS-1$
		return actions;
	}
		
	protected IDatapoolListener getDatapoolListener()
	{
		if(this.datapoolListener == null)
			datapoolListener = new DatapoolListener();
		return this.datapoolListener;
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#load()
	 */
	public void load()
	{
		generalInfoSection.setInput(getDatapool());
		variablesSection.setInput(getDatapool());
		equivalenceClassSection.setInput(getDatapool());
		// default equivalence class is not in data model thus does not get persisted. 
		// Add this functionality when this is resolved.
		/*int defIndex = ((IDatapool)getDatapool()).getDefaultEquivalenceClassIndex();
		Tree tree = equivalenceClassSection.getTreeViewer().getTree();
		if(defIndex > -1 && tree.getItemCount() > defIndex)
		{
			TreeItem defItem = tree.getItems()[defIndex];
			defItem.setChecked(true);
		}*/
		getDatapool().addDatapoolListener(getDatapoolListener());
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#updateTitle()
	 */
	public void updateTitle()
	{
		String title = ((IWorkbenchAdapter)((IAdaptable)getDatapool()).getAdapter(IWorkbenchAdapter.class)).getLabel(getDatapool());
		if(title != null)
			getBaseEditorExtension().getHyadesEditorPart().setTitle(title);
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#activated()
	 */
	public boolean activated()
	{
		return true;
	}

    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
     */
    public void selectionChanged(SelectionChangedEvent event)
    {
        if(event.getSelectionProvider() == variablesSection.getTreeViewer())
            focusedViewer = variablesSection.getTreeViewer();
        else if(event.getSelectionProvider() == equivalenceClassSection.getTreeViewer())
            focusedViewer = equivalenceClassSection.getTreeViewer();
        
        Object[] listeners = selectionChangeListenerList.getListeners();
        for(int i = 0; i < listeners.length; i++)
        {
            ((ISelectionChangedListener)listeners[i]).selectionChanged(event);
        }
    }
	
    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ISelectionProvider#addSelectionChangedListener(org.eclipse.jface.viewers.ISelectionChangedListener)
     */
    public void addSelectionChangedListener(ISelectionChangedListener listener)
    {
        selectionChangeListenerList.add(listener);
    }
    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ISelectionProvider#getSelection()
     */
    public ISelection getSelection()
    {
        if(focusedViewer == null)
            focusedViewer = variablesSection.getTreeViewer();
        return focusedViewer.getSelection();
    }
    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ISelectionProvider#removeSelectionChangedListener(org.eclipse.jface.viewers.ISelectionChangedListener)
     */
    public void removeSelectionChangedListener(
            ISelectionChangedListener listener)
    {
        selectionChangeListenerList.remove(listener);
    }
    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ISelectionProvider#setSelection(org.eclipse.jface.viewers.ISelection)
     */
    public void setSelection(ISelection selection)
    {
        Object sel = ((IStructuredSelection)selection).getFirstElement();
        if(sel instanceof IDatapoolEquivalenceClass)
        {
            int index = getDatapool().getEquivalenceClassIndexById(((IDatapoolEquivalenceClass)sel).getId());
            if(index > -1 && index < equivalenceClassPropertySourceItems.size())
                equivalenceClassSection.getTreeViewer().setSelection(new StructuredSelection(equivalenceClassPropertySourceItems.get(index)));
        }
    }
    
}
