/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ResourceUtil.java,v 1.6 2005/03/02 23:41:34 bjiang Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.model;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.ui.IElementFactory;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IPersistableElement;
import org.eclipse.ui.views.properties.IPropertySource;

import org.eclipse.hyades.ui.util.IDisposable;

/**
 * @author marcelop
 * @since 0.2.0
 */
public class ResourceUtil
{
	public static class PersistableElement
	implements IPersistableElement, IDisposable
	{
		private Resource resource;
		
		/**
		 * Constructor for PersistableElement
		 * @param namedElement
		 */
		public PersistableElement(Resource resource)
		{
			this.resource = resource;
		}
		
		/**
		 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
		 */
		public void dispose()
		{
			resource = null;
		}

		/**
		 * @see org.eclipse.ui.IPersistableElement#getFactoryId()
		 */
		public String getFactoryId()
		{
			return ElementFactory.class.getName();
		}

		/**
		 * @see org.eclipse.ui.IPersistableElement#saveState(org.eclipse.ui.IMemento)
		 */
		public void saveState(IMemento memento)
		{
			ElementFactory.save(memento, resource);
		}
	}
	
	
	public static class ElementFactory
	implements IElementFactory
	{
		/*
		 * TAG* constants are used in the IMemento
		 */
		public static final String TAG_FILE = "FILE"; //$NON-NLS-1$

		/**
		 * Saves information of the logical folder in the memento. 
		 * @param memento
		 * @param logicalFolder
		 */
		public static void save(IMemento memento, Resource resource)
		{
			memento.putString(TAG_FILE, EMFUtil.getFilePath(resource));
		}	

		/**
		 * @see org.eclipse.ui.IElementFactory#createElement(org.eclipse.ui.IMemento)
		 */
		public IAdaptable createElement(IMemento memento)
		{
			String filePath = memento.getString(TAG_FILE);
			if(filePath == null)
				return null;

			IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(filePath);
			if((resource == null) || (resource.getType() != IResource.FILE))
				return null;
			
			EObject[] eObjects = EMFUtil.load(null, (IFile)resource);
			if(eObjects.length > 0)
				return new ResourceAdaptable(eObjects[0].eResource());
				
			return null;		
		}
	}

	public static class ResourceAdaptable
	implements IAdaptable, IDisposable
	{
		private Resource resource;
		
		/**
		 * Constructor for ResourceAdaptable
		 * @param resource
		 */
		public ResourceAdaptable(Resource resource)
		{
			this.resource = resource;			
		}
		
		/**
		 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
		 */
		public void dispose()
		{
			resource = null;
		}
		
		/**
		 * Returns the resource this adaptable is decorating.
		 * @return Resource
		 */
		public Resource getResource()
		{
			return resource;
		}
		
		/**
		 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
		 */
		public Object getAdapter(Class adapter)
		{
			if(Resource.class.isAssignableFrom(adapter))
				return getResource();
			
			if(IPersistableElement.class == adapter)
				return new PersistableElement(getResource());

			IFile file = EMFUtil.getWorkspaceFile(getResource()); 
			if(IResource.class.isAssignableFrom(adapter))
				return file;
				
			if(file != null)
			{
				if(IPropertySource.class == adapter)
					return file.getAdapter(IPropertySource.class);
			} 
				
			return null;
		}
	}
}
