/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PasteAction.java,v 1.4 2005/03/21 17:47:35 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.action;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.TestUISafeRunnable;
import org.eclipse.hyades.test.ui.navigator.actions.IPaster;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.actions.CopyProjectOperation;
import org.eclipse.ui.part.ResourceTransfer;

/**
 * @author jgout
 * @since 3.3
 */
public class PasteAction extends Action implements IDisposable {
	protected Clipboard clipboard;
	private IStructuredSelection selection;
	private Shell shell;
	
	public PasteAction(Shell shell, Clipboard clipboard) {
		super(TestUIPlugin.getString("command.Paste")); //$NON-NLS-1$
		if (shell == null)
			throw new IllegalArgumentException("Unable to perform paste action due to 'shell == null'"); //$NON-NLS-1$
		this.shell = shell;
		if (clipboard == null)
			throw new IllegalArgumentException("Unable to perform paste action due to 'clipboard == null'"); //$NON-NLS-1$
		this.clipboard = clipboard;
	}
	
	public void dispose() {
		if(clipboard != null) {
			clipboard.dispose();
		}
		clipboard = null;
		shell = null;
	}
	
	public void run() {
		final Object target = selection != null ? selection.getFirstElement() : null;
		if(target != null) {
			IPaster [] pasters = PasteExtensionManager.getInstance().getPasters(target);
			for (int i = 0; i < pasters.length; i++) {
				final IPaster paster = pasters[i];				
				class TestUISafeRunnableWithStatus extends TestUISafeRunnable {
					private boolean isOk;
					
					public void run() throws Exception {
						isOk = paster.isPasteAllowedFor(clipboard, target) &&
								paster.performPaste(clipboard, target);
					}
					
					public boolean isOk() {
						return isOk;
					}
				}
				TestUISafeRunnableWithStatus runnable = new TestUISafeRunnableWithStatus();
				Platform.run(runnable);
				if(runnable.isOk()) {
					return;
				}
			}
		}
		//- special case of pasting a project regardless the selection
		IResource [] res = getResourceFromClipboard();
		boolean isProjectRes = res != null && res.length > 0 && res[0].getType() == IResource.PROJECT;
		if (isProjectRes) {
			for (int i = 0; i < res.length; i++) {
				CopyProjectOperation operation = new CopyProjectOperation(this.shell);
				operation.copyProject((IProject) res[i]);
			}
		}
	}

	private IResource [] getResourceFromClipboard() {
		final IResource[][] clipboardData = new IResource[1][];
		shell.getDisplay().syncExec(new Runnable() {
			public void run() {
				//- clipboard must have resources or files
				ResourceTransfer resTransfer = ResourceTransfer.getInstance();
				clipboardData[0] = (IResource[]) clipboard.getContents(resTransfer);
			}
		});
		return clipboardData[0];
	}
	
	private boolean doesClipboardContainProjects() {
		//- if clipboard contains a project, paste is valid if project is open
		IResource [] res = getResourceFromClipboard();
		boolean isProjectRes = res != null && res.length > 0 && res[0].getType() == IResource.PROJECT;
		if (isProjectRes) {
			for (int i = 0; i < res.length; i++) {
				//- make sure that all resource data are open projects
				if (res[i].getType() != IResource.PROJECT || ((IProject) res[i]).isOpen() == false)
					return false;
			}
			return true;
		}
		return false;
	}
	
	public boolean isApplicableForSelection() {
		//- regardless the selection if clipboard contains a project, this is possible to paste it.
		if (doesClipboardContainProjects()) {
			return true;
		}
		//- multiple selection for pasting is not allowed
		if (selection == null || selection.isEmpty() || selection.size() > 1) {
			return false;
		}	
		//- selection contains only one element
		//- is there at least one client registered that is able to perform a paste action in this context ?
		if(PasteExtensionManager.getInstance().isPasteAllowedFor(clipboard, selection.getFirstElement())) {
			return true;
		}
		return false;
	}
	
	public void selectionChanged(IStructuredSelection structuredSelection) {
		selection = structuredSelection;
	}
}
