/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RenameAction.java,v 1.5 2005/05/10 13:26:38 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.action;

import java.lang.reflect.InvocationTargetException;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer;
import org.eclipse.hyades.ui.internal.navigator.TreeNavigator;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TreeEditor;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.actions.TextActionHandler;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.internal.ide.IDEWorkbenchMessages;
import org.eclipse.ui.internal.progress.ProgressMonitorJobsDialog;

/**
 * @author jgout
 * @since 3.3
 */
public class RenameAction extends Action implements IDisposable {
	private IStructuredSelection selection;
	protected Composite textEditorParent;
	private TreeEditor treeEditor;
	protected Tree navigatorTree;
	protected Text textEditor;
	private TextActionHandler textActionHandler;
	protected TreeNavigator navigator;
	protected IContainer renamedResource;

	public RenameAction(TreeNavigator treeNav) {
		//- reuse an internal platform message because of PII freeze 
		super(IDEWorkbenchMessages.getString("RenameResourceAction.text")); //$NON-NLS-1$
		this.navigator = treeNav;
		navigatorTree = treeNav.getTreeViewer().getTree();
		this.treeEditor = new TreeEditor(navigatorTree);
		renamedResource = null;
	}

	public void dispose() {
		selection = null;
		disposeTextWidget();
	}

	public boolean isApplicableForSelection() {
		if (selection == null || selection.isEmpty()) {
			return false;
		}
		Object objSelected = selection.getFirstElement();
		if (objSelected instanceof IContainer) {
			return true;
		} else if (objSelected instanceof IProxyNode) {
			IProxyNode proxy = (IProxyNode) objSelected;
			IProxyNodeRenamer renamer = (IProxyNodeRenamer) proxy.getAdapter(IProxyNodeRenamer.class);
			if (renamer != null) {
				return renamer.isApplicableFor();
			} else {
				//- this means that there is no renamer for this proxy
				return false;
			}
		} else {
			return false;
		}
	}

	/**
	 * On Mac the text widget already provides a border when it has focus, so there is no need to draw another one.
	 * The value of returned by this method is usd to control the inset we apply to the text field bound's in order to get space for drawing a border.
	 * A value of 1 means a one-pixel wide border around the text field. A negative value supresses the border.
	 * However, in M9 the system property "org.eclipse.swt.internal.carbon.noFocusRing" has been introduced
	 * as a temporary workaround for bug #28842. The existence of the property turns the native focus ring off
	 * if the widget is contained in a main window (not dialog).
	 * The check for the property should be removed after a final fix for #28842 has been provided.
	 * This code has been imported from RenameResourceAction
	 */
	private static int getCellEditorInset(Control c) {
		if ("carbon".equals(SWT.getPlatform())) { //$NON-NLS-1$
			//- special case for MacOS X 
			if (System.getProperty("org.eclipse.swt.internal.carbon.noFocusRing") == null || c.getShell().getParent() != null) //$NON-NLS-1$
				//- native border
				return -2;
		}
		//- one pixel wide black border
		return 1;
	}

	/**
	 * This code has been imported from RenameResourceAction	 * 
	 * @return composite
	 */
	private Composite createParent() {
		Composite result = new Composite(navigatorTree, SWT.NONE);
		TreeItem[] selectedItems = navigatorTree.getSelection();
		treeEditor.horizontalAlignment = SWT.LEFT;
		treeEditor.grabHorizontal = true;
		treeEditor.setEditor(result, selectedItems[0]);
		return result;
	}

	/**
	 * Create the text editor widget.
	 * This code has been imported from RenameResourceAction
	 * @param renamer
	 * 
	 */
	private void createTextEditor(final IProxyNodeRenamer renamer) {
		//- Create text editor parent.  This draws a nice bounding rect.
		textEditorParent = createParent();
		textEditorParent.setVisible(false);
		final int inset = getCellEditorInset(textEditorParent);
		if (inset > 0) // only register for paint events if we have a border
			textEditorParent.addListener(SWT.Paint, new Listener() {
				public void handleEvent(Event e) {
					Point textSize = textEditor.getSize();
					Point parentSize = textEditorParent.getSize();
					e.gc.drawRectangle(0, 0, Math.min(textSize.x + 4, parentSize.x - 1), parentSize.y - 1);
				}
			});
		//- Create inner text editor.
		textEditor = new Text(textEditorParent, SWT.NONE);
		textEditor.setFont(navigatorTree.getFont());
		textEditorParent.setBackground(textEditor.getBackground());
		textEditor.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event e) {
				Point textSize = textEditor.computeSize(SWT.DEFAULT, SWT.DEFAULT);
				//- Add extra space for new characters.
				textSize.x += textSize.y;
				Point parentSize = textEditorParent.getSize();
				textEditor.setBounds(2, inset, Math.min(textSize.x, parentSize.x - 4), parentSize.y - 2 * inset);
				textEditorParent.redraw();
			}
		});
		textEditor.addListener(SWT.Traverse, new Listener() {
			public void handleEvent(Event event) {
				switch (event.detail) {
					case SWT.TRAVERSE_ESCAPE :
						//- Do nothing in this case
						disposeTextWidget();
						event.doit = true;
						event.detail = SWT.TRAVERSE_NONE;
						break;
					case SWT.TRAVERSE_RETURN :
						performRename(renamer, textEditor.getText());
						disposeTextWidget();
						event.doit = true;
						event.detail = SWT.TRAVERSE_NONE;
						break;
				}
			}
		});
		textEditor.addFocusListener(new FocusAdapter() {
			public void focusLost(FocusEvent fe) {
				disposeTextWidget();
			}
		});
		if (textActionHandler != null)
			textActionHandler.addText(textEditor);
	}

	protected void performRename(final IProxyNodeRenamer renamer, final String newName) {
		Runnable query = new Runnable() {
			public void run() {
				if(renamer == null) {
					//- this neams that a folder or a project node was renamed
					moveResource(renamedResource, newName);
				} else {
					boolean shouldRefresh = renamer.performRename(newName);
					if(shouldRefresh) {
						navigator.getTreeViewer().refresh();
					}
				}
				if (navigatorTree != null && !navigatorTree.isDisposed()) {
				    navigatorTree.setFocus();
				    //- find the new item to select it as it was before renaming
				    TreeItem[] items = navigatorTree.getItems();
				    findItem(newName, items);
				}
			}

			private void moveResource(final IResource res, String name) {
				final IPath newPath = res.getFullPath().removeLastSegments(1).append(name);
				WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
					public void execute(IProgressMonitor monitor) {
						try {
							res.move(newPath, IResource.KEEP_HISTORY | IResource.SHALLOW, new SubProgressMonitor(monitor, 50));
						} catch (CoreException e) {
							e.printStackTrace();
						}
					}
				};
				try {
					new ProgressMonitorJobsDialog(Display.getCurrent().getActiveShell()).run(true, false, op);
				} catch (InvocationTargetException e) {
					e.printStackTrace();
				} catch (InterruptedException e) {
					//- can't be canceled
				}
			}
				
			private boolean findItem(final String name, TreeItem[] items) {
				for(int i = 0; i < items.length; i++) {
					if(items[i].getText().equals(name)) {
					    navigator.selectReveal(new StructuredSelection(items[i].getData()));
					    return true;
					}
					if(items[i].getItemCount() > 0) {
						if(findItem(name, items[i].getItems())) {
							return true;
						}
					}
				}
				return false;
			}
		};
		navigatorTree.getShell().getDisplay().asyncExec(query);
	}

	/**
	 * Close the text widget and reset the editorText field.
	 * This code has been imported from RenameResourceAction
	 */
	protected void disposeTextWidget() {
		if (textActionHandler != null)
			textActionHandler.removeText(textEditor);
		if (textEditorParent != null) {
			textEditorParent.dispose();
			textEditorParent = null;
			textEditor = null;
			treeEditor.setEditor(null, null);
		}
	}

	/**
	 * This code is inspered from RenameResourceAction
	 * @param oldName current value of the selected item
	 * @param renamer
	 */
	private void getNewNameInline(String oldName, IProxyNodeRenamer renamer) {
		//- Make sure text editor is created only once. 
		if (textEditorParent == null) {
			createTextEditor(renamer);
		}
		textEditor.setText(oldName);
		//- Open text editor with initial size.
		textEditorParent.setVisible(true);
		Point textSize = textEditor.computeSize(SWT.DEFAULT, SWT.DEFAULT);
		//- Add extra space for new characters.
		textSize.x += textSize.y;
		Point parentSize = textEditorParent.getSize();
		int inset = getCellEditorInset(textEditorParent);
		textEditor.setBounds(2, inset, Math.min(textSize.x, parentSize.x - 4), parentSize.y - 2 * inset);
		textEditorParent.redraw();
		textEditor.selectAll();
		textEditor.setFocus();
	}

	public void run() {
		Object objSelected = selection.getFirstElement();
		if (objSelected instanceof IContainer) {
			renamedResource = (IContainer)objSelected;
			//- reuse the inline editor to rename a project or folder node
			getNewNameInline(((IContainer)objSelected).getName(), null);
		} else if (objSelected instanceof IProxyNode) {
			IProxyNode proxy = (IProxyNode) objSelected;
			IProxyNodeRenamer renamer = (IProxyNodeRenamer) proxy.getAdapter(IProxyNodeRenamer.class);
			if (renamer != null) {
				RenamerUIStatus status = renamer.performUserInteraction(proxy.getText());
				switch (status.getStatus()) {
					case RenamerUIStatus.OK :
						performRename(renamer, status.getNewName());
						break;
					case RenamerUIStatus.CANCEL :
						//- nothing to do 
						break;
					case RenamerUIStatus.INLINE_EDITOR :
						getNewNameInline(proxy.getText(), renamer);
						break;
				}
			}
		}
	}

	public void selectionChanged(IStructuredSelection structuredSelection) {
		selection = structuredSelection;
	}
}