/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EMFResourceProxyFactory.java,v 1.7 2005/04/20 15:23:35 dguilbaud Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import org.eclipse.core.resources.IFile;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.navigator.FileProxyNode;
import org.eclipse.hyades.test.ui.navigator.HyadesProxyNodeFactory;
import org.eclipse.hyades.test.ui.navigator.IFileProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IPersistableFileProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.ui.IMemento;

/** Generic Proxy Factory for EMF files.
 * It opens the model stored inside the file. 
 * If the model contents only one object, the factory won't create a file level node.
 * An exception is made for execution file which always show the file level even if the file content only one TPFExecutionResult. 
 * Delegation is made for typed object.
 *  
 * @author jgout
 * @since 3.2
 */
public class EMFResourceProxyFactory implements IFileProxyFactory, IPersistableFileProxyFactory {
		
	/** This method is called to build a node that contains multiple EMF objects as children.
	 *  The object created depends of the extension of the file (.datapool -> DatapoolFileProxyNode)
	 *  using as name the name of the file since if there are multiple objects inside, we can't
	 *  know which one should be used to name the node.
	 * @param file
	 * @return file level proxy node
	 */
	private FileProxyNode createFileProxy(IFile file, Resource resource) {
		FileProxyNode fileProxy = null;
		String extension = file.getFileExtension();
		String fileName = file.getName();
		
		//- remove the extension from the file name 
		fileName = fileName.substring(0, fileName.length() - extension.length() - 1);
		if(extension != null) {
			if(extension.equals(ICommonConstants.ARTIFACT_FILE_EXTENSION)) {
				fileProxy = new ArtifactFileProxyNode(file, resource);
			} else if(extension.equals(ICommonConstants.DATAPOOL_FILE_EXTENSION)) {
				fileProxy = new DatapoolFileProxyNode(file, resource);
			} else if(extension.equals(ICommonConstants.LOCATION_FILE_EXTENSION)) {
				fileProxy = new LocationFileProxyNode(file, resource);
			} else if(extension.equals(ICommonConstants.DEPLOYMENT_FILE_EXTENSION)) {
				fileProxy = new DeploymentFileProxyNode(file, resource);
			} else if(extension.equals(ICommonConstants.TEST_SUITE_FILE_EXTENSION)) {
				fileProxy = new TestSuiteFileProxyNode(file, resource);
			}  else if(extension.equals(ICommonConstants.TEST_COMPONENT_EXTENSION)) {
				fileProxy = new TestComponentFileProxyNode(file, resource);
			}  else if(extension.equals(ICommonConstants.EXECUTION_FILE_EXTENSION)) {
				fileProxy = new ExecutionFileProxyNode(file, resource);
			} else {
				fileProxy = null;
			}
		} else {
			fileProxy = null;
		}
		return (fileProxy != null && fileProxy.getChildren().length > 0) ? fileProxy : null;
	}

	private boolean isLoaded(IFile file) {
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString());
		ResourceSet rs = EMFUtil.getResourceSet();
		return rs.getResource(uri, false) != null;
	}
	
	//- this method load the resource associated to the given file using the test navigator resource set
	private Resource getEMFResource(IFile file) {
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString());
		ResourceSet rs = EMFUtil.getResourceSet();
		return rs.getResource(uri, true);
	}
	
	/** Create a proxy node (which can be a full tree of proxies) from the given file.
	 *  The file is supposed to be a EMF file.
	 *  This method loads the content of the file and unload the resource at the end.
	 * @param file the file to convert
	 * @retrun the root of the proxy node tree
	 */
	public IProxyNode create(IFile file) {
		IProxyNode res = null;
		boolean wasAlreadyLoaded = isLoaded(file);
		Resource resource = getEMFResource(file);
		EObject[] eObjects;
		if(resource != null) {
			eObjects = (EObject[]) resource.getContents().toArray(new EObject[resource.getContents().size()]); 
			if(eObjects.length == 0) {
				//- nothing in this file, just unload
			} else if (eObjects.length == 1) {
				//- only one root object, so create proxy for this object 
				res = HyadesProxyNodeFactory.getInstance().create(eObjects[0], file.getParent());
			} else {
				//- there are multiple root objects, create a file proxy level and a content level
				//- create the root proxy (file node)
				res = createFileProxy(file, resource);
			}
			if (!wasAlreadyLoaded) {
				resource.unload();
			}
		}
		return res;
	}
	
	/** Create a proxy for the given file using the memento. This memento is the proxy state stored previously.
	 * 
	 * @param file the file to convert
	 * @param memento the state of the proxy previously saved by IPersistableProxyNode.saveState().
	 * @retrun the root of the proxy node tree
	 * @see org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode
	 */
	public IProxyNode recreate(IFile file, IMemento memento) {
		//- try to retrieve one of the default hyades proxy nodes
		IProxyNode proxy = HyadesProxyNodeFactory.getInstance().recreate(memento, file.getParent());
		//- 
		if(proxy == null) {
			String nodeKind = memento.getString(TestUIConstants.TAG_NODE_KIND);
			if(nodeKind.equals(TestUIConstants.ARTIFACT_FILE_NODE)) {
				return new ArtifactFileProxyNode(file, memento.getString(TestUIConstants.TAG_NAME), memento.getChildren(TestUIConstants.TAG_CHILD));
			} else if(nodeKind.equals(TestUIConstants.DATAPOOL_FILE_NODE)) {
				return new DatapoolFileProxyNode(file, memento.getString(TestUIConstants.TAG_NAME), memento.getChildren(TestUIConstants.TAG_CHILD));
			} else if(nodeKind.equals(TestUIConstants.DEPLOY_FILE_NODE)) {
				return new DeploymentFileProxyNode(file, memento.getString(TestUIConstants.TAG_NAME), memento.getChildren(TestUIConstants.TAG_CHILD));
			} else if(nodeKind.equals(TestUIConstants.EXECUTION_FILE_NODE)) {
				return new ExecutionFileProxyNode(file, memento.getString(TestUIConstants.TAG_NAME), memento.getChildren(TestUIConstants.TAG_CHILD));
			} else if(nodeKind.equals(TestUIConstants.LOCATION_FILE_NODE)) {
				return new LocationFileProxyNode(file, memento.getString(TestUIConstants.TAG_NAME), memento.getChildren(TestUIConstants.TAG_CHILD));
			} else if(nodeKind.equals(TestUIConstants.TESTCOMPONENT_FILE_NODE)) {
				return new TestComponentFileProxyNode(file, memento.getString(TestUIConstants.TAG_NAME), memento.getChildren(TestUIConstants.TAG_CHILD));
			} else if(nodeKind.equals(TestUIConstants.TESTSUITE_FILE_NODE)) {
				return new TestSuiteFileProxyNode(file, memento.getString(TestUIConstants.TAG_NAME), memento.getChildren(TestUIConstants.TAG_CHILD));
			}
		} else {
			return proxy;
		}
		return null;
	}
}
