/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileFactoryManager.java,v 1.4 2005/02/16 22:22:11 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.navigator.IFileProxyFactory;

/**
 * @author jgout
 * @since 3.2
 */
public class FileFactoryManager {
	
	private static FileFactoryManager instance;
	private Map factories;
	private Map files;
	
	public static FileFactoryManager getInstance() {
		if(instance == null) {
			instance = new FileFactoryManager();
		}
		return instance;
	}
	
	private FileFactoryManager() {
		factories = new HashMap();
		files = new HashMap();
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".testNavigatorFileProxyFactory"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) {
				IConfigurationElement element = members[i];
				if ("factory".equals(element.getName())) { //$NON-NLS-1$
					String id = element.getAttribute("id"); //$NON-NLS-1$
					addFactory(id, element);
				} else if("file".equals(element.getName())) { //$NON-NLS-1$
					//- get extension of files we want to make visible
					String extension = element.getAttribute("extension"); //$NON-NLS-1$
					String factoryID = element.getAttribute("factoryID"); //$NON-NLS-1$
					if(extension != null && extension.length() != 0) {
						addFile(extension, factoryID);
					}
				}
			}
		}
		//- for backward compatibility of extension point
		extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".testNavigatorFileFactory"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) {
				IConfigurationElement element = members[i];
				if ("file".equals(element.getName())) { //$NON-NLS-1$
					//- get extension of files we want to make visible
					String extension = element.getAttribute("extension"); //$NON-NLS-1$
					if(extension != null && extension.length() != 0) {
						//- build the factoryID for the name of the factory class given in the extension
						String id = element.getAttribute("factory"); //$NON-NLS-1$
						addFactory(id, element);
						addFile(extension, id);
					}
				}
			}
		}		

	}
	
	private void addFactory(String id, IConfigurationElement element) {
		if(!factories.containsKey(id)) {
			factories.put(id, element);
		} else {
			TestUIPlugin.logInfo("ID: "+id+" already used in a previously registered factory"); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}
	
	private void addFile(String extension, String id) {
		if(!files.containsKey(extension)) {
			//- if the given extension is not yet in the internal map, add it
			ArrayList l = new ArrayList();
			l.add(id);
			files.put(extension, l);
		} else {
			//- extension already registered
			ArrayList facID = (ArrayList)files.get(extension);
			if(!facID.contains(id)) {
				facID.add(id);
			}
		}
	}
	
	/** Returns the list of factories registered to the given file extension
	 *  Caller should manage the choice of the factory to use.
	 *  FileProxyManager is the main caller of this method.
	 *  The rule is that if a factory is not the right one for a file it should return null.
	 *  That way, caller can iterate until it finds a not null result of the create method.
	 * 
	 * @param extension the file name extension 
	 * @return the list of factories, this list can be empty if extension is null or empty
	 */
	public ArrayList getFactories(String extension) {
		if(extension == null || extension.length() == 0) return new ArrayList();
		if(files.containsKey(extension)) {
			LinkedList removed = new LinkedList();
			ArrayList l = (ArrayList)files.get(extension);
			//- check if all factories have been loaded
			for (int i = 0; i < l.size(); i++) {
				Object value = l.get(i);
				if(value instanceof String) {
					//- the element in the list is not the factory itself but its id
					IFileProxyFactory factory = getFactory((String)value);
					if(factory != null) {
						l.set(i, factory);
					} else {
						//- bad or unknown id inserted in the file list, remove this id
						removed.add(new Integer(i));
					}
				} else if(!(value instanceof IFileProxyFactory)) {
					//- strange situation: unknown class in the factory list ?!
					TestUIPlugin.logError("List of factories for file: "+extension+" contains something else than a factory"); //$NON-NLS-1$ //$NON-NLS-2$
					removed.add(new Integer(i));
				}
			}
			//- if bad elements have been found during factories analysis, we should to remove them
			for (Iterator it = removed.iterator(); it.hasNext();) {
				l.remove(((Integer) it.next()).intValue());
			}
			//- return factories
			return l;
		} else {
			return new ArrayList();
		}
	}

	/** Returns the file proxy factory registered with the given id or <code>null</code> if such a factory is not found. 
	 * 
	 * @param id the id of the serached factory. 
	 * @return the factory which has the given id or <code>null</code> if it an unknown id.
	 */
	public IFileProxyFactory getFactory(String id) {
		if(factories.containsKey(id)) {
			Object value = factories.get(id);
			if (value instanceof IConfigurationElement) {
				//- if the factory has not yet been loaded
				IConfigurationElement element = (IConfigurationElement) value;
				IFileProxyFactory factory; //$NON-NLS-1$
				try {
					if(element.getName().equals("file")) { //$NON-NLS-1$
						//- old extension point
						factory = (IFileProxyFactory)element.createExecutableExtension("factory"); //$NON-NLS-1$
					} else {
						//- new one
						factory = (IFileProxyFactory)element.createExecutableExtension("class"); //$NON-NLS-1$
					}
					//- replace the value in the map by the factory it self
					factories.put(id, factory);
					return factory;
				} catch (CoreException e) {
					//- remove this element since we can not create a factory from it
					factories.remove(id);
					TestUIPlugin.logError("Enable to create the factory instance from configuration element");  //$NON-NLS-1$//$NON-NLS-2$
					return null;
				}
			} else if(value instanceof IFileProxyFactory) {
				//- the factory has been already loaded
				return (IFileProxyFactory)value; 
			} else {
				//- the value is not a factory ?!@!#$! (should never occur)
				TestUIPlugin.logError("Factory Id: "+id+ " associated to something else than a factory");  //$NON-NLS-1$//$NON-NLS-2$
				return null;
			}
		} else {
			//- unknown factory ID
			return null;
		}
	}
	
	/** Return whether the given file extension has been registered 
	 * @param fileExtension
	 * @return true if the given file extension has been registered yet and false otherwise
	 */
	public boolean isRegistered(String fileExtension) {
		return files.containsKey(fileExtension);
	}
}
