/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TypedElementFactoryManager.java,v 1.5 2005/04/19 16:05:53 dguilbaud Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.proxy;

import java.util.HashMap;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.navigator.IPersistableTypedElementProxyFactory;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITypedElementProxyFactory;
import org.eclipse.ui.IMemento;

/**
 * @author jgout
 * @since 3.2
 */
public class TypedElementFactoryManager {
	
	private static TypedElementFactoryManager instance;
	private HashMap factories;
	private HashMap types;
	
	/**
	 * Factory to used to create proxy from unknown type element.
	 * Methods use the default public hyades factory
	 * @author jgout
	 */
	class DefaultTypedElementFactory implements IPersistableTypedElementProxyFactory {
		
		public IProxyNode recreate(IMemento memento, Object parent) {
			String nodeKind = memento.getString(TestUIConstants.TAG_NODE_KIND);
			if(nodeKind.equals(TestUIConstants.TESTSUITE_NODE)) {
				return new DefaultTestSuiteProxyNode(memento, parent);
			} else if(nodeKind.equals(TestUIConstants.TESTCASE_NODE)) {
				return new DefaultTestCaseProxyNode(memento, parent);
			} else if(nodeKind.equals(TestUIConstants.EXECUTION_RESULT_NODE)) {
				return new DefaultExecutionResultProxyNode(memento, parent);
			} else if(nodeKind.equals(TestUIConstants.TESTCOMPONENT_NODE)) {
				return new DefaultTestComponentlProxyNode(memento, parent);
			}
			return null;
		}
		
		public IProxyNode create(TPFExecutionResult result, Object parent) {
			return new DefaultExecutionResultProxyNode(result, parent);
		}
		
		public IProxyNode create(TPFTest test, Object parent) {
			if(test instanceof TPFTestSuite) {
				return new DefaultTestSuiteProxyNode((TPFTestSuite)test, parent);
			} else if(test instanceof TPFTestCase) {
				return new DefaultTestCaseProxyNode((TPFTestCase)test, parent);
			} else return null;
		}
		
		public IProxyNode create(TPFTestComponent component, Object parent) {
			return new DefaultTestComponentlProxyNode(component, parent);
		}
	}
	
	public static TypedElementFactoryManager getInstance() {
		if(instance == null) {
			instance = new TypedElementFactoryManager();
		}
		return instance;
	}
	
	private TypedElementFactoryManager() {
		factories = new HashMap();
		types = new HashMap();
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".testNavigatorTypedElementProxyFactory"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) {
				IConfigurationElement element = members[i];
				if ("factory".equals(element.getName())) { //$NON-NLS-1$
					String id = element.getAttribute("id"); //$NON-NLS-1$
					addFactory(id, element);
				} else if("typedElement".equals(element.getName())) { //$NON-NLS-1$
					//- get extension of files we want to make visible
					String type = element.getAttribute("type"); //$NON-NLS-1$
					String factoryID = element.getAttribute("factoryID"); //$NON-NLS-1$
					if(type != null && type.length() != 0) {
						addType(type, factoryID);
					}
				}
			}
		}
		//- backward compatibility .. this following statements should be removed in the future version
		extPoint = Platform.getExtensionRegistry().getExtensionPoint(TestUIPlugin.getID() + ".testNavigatorTypedElementFactory"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] members = extPoint.getConfigurationElements();
			for (int i = 0; i < members.length; i++) {
				IConfigurationElement element = members[i];
				if ("typedElement".equals(element.getName())) { //$NON-NLS-1$
					//- get extension of files we want to make visible
					String typeName = element.getAttribute("type"); //$NON-NLS-1$
					if(typeName != null && typeName.length() != 0) {
						//- build the factoryID for the name of the factory class given in the extension
						String id = element.getAttribute("factory"); //$NON-NLS-1$
						addFactory(id, element);
						addType(typeName, id);
					}
				}
			}
		}
	}
	
	private void addFactory(String id, IConfigurationElement element) {
		if(!factories.containsKey(id)) {
			//- if the given type is not yet in the internal map, add it
			factories.put(id, element);
		} else {
			TestUIPlugin.logInfo("ID: "+id+" already used in a previously registered factory"); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}
	
	private void addType(String type, String factoryID) {
		if(!types.containsKey(type)) {
			types.put(type, factoryID);
		} else {
			TestUIPlugin.logInfo("type: "+type+" already registered"); //$NON-NLS-1$ //$NON-NLS-2$			
		}
	}
	
	public ITypedElementProxyFactory getFactory(String typeName) {
		if(types.containsKey(typeName)) {
			return getFactoryFromID((String)types.get(typeName));
		} else {
				return null;
		}
	}

	/** Returns the factory associated to the given id. 
	 * The factory should have been registered using the following extension point:  <code>org.eclipse.hyades.test.ui.testNavigatorTypedElementProxyFactory</code> 
	 * @param factoryID the id of a registered factory.
	 * @return the typed element proxy factory which has been registered with the given id.
	 */
	public ITypedElementProxyFactory getFactoryFromID(String factoryID) {
		Object value = factories.get(factoryID);
		if (value instanceof IConfigurationElement) {
			//- if the factory has not yet been loaded
			IConfigurationElement element = (IConfigurationElement) value;
			try {
				ITypedElementProxyFactory factory = null;
				if("factory".equals(element.getName())) { //$NON-NLS-1$
					factory = (ITypedElementProxyFactory)element.createExecutableExtension("class"); //$NON-NLS-1$
				} else {
					//- old extension point
					factory = (ITypedElementProxyFactory)element.createExecutableExtension("factory"); //$NON-NLS-1$
				}
				return factory;
			} catch (CoreException e) {
				TestUIPlugin.logError(e);
				//- problem building the instance of factory using reflection
				return new DefaultTypedElementFactory();
			} 
		} else {
			//- the factory has been loaded yet
			if (value instanceof ITypedElementProxyFactory) {
				return (ITypedElementProxyFactory) value;
			} else {
				TestUIPlugin.logError("internal error"); //$NON-NLS-1$
				//- problem with stored factory (this should never appear)
				return null;
			}
		}
	}

}
