/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AgentControllerPage.java,v 1.7 2005/02/16 22:21:33 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.security.preference;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

import org.eclipse.hyades.execution.security.KeystoreHelper;
import org.eclipse.hyades.security.SecurityImages;
import org.eclipse.hyades.security.SecurityPlugin;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.security.util.CertificateProperties;
import org.eclipse.hyades.security.util.SecurityUI;
import org.eclipse.hyades.security.util.TString;

public class AgentControllerPage
	extends PreferencePage
	implements IWorkbenchPreferencePage, SelectionListener {

	private final static String[] COLUMNS = { "STRING", "STRING" };

	private TableViewer tableViewer;
	private ArrayList tableElements = new ArrayList();
	private Button btnAdd;
	private Button btnRemove;
	private Button btnRename;
	private Button btnProperties;
	private KeyStore ks;
	
	private class MyTableElement {
		private String alias;
		private Object value;

		public MyTableElement(String alias,String value)
		{
			this.alias = alias;
			this.value = value;
		}
		public String getAlias() {
			return alias;
		}
		public void setAlias(String text) {
			this.alias = text;
		}
		public Object getValue() {
			return value;
		}
		public String getType(){
			if(value instanceof Certificate){
				return SecurityPlugin.getResourceString("STR_TRUSTED_CERTIFICATE");
			}
			else if(value instanceof Key){
				return SecurityPlugin.getResourceString("STR_KEY_ENTRY");
			}
			return SecurityPlugin.getResourceString("STR_UNKNOWN_ENTRY");
		}
		
		public void setValue(Object value) {
			this.value = value;
		}
	}
	private class PreferenceContentProvider implements IStructuredContentProvider {
		/**
		 * @see IStructuredContentProvider#getElements(Object)
		 */
		public Object[] getElements(Object element) {
			if (element instanceof ArrayList)
				return ((ArrayList)element).toArray();

			return new Object[0];
		}
		/**
		 * @see IContentProvider#dispose()
		 */
		public void dispose() {
		}
		/**
		 * @see IContentProvider#inputChanged(Viewer, Object, Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
	}
	private class PreferenceLabelProvider
		extends LabelProvider
		implements ITableLabelProvider {
		/**
		 * @see ITableLabelProvider#getColumnImage(Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}
		/**
		 * @see ITableLabelProvider#getColumnText(Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			if (element instanceof MyTableElement) {
				MyTableElement myTableElement = (MyTableElement) element;

				if (columnIndex == 0)
					return myTableElement.getAlias();

				if (columnIndex == 1)
					return myTableElement.getType();
			}
			return "";
		}
	}
	private class PreferenceCellModifier implements ICellModifier {
		private Viewer viewer;

		public PreferenceCellModifier(Viewer viewer) {
			this.viewer = viewer;
		}
		/**
		 * @see ICellModifier#canModify(Object, String)
		 */
		public boolean canModify(Object element, String property) {
			if (element instanceof MyTableElement) {
				MyTableElement myTableElement = (MyTableElement) element;
				if (property.equals(COLUMNS[0]))
					return true;

				if (property.equals(COLUMNS[1]))
					return false;
			}
			return false;
		}
		/**
		 * @see ICellModifier#getValue(Object, String)
		 */
		public Object getValue(Object element, String property) {
			if (element instanceof MyTableElement) {
				MyTableElement myTableElement = (MyTableElement) element;

				if (property.equals(COLUMNS[0]))
					return myTableElement.getAlias();

				if (property.equals(COLUMNS[1]))
					return getValueIndex(myTableElement.getType());
			}
			return null;
		}
		/**
		 * @see ICellModifier#modify(Object, String, Object)
		 */
		public void modify(Object element, String property, Object value) {
			if (element instanceof TableItem) {
				Object myElement = ((TableItem) element).getData();

				if (myElement instanceof MyTableElement) {
					MyTableElement myTableElement = (MyTableElement) myElement;

					if (property.equals(COLUMNS[0]))
						myTableElement.setAlias((String) value);

					viewer.refresh();
				}
			}
		}
		private Integer getValueIndex(String value) {
			return new Integer(0);
		}
		private String getValue(Integer index) {
			return null;
		}
	}

	
	public class AddDialog extends Dialog implements Listener{

        private SecurityUI _securityUI;
        private Certificate fCertificate;
        private Shell shell;
		
		public AddDialog(Shell shell){
			super(shell);
			this.shell = shell;
		}
		
		protected Control createDialogArea(Composite parent){ 

			getShell().setText(SecurityPlugin.getResourceString("STR_ADD_CERT_DLG_TITLE"));
			Composite content = new Composite(parent, SWT.NULL);
			GridLayout layout = new GridLayout();
			GridData data = GridUtil.createFill();
			layout.numColumns = 1;
			content.setLayout(layout);
			content.setLayoutData(data);

			_securityUI = new SecurityUI(shell);
			_securityUI.createContents(content);
			_securityUI.registerListener(this);
			return content;
		}
		
		protected Control createButtonBar(Composite parent) {
			Control control = super.createButtonBar(parent);
			//shell.setDefaultButton(getButton(IDialogConstants.OK_ID));
			getButton(IDialogConstants.OK_ID).setEnabled(false);			
			return control;
			
		}
		
		protected void okPressed(){

			try{	
				fCertificate = _securityUI.loadCertificate(ks);
			}
			catch(FileNotFoundException e){

				String text = SecurityPlugin.getResourceString("STR_KEY_IO_ERROR_");
				text = TString.change(text, "%1", _securityUI.getPath());
				String msg = SecurityPlugin.getResourceString("STR_CERTIFICATE_LOAD_EXC_");
				
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,e);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				return;
				
			}
			catch(IOException e){

				String text = SecurityPlugin.getResourceString("STR_LOAD_IO_EXC_");
				text = TString.change(text, "%1", _securityUI.getPath());

				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("STR_CERTIFICATE_LOAD_EXC_");
				
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,e);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				return;
											
			}
			catch(CertificateException exc){

				String text = SecurityPlugin.getResourceString("STR_CERTIFICATE_EXC_");
				text = TString.change(text, "%1", _securityUI.getPath());

				String msg = SecurityPlugin.getResourceString("STR_CERTIFICATE_LOAD_EXC_");
				
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				return;

			}
			catch(KeyStoreException exc){
				String text = SecurityPlugin.getResourceString("STR_KEY_STORE_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("STR_CERTIFICATE_LOAD_EXC_");
				
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				return;

			}

			MyTableElement elem = new MyTableElement(_securityUI.getAliasName(), SecurityPlugin.getResourceString("STR_TRUSTED_CERTIFICATE"));
			elem.setValue(fCertificate);
			tableElements.add(elem);
						
			super.okPressed();
		}
		
		public void handleEvent(Event e){
			getButton(IDialogConstants.OK_ID).setEnabled(_securityUI.validateDialog());
		}
		
		
		public Certificate getCertificate(){
			return fCertificate;
		}
		
	}

	public class RenameDialog extends Dialog implements Listener{
		
		private Text txtName;
		private String newAlias;
		private String oldAlias;
		public RenameDialog(Shell shell, String oldValue){
			super(shell);
			oldAlias = oldValue;
		}
		
		protected Control createDialogArea(Composite parent){

			getShell().setText(SecurityPlugin.getResourceString("STR_RENAME_CERT_DLG_TITLE"));
			Composite content = new Composite(parent, SWT.NONE);
			GridLayout layout = new GridLayout();
			GridData data = GridUtil.createFill();
			data.widthHint = 350;
			layout.numColumns = 2;
			content.setLayout(layout);
			content.setLayoutData(data);

			Label lblName = new Label(content, SWT.NONE);
			lblName.setText(SecurityPlugin.getResourceString("STR_CERTIFICATE_ALIAS"));
			
			txtName = new Text(content, SWT.BORDER);
			txtName.setText(oldAlias);
			txtName.selectAll();
			txtName.addListener(SWT.Modify, this);
			
			data = GridUtil.createHorizontalFill();
			data.widthHint = 200;
			txtName.setLayoutData(data);

			return content;
		}
		
		private void validateDialog(){
			if(txtName.getText().trim().length()==0)
				getButton(IDialogConstants.OK_ID).setEnabled(false);
			else
				getButton(IDialogConstants.OK_ID).setEnabled(true);
		}
		
		public void handleEvent(Event e){
			if(e.widget.equals(txtName))
				validateDialog();
		}
		
		protected Control createButtonBar(Composite parent) {
			Control control = super.createButtonBar(parent);			
			validateDialog();
			return control;
			
		}

		public String getNewAlias(){
			return newAlias;
		}
		
		protected void okPressed(){
			newAlias = txtName.getText();
			super.okPressed();
		}

	}
		
	public AgentControllerPage() {
		super();
		setPreferenceStore(SecurityPlugin.getDefault().getPreferenceStore());
	}
	
	protected Control createContents(Composite parent) {
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayoutData(GridUtil.createFill());
		composite.setLayout(new GridLayout());

		TabFolder folder = new TabFolder(composite, SWT.NONE);
		folder.setLayout(new GridLayout());
		folder.setLayoutData(GridUtil.createFill());
		
		TabItem item;

		item = new TabItem(folder, SWT.NONE);
		item.setText(SecurityPlugin.getResourceString("STR_PREF_PD_AC_SECURITY"));
		item.setImage(SecurityImages.getImage(SecurityImages.IMG_CERTIF_FILE));
		item.setControl(createSecurityTab(folder));
		
		return composite;
	}
	
	private Control createSecurityTab(Composite parent)
	{		
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginWidth = 5;
		layout.verticalSpacing = 10;
		layout.numColumns = 2;
		composite.setLayout(layout);
		composite.setLayoutData(GridUtil.createFill());
		
		Label label = new Label(composite, SWT.NULL);
		label.setText(SecurityPlugin.getResourceString("STR_PREF_SEC_DESCRIPTION"));
		GridData data = new GridData();
		data.horizontalSpan = 2;
		label.setLayoutData(data);
				
		createTableViewer(composite);

		Composite buttons = new Composite(composite, SWT.NONE);
		data = new GridData(GridData.FILL_VERTICAL);
		buttons.setLayoutData(data);
		layout = new GridLayout();
		layout.numColumns = 1;
		buttons.setLayout(layout);
		
		createButtons(buttons);
		initializeValues();
		
		btnAdd.addSelectionListener(this);
		btnRename.addSelectionListener(this);
		btnRemove.addSelectionListener(this);
		btnProperties.addSelectionListener(this);
		
		return composite;
	}
	
	private void createTableViewer(Composite parent) {
		// Create the table viewer.
		tableViewer =
			new TableViewer(
				parent,
				SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION );

		// Create the table control.
		Table table = tableViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		GridData data = GridUtil.createFill();
		data.heightHint = 50;
		data.widthHint = 100;
		table.setLayoutData(data);

		TableLayout tableLayout = new TableLayout();
		CellEditor[] cellEditors = new CellEditor[COLUMNS.length];

		TableColumn aliasColumn = new TableColumn(table, SWT.LEFT);
		aliasColumn.setText(SecurityPlugin.getResourceString("STR_PREF_ALIAS_NAME"));
		tableLayout.addColumnData(new ColumnPixelData(100));
		cellEditors[0] = new TextCellEditor(table);

		TableColumn certificateColumn = new TableColumn(table, SWT.LEFT);
		certificateColumn.setText(SecurityPlugin.getResourceString("STR_PREF_CERTIFICATES"));
		tableLayout.addColumnData(new ColumnPixelData(120));

		table.setLayout(tableLayout);
		

		// Adjust the table viewer.
		tableViewer.setColumnProperties(COLUMNS);
		tableViewer.setContentProvider(new PreferenceContentProvider());
		tableViewer.setLabelProvider(new PreferenceLabelProvider());
		tableViewer.getTable().addSelectionListener(this);

		TableViewerSorter.setTableSorter(tableViewer, 0, true);
	}
	
	private void createButtons(Composite parent){
		
		btnAdd = new Button(parent, SWT.PUSH);
		btnAdd.setText(SecurityPlugin.getResourceString("STR_ADD_LBL"));
		GridData data = GridUtil.createHorizontalFill();
		//data.horizontalIndent = 6;
		btnAdd.setLayoutData(data);
		
		btnRemove = new Button(parent, SWT.PUSH);
		btnRemove.setText(SecurityPlugin.getResourceString("STR_REMOVE_LBL"));
		data = GridUtil.createHorizontalFill();
		//data.horizontalIndent = 6;		
		btnRemove.setLayoutData(data);
		btnRemove.setEnabled(false);		

		Composite space = new Composite(parent, SWT.NONE);
		data = GridUtil.createHorizontalFill();
		data.verticalSpan = 2;
		//data.horizontalIndent = 6;
		space.setLayoutData(data);

		btnRename = new Button(parent, SWT.PUSH);
		btnRename.setText(SecurityPlugin.getResourceString("STR_RENAME_LBL"));
		data = GridUtil.createHorizontalFill();
		//data.horizontalIndent = 6;
		btnRename.setLayoutData(data);
		btnRename.setEnabled(false);
		
		btnProperties = new Button(parent, SWT.PUSH);
		btnProperties.setText(SecurityPlugin.getResourceString("STR_PROPERTIES_LBL"));
		data = GridUtil.createHorizontalFill();
		//data.horizontalIndent = 6;
		btnProperties.setLayoutData(data);
		btnProperties.setEnabled(false);		
	}


	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}
	public void widgetSelected(SelectionEvent event) {
		
		if(event.widget == btnAdd){
			AddDialog dlg = new AddDialog(getShell());
			dlg.open();
			
			if(dlg.getReturnCode() == Window.OK)
			{				
				tableViewer.refresh();			
			}
			
		}else if(event.widget == btnRemove){
			IStructuredSelection elem = (IStructuredSelection)tableViewer.getSelection();
			if(elem.size() > 0){
				Iterator i = elem.iterator();
				while(i.hasNext()){
					try{	
						MyTableElement current = (MyTableElement)i.next();
						ks.deleteEntry(current.getAlias());
						tableElements.remove(current);
					}
					catch(KeyStoreException e){
					}
				}
				
				tableViewer.refresh();				
			}
		
		}
		else if(event.widget == btnRename){
			IStructuredSelection elem = (IStructuredSelection)tableViewer.getSelection();
			if(elem.size() == 1){
				MyTableElement sel = (MyTableElement)elem.getFirstElement();
				RenameDialog dlg = new RenameDialog(getShell(), sel.getAlias());
				dlg.open();
														
				if(dlg.getReturnCode() == Window.OK)
				{
					try{	
						KeystoreHelper.addCertificateToKeyStore(AgentControllerPage.this.ks,(Certificate)sel.getValue(),dlg.getNewAlias());
						ks.deleteEntry(sel.getAlias());
						sel.setAlias(dlg.getNewAlias());
						tableViewer.refresh();							
					}
					catch(KeyStoreException e){
					}						
				}
			}
			
		}else if(event.widget == btnProperties){
			IStructuredSelection elem = (IStructuredSelection)tableViewer.getSelection();
			if(elem.size() == 1){
				MyTableElement sel = (MyTableElement)elem.getFirstElement();
				CertificateProperties dlg = new CertificateProperties(getShell());
				dlg.init(sel.getAlias(), sel.getValue());
				
				dlg.open();
														
			}
			
		} 

		boolean sel = tableViewer.getSelection().isEmpty();
		btnRename.setEnabled(!sel);
		btnRemove.setEnabled(!sel);
		btnProperties.setEnabled(!sel);

	}

	/**
	 * @see PreferencePage#computeSize()
	 */
	public Point computeSize() {
		Point p = super.computeSize();

		return p;
	}
	/**
	 * @see IWorkbenchPreferencePage#init(IWorkbench)
	 */
	public void init(IWorkbench workbench) {
	}
	/**
	 * @see PreferencePage#performDefaults()
	 */
	protected void performDefaults() {
		super.performDefaults();

	}
	/**
	 * @see PreferencePage#performOk()
	 */
	public boolean performOk() {

		String storePath = SecurityPlugin.getKeyStoreLocation();
		String passw = SecurityPlugin.getWorkspaceName();
		try {
							        
			KeystoreHelper.persistKeyStore(ks, storePath, passw);
			
		}
		catch(IOException e){

			String text = SecurityPlugin.getResourceString("STR_IO_SAVE_ERROR_");
			text = TString.change(text, "%1", storePath);

			text = TString.change(text, "%1", storePath);
			String msg = SecurityPlugin.getResourceString("STR_KEYSTORE_SAVE_ERROR_");
			
			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,e);
			ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			return false;
										
		}
		catch(CertificateException exc){

			String text = SecurityPlugin.getResourceString("STR_CERTIFICATE_STORE_ERROR_");
			text = TString.change(text, "%1", storePath);
			String msg = SecurityPlugin.getResourceString("STR_KEYSTORE_SAVE_ERROR_");
			
			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc);
			ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			return false;

		}
		catch(KeyStoreException exc){
			String text = SecurityPlugin.getResourceString("STR_UNINIT_KEYSTORE_ERROR_");
			text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
			String msg = SecurityPlugin.getResourceString("STR_KEYSTORE_SAVE_ERROR_");
			
			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc);
			ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			return false;

		}
		catch(NoSuchAlgorithmException exc2)
		{
			String text = SecurityPlugin.getResourceString("STR_ALGORITHM_ERROR_");
			text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
			String msg = SecurityPlugin.getResourceString("STR_KEYSTORE_SAVE_ERROR_");
			
			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc2);
			ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			return false;

		}
		return true;

	}

	/**
	 * Loads certificates from the key store.
	 */
	private void initializeValues() {
		
		
		String storePath = SecurityPlugin.getKeyStoreLocation();
		String passw = SecurityPlugin.getWorkspaceName();

		try {
			ks = KeystoreHelper.createKeyStore(storePath, passw);
			
			Enumeration aliases = ks.aliases();
			
			
			while(aliases.hasMoreElements())
			{
				String alias = (String)(aliases.nextElement());
				/* The alias may be either a key or a certificate */
				java.security.cert.Certificate cert = ks.getCertificate(alias);
				if(cert != null) {
					MyTableElement elem = new MyTableElement(alias, SecurityPlugin.getResourceString("STR_TRUSTED_CERTIFICATE"));
					elem.setValue(cert);
					tableElements.add(elem);
				}
				else {
					try {

						Key key=ks.getKey(alias, passw.toCharArray());
						MyTableElement elem = new MyTableElement(alias, SecurityPlugin.getResourceString("STR_KEY_ENTRY"));
						elem.setValue(key);
						tableElements.add(elem);						
					}
					catch(UnrecoverableKeyException e) {
						/* Probably ignore the key in this case */	
					}
				}
					
					
			}			
		}
		catch(IOException e){

			String text = SecurityPlugin.getResourceString("STR_LOAD_IO_EXC_");
			text = TString.change(text, "%1", storePath);
			String msg = SecurityPlugin.getResourceString("STR_LOAD_KEYSTORE_ERROR_");
			
			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,e);
			ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
										
		}
		catch(CertificateException exc){

			String text = SecurityPlugin.getResourceString("STR_KEY_LOAD_ERROR_");
			text = TString.change(text, "%1", storePath);
			String msg = SecurityPlugin.getResourceString("STR_LOAD_KEYSTORE_ERROR_");
			
			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc);
			ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);

		}
		catch(KeyStoreException exc){

			String text = SecurityPlugin.getResourceString("STR_INITIALIZE_ERROR_");
			text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
			String msg = SecurityPlugin.getResourceString("STR_LOAD_KEYSTORE_ERROR_");

			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc);
			ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);

		}
		catch(NoSuchProviderException exc2){

			String text = SecurityPlugin.getResourceString("STR_SECURITY_PROVIDER_ERROR_");
			String msg = SecurityPlugin.getResourceString("STR_INITIALIZE_ERROR_");
			msg = TString.change(msg, "%1", SecurityPlugin.getKeyStoreLocation());
			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc2);
			ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);

		
		}
		catch(NoSuchAlgorithmException exc2)
		{
			String text = SecurityPlugin.getResourceString("STR_ALGORITHM_ERROR_");
			text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
			String msg = SecurityPlugin.getResourceString("STR_LOAD_KEYSTORE_ERROR_");
			
			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc2);
			ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);

		}
		
		tableViewer.setInput(tableElements);
	}
		
}
