/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ConnectUtil.java,v 1.15 2005/04/12 22:00:49 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.security.util;

import java.io.*;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.*;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.text.DateFormat;
import java.util.Enumeration;

import javax.net.ssl.TrustManager;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;

import org.eclipse.hyades.execution.security.*;
import org.eclipse.hyades.internal.execution.local.control.*;
import org.eclipse.hyades.internal.execution.security.*;
import org.eclipse.hyades.security.SecurityPlugin;


public class ConnectUtil
{
  public static final int CONNECTION_SUCCESS = 0;
  public static final int CONNECTION_CONTROLLER_ERROR = 1;
  public static final int CONNECTION_HOST_ERROR = 2;
  public static final int CONNECTION_PORT_ERROR = 3;
  public static final int CONNECTION_SECURITY_NOT_SUPPORTED = 4;
  
  private Application _app;
  private String _port;
  private String _hostName;
  private InetAddress _hostAddr;  
  private String _userId;
  private Node   _node;
  private ISecureClientParameters _parameters;
  
  private ConnectErrorMsg _connectErrorMessage;
  
  	private class ConnectErrorMsg
	{
  		private String _errorMessage;
  		private Status _status;  		
  		
  		public ConnectErrorMsg(String errorMessage, Status status)
  		{
  			_errorMessage = errorMessage;
  			_status = status;
  		}
  		
  		public String getErrorMessage()
  		{
  			return _errorMessage;
  		}
  		
  		public Status getErrorStatus()
  		{
  			return _status;
  		}
	}
  
    public ConnectUtil(String hostName, String port, Application app)
    {
    	this(hostName, port, null, app);
    }

    public ConnectUtil(String hostName, String port
    							, String userId
    							, Application app)
    {
    	_hostName = hostName;
    	_port     = port;
    	_userId   = userId;
    	_app      = app;
    	
    	_connectErrorMessage = null;
    }

    public ConnectUtil(InetAddress hostAddr, String port, Application app)
    {
    	this(hostAddr, port, null, app);
    }

    public ConnectUtil(InetAddress hostAddr, String port
    							, String userId
    							, Application app)
    {
    	_hostAddr = hostAddr;
    	_hostName = _hostAddr.getHostName();
    	_port     = port;
    	_userId   = userId;
    	_app      = app;
    }
    
    private ISecureClientParameters getClientParms() {
    	
		IConfigurationElement[] config =
			Platform.getExtensionRegistry().getConfigurationElementsFor(SecurityPlugin.getPluginId(), "JSSESecurityProviders");

		for (int idx = 0; idx < config.length; idx++) {
			IConfigurationElement elem = config[idx];
	
			String typeAttr = elem.getAttribute("name");
			String classAttr = elem.getAttribute("class");
			
			/* For each of our security providers, instantiate the instances */	
			if(classAttr!=null) {
				try {
					Object realization=elem.createExecutableExtension("class");
					if(realization instanceof ISecureClientParameters) {
						ISecureClientParameters parms=(ISecureClientParameters)realization;
						/* Add the provider from the realization */
						Security.addProvider(parms.getSecurityProvider());
						parms.getKeystoreManager().setProvider(parms.getSecurityProvider());
						
						
						return (ISecureClientParameters)realization;	
					}
				}
				catch(CoreException e) {
				}
			}
		}
		return null;
    }
    
    
    public void showErrorDialogForConnect()
    {
    	if (_connectErrorMessage != null)
    	{
    		openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"),_connectErrorMessage.getErrorMessage(),_connectErrorMessage.getErrorStatus());
    	}
    }
    
    public final int connect()
    {
    	return connect(true);
    }

	public final int connect(boolean showErrorMsgs)
	{
		_connectErrorMessage = null;
		try {
			if(_hostAddr == null)
				_hostAddr = InetAddress.getByName(_hostName);
		}
		catch(UnknownHostException exc)
		{
			resetConnection();
						
			String text = SecurityPlugin.getResourceString("TIMEOUT_NODE_ERROR_");
			text = TString.change(text, "%1", _hostName);
			String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
			msg = TString.change(msg, "%1", _hostName);
			
			Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,exc);
			
			_connectErrorMessage = new ConnectErrorMsg(msg, err);			
			
			//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"),msg,err);
			if (showErrorMsgs) showErrorDialogForConnect();			
					       
			return CONNECTION_HOST_ERROR;			
		}
		
		try {
			
			/* Determine if we have aconnection with this node/user/app existing */
			if(checkConnectionExists()) {
				return CONNECTION_SUCCESS;
			}
			
			
			if(_node == null)
			{
				_node = NodeFactory.createNode(_hostAddr, _app);
			}
			
			_node.connect(Integer.parseInt(_port));
		}
		catch(UnknownHostException exc)
		{
			resetConnection();
						
			String text = SecurityPlugin.getResourceString("TIMEOUT_NODE_ERROR_");
			text = TString.change(text, "%1", _hostName);
			String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
			msg = TString.change(msg, "%1", _hostName);
			
			Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,exc);
			
			_connectErrorMessage = new ConnectErrorMsg(msg, err);
			
			//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"),msg,err);		
			if (showErrorMsgs) showErrorDialogForConnect();
			
			return CONNECTION_HOST_ERROR;
		}
		catch(SecureConnectionRequiredException exc) {  
			
			return connectUnderException(exc, showErrorMsgs);
		}
		catch (LoginFailedException exc) {
			return connectUnderException(exc, showErrorMsgs);	
		}
		
		catch(UntrustedAgentControllerException exc) {
			TrustManager[] tManagers = _node.getSecurityParameters().getKeystoreManager().getTrustManagers();
			X509Certificate cert = null;
			try{
				for (int i=0; i<tManagers.length; i++){
					Method getReceivedCerts = tManagers[i].getClass().getMethod("getReceivedCerts", null);
					Object certs = getReceivedCerts.invoke(tManagers[i], null);
					if (certs != null){
						cert = (X509Certificate) ((Object[]) certs)[0];
					}
				}
			}catch(Exception e){
				System.out.println(e.toString());
			}

			/* RKD:  First we open a dialog stating that the server is not trusted and ask the user to continue */
			String text = SecurityPlugin.getResourceString("STR_CERTIF_UNAV_ERROR_");    
			text = TString.change(text, "%1", _hostName);
			
			
			String txt = TString.change(SecurityPlugin.getResourceString("STR_CERTIFICATE_ALERT"), "%1", _hostName);
			int rc = 0;
			boolean ok = false;
			do {
				String[] btnLabels = new String[]{SecurityPlugin.getResourceString("STR_YES"), SecurityPlugin.getResourceString("STR_NO"), SecurityPlugin.getResourceString("STR_VIEW_DETAIL")};
				rc = openQuestionDialog(SecurityPlugin.getResourceString("SEC_MSG"), txt, btnLabels);
				//import certificate
				if (rc == 0){
					ok = openSaveCertificateDialog(cert);
					if (!ok)
						ok = openQuestionDialog(SecurityPlugin.getResourceString("STR_CERTIFICATE_PROP_TITLE"), SecurityPlugin.getResourceString("STR_UNTRUSTED_AGENT_CONTROLLER"), null) == 0;
				} 
				//chose not to import certificate
				else if (rc == 1){
					ok = openQuestionDialog(SecurityPlugin.getResourceString("STR_CERTIFICATE_PROP_TITLE"), SecurityPlugin.getResourceString("STR_UNTRUSTED_AGENT_CONTROLLER"), null) == 0;
				}
				//view certificate details
				else {
					openCertificateDetailDialog(cert);
				}
			} while (rc == 2);
			if(!ok) {
				
				resetConnection();
				
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,exc);
				
				_connectErrorMessage = new ConnectErrorMsg(msg, err);
				
				//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				if (showErrorMsgs) showErrorDialogForConnect();
				
				return CONNECTION_CONTROLLER_ERROR;
			}
			else {
				ISecureClientParameters params=_node.getSecurityParameters();
				if(params!=null) {
					params.disableServerAuthentication();
				}
				return connect(showErrorMsgs);
			
			}
			
		}
		catch(AgentControllerUnavailableException exc)
		{
			String text = SecurityPlugin.getResourceString("STR_AGENT_CONTROLLER_UNAV_ERROR_");
			    
			text = TString.change(text, "%1", _hostName);
						
			resetConnection();
			
			String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
			msg = TString.change(msg, "%1", _hostName);
			
			Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,exc);
			
			_connectErrorMessage = new ConnectErrorMsg(msg, err);
			
			//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			if (showErrorMsgs) showErrorDialogForConnect();
			
			return CONNECTION_CONTROLLER_ERROR;
		}		
		catch(NumberFormatException exc)
		{
			resetConnection();
			
			String text = SecurityPlugin.getResourceString("INVALID_PORT_ERROR_");
			text = TString.change(text, "%1", String.valueOf(_port));
			String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
			msg = TString.change(msg, "%1", _hostName);
			
			Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,exc);
			
			_connectErrorMessage = new ConnectErrorMsg(msg, err);
			//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			if (showErrorMsgs) showErrorDialogForConnect();
			
			return CONNECTION_PORT_ERROR;
		}
				
		return CONNECTION_SUCCESS;
	}
	
	public String getUserId()
	{
		return _userId;
	}
	
	public Node getNode()
	{
		return _node;
	}
	
	private void resetConnection()
	{
	    if(_userId != null) {
	       UserFactory.removeUser(_app, _userId);
	    }
	    if(_node!=null) {
	    	_node.setSecurityParameters(null);
	    }
		
	}
	
	private ConnectivityDialog dlg; 
	private int connectUnderException(AgentControllerUnavailableException exc, boolean showErrorMsgs) {
		try
		 {			 	
			 	User user = null;
			 	if(_userId != null && !_userId.equals("")) {
			 	 	user = UserFactory.getUser(_app, _userId);
			 	}
			 	 	
			 	if(user == null)
			 	{
			 		/* RKD: This dialog should be only put up if the server is using password authentication
			 		 */
			 		 if(exc instanceof SecureConnectionRequiredException && ((SecureConnectionRequiredException)exc).isPasswordProtected()) {
				 		 	
				 		/* RKD: This dialog needs to state the security requirements of the server */
			 		 	Display.getDefault().syncExec(new Runnable() {
							public void run() {
					 		 	dlg = new ConnectivityDialog(getValidShell(), _hostName, _userId, (_userId == null || _userId.equals("")));
						 		dlg.setDescription(SecurityPlugin.getResourceString("STR_PWD_REQ_INFO_"));
						 		dlg.create();
						 		openDialog(dlg);
							}
			 		 	});
				 		
				 		if(dlg.getReturnCode() != Window.OK) {
				 			
							resetConnection();
										
							String text = SecurityPlugin.getResourceString("STR_PWD_REQ_INFO_");
							String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
							msg = TString.change(msg, "%1", _hostName);
							
							Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,null);
							
							_connectErrorMessage = new ConnectErrorMsg(msg, err);
							
							//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"),msg,err);		
							if (showErrorMsgs) showErrorDialogForConnect();
							
				 		   return CONNECTION_HOST_ERROR;
				 		}
				 		   		
				 		_userId = dlg.getUserId();
				 		 
				 		if(UserFactory.getUser(_app, _userId) != null)
				 		{//remove the user first
				 			UserFactory.removeUser(_app, _userId);
				 		}
			 		
				 		user=UserFactory.createUser( _app, _userId, dlg.getPassword());	
			 		 }
			 		 else {
			 		 	/* RKD:  I don't think this is required anymore */
			 		 	_userId="ignoredUserId";
			 		 	user=UserFactory.createUser( _app, _userId, "dummyPass");
			 		 }	 		
			 	}  
			 	else if(exc instanceof LoginFailedException) {
			 		
			 		/* This dialog should state that the previous login failed */
			 		Display.getDefault().syncExec(new Runnable() {
						public void run() {
					 		dlg = new ConnectivityDialog(getValidShell(), _hostName, _userId);
					 		dlg.setDescription(SecurityPlugin.getResourceString("STR_LOGIN_FAILED_INFO_"));
					 		dlg.create();
						 	openDialog(dlg);
						}
			 		});
				 		
			 		if(dlg.getReturnCode() != Window.OK) {
			 			resetConnection();
									
						String text = SecurityPlugin.getResourceString("STR_PWD_REQ_INFO_");
						String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
						msg = TString.change(msg, "%1", _hostName);
						
						Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,null);
						_connectErrorMessage = new ConnectErrorMsg(msg, err);
						
						//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"),msg,err);		
						if (showErrorMsgs) showErrorDialogForConnect();
						
			 		   return CONNECTION_HOST_ERROR;
			 		}
			 		   	
			 		   		
			 		_userId = dlg.getUserId();
			 		
					user=UserFactory.getUser(_app, _userId);
			 		if(user != null)
			 		{//remove the user first
			 			user.setPassword(dlg.getPassword());
			 		}
			 		else {
		 		
			 			try {
			 				user=UserFactory.createUser( _app, _userId, dlg.getPassword());	
			 			}
			 			catch(DuplicateUserException e) {
						 }
			 		}			 		
		 		} 	 	
		 		
		 		if(exc instanceof SecureConnectionRequiredException) {
			 	 	_port=new Long(((SecureConnectionRequiredException)exc).getSecurePort()).toString();
			 	 
			 	}
			 	else if(exc instanceof LoginFailedException) {
			 		_port=new Long(((LoginFailedException)exc).getSecurePort()).toString();
			 	}
			 	
			 	if(_node.getSecurityParameters()==null) {
					ISecureClientParameters parms=getClientParms();
			 		if(parms==null) {
						resetConnection();
			 		
						String text = SecurityPlugin.getResourceString("STR_CLIENT_CANNOT_USE_SECURITY_ERROR_");
						text = TString.change(text, "%1", _hostName);
						String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
						msg = TString.change(msg, "%1", _hostName);
	
						Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text, null);
						_connectErrorMessage = new ConnectErrorMsg(msg, err);
						
						//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg, err);
						if (showErrorMsgs) showErrorDialogForConnect();
						
						return CONNECTION_SECURITY_NOT_SUPPORTED;
			 			
			 		}
			 		else {
						/* Load the keystore */ 
					 	_node.setSecurityParameters(getClientParms());
					 	_node.getSecurityParameters().getKeystoreManager().loadKeystore(_node.getSecurityParameters().getKeystoreFileName(), _node.getSecurityParameters().getKeystoreFilepassword());			 			
			 		}
			 		
			 	}

		 		
			 	/* Update the user info on this node */
			 	_node.setUser(user);
			 	
			 	return connect(showErrorMsgs);
			 	 
			 	
			 }
			 catch(DuplicateUserException e)
			 {
						 				 	
			 	resetConnection();
			 }		 				 	 
			 catch(IOException e)
			 {
				
				try {
					 				
				 	if(_node.getSecurityParameters()==null) {
				 		resetConnection();
			 		
			 			String text = SecurityPlugin.getResourceString("STR_CLIENT_CANNOT_USE_SECURITY_ERROR_");
						text = TString.change(text, "%1", _hostName);
						String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
						msg = TString.change(msg, "%1", _hostName);
						
						Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text, e);
						_connectErrorMessage = new ConnectErrorMsg(msg, err);
						
						//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg, err);
						if (showErrorMsgs) showErrorDialogForConnect();
						
				 		return CONNECTION_SECURITY_NOT_SUPPORTED;
				 	}
					else {
						/* Create an empty keystore  keystore */
						KeystoreHelper.createKeyStore(SecurityPlugin.getKeyStoreLocation(), SecurityPlugin.getWorkspaceName());
						/* Load the keystore */ 
						 _node.setSecurityParameters(getClientParms());
						 _node.getSecurityParameters().getKeystoreManager().loadKeystore(_node.getSecurityParameters().getKeystoreFileName(), _node.getSecurityParameters().getKeystoreFilepassword());		
	
	
					}
				 	
				 	
				 	// Update the user info on this node 
				 	_node.setUser(UserFactory.getUser(_app, _userId));
					
					return connect(showErrorMsgs);
					
				}
				catch(Exception e1) {
					
			 		resetConnection();
			 		
					String text = SecurityPlugin.getResourceString("STR_KEY_NOT_FOUND_ERROR_");
					text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
					String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
					msg = TString.change(msg, "%1", _hostName);
					
					Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
					_connectErrorMessage = new ConnectErrorMsg(msg, err);
					
					//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
					if (showErrorMsgs) showErrorDialogForConnect();
					
					return CONNECTION_HOST_ERROR;
				}
			 }
			 
			 catch(CertificateException e)
			 {
				resetConnection();
			 	
				String text = SecurityPlugin.getResourceString("STR_KEY_LOAD_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
				_connectErrorMessage = new ConnectErrorMsg(msg, err);
				
				//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				if (showErrorMsgs) showErrorDialogForConnect();
				
				return CONNECTION_HOST_ERROR;
			 }	
			 catch(KeyManagementException e)
			 {
				resetConnection();
			 	
				String text = SecurityPlugin.getResourceString("STR_KEY_MANAG_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
				_connectErrorMessage = new ConnectErrorMsg(msg, err);
				
				//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				if (showErrorMsgs) showErrorDialogForConnect();
				
				return CONNECTION_HOST_ERROR;
			 }			
			 catch(KeyStoreException e)
			 {
				resetConnection();
			 	
				String text = SecurityPlugin.getResourceString("STR_KEY_STORE_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
				_connectErrorMessage = new ConnectErrorMsg(msg, err);
				
				//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				if (showErrorMsgs) showErrorDialogForConnect();
				
				return CONNECTION_HOST_ERROR;
			 }
			 catch(NoSuchAlgorithmException e)
			 {
				resetConnection();
			 	
				String text = SecurityPlugin.getResourceString("STR_ALGORITHM_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
				_connectErrorMessage = new ConnectErrorMsg(msg, err);
				
				//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				if (showErrorMsgs) showErrorDialogForConnect();
				
				return CONNECTION_HOST_ERROR;							
			 }			
			 catch(UnrecoverableKeyException e)
			 {
				resetConnection();
			 	
				String text = SecurityPlugin.getResourceString("STR_UNREC_KEY_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
				_connectErrorMessage = new ConnectErrorMsg(msg, err);
				
				//openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				if (showErrorMsgs) showErrorDialogForConnect();
				
				return CONNECTION_HOST_ERROR;				
			 }	
			 
			 
			 return CONNECTION_SUCCESS;
			 					 		
	}
	
	private boolean checkConnectionExists() {
	
		Node node=NodeFactory.getNode(_hostAddr, _app);
		if(node!=null) {
			if(node.isConnected()) {
				_node=node;
				return true;
			}	
		}
		
		return false;
	}

	private void openErrorDialog(final String title, final String msg, final Status err)
	{
		Display.getDefault().syncExec(new Runnable()
		{
			public void run()
			{
				ErrorDialog.openError(getValidShell(),title, msg,err);
			}
		});
	}
	
	private void openDialog(final Dialog dialog)
	{
		dialog.getShell().getDisplay().syncExec(new Runnable()
		{
			public void run()
			{
				dialog.open();
			}
		});
	}

	private static Shell getValidShell()
	{
		Shell shell = null;
		IWorkbench workbench = PlatformUI.getWorkbench();
		if(workbench != null)
		{
			if(workbench.getActiveWorkbenchWindow() != null)
			{
				shell = workbench.getActiveWorkbenchWindow().getShell();
				if((shell != null) && (!shell.isDisposed()))
					return shell;
			}
			
			if(workbench.getWorkbenchWindows().length > 0)
			{
				shell = workbench.getWorkbenchWindows()[0].getShell();
				if((shell != null) && (!shell.isDisposed()))
					return shell;
			}
		}
		
		return null;
	}		
		
	private int btnPressed = 0;
	private int openQuestionDialog(final String title, final String msg, final String[] btnLabels)
	{		
		Display.getDefault().syncExec(new Runnable()
		{
			public void run()
			{
				if (btnLabels != null){
					MessageDialog dialog = new MessageDialog(getValidShell(), title, null, msg, MessageDialog.QUESTION, btnLabels, 0);
					btnPressed = dialog.open();
				} else
					btnPressed = MessageDialog.openQuestion(getValidShell(), title, msg) == true ? 0:1;
			}
		});
		
		return btnPressed;
	}
	
	private void openCertificateDetailDialog(final X509Certificate certificate){
		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				CertificateDetails dlg = new CertificateDetails(getValidShell());
				dlg.init(certificate);
				dlg.open();
			}
		});
	}

	private boolean okPressed = false;
	private boolean openSaveCertificateDialog(final X509Certificate cert) {
		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				SaveCertificate dlg = new SaveCertificate(getValidShell());
				dlg.init(cert);
				okPressed = dlg.open()==0;
			}
		});
		return okPressed;
	}

	private String toHexString(byte[] data){
		StringBuffer hexString = new StringBuffer();
		int i = 0;
		for (; i < data.length-1; i++) {
			hexString.append(
				Integer.toHexString(0xFF & data[i] | 0x100).substring(1));
			if ((i+1)%16 == 0)
				hexString.append("\n");
			else if ((i+1)%8 == 0)
				hexString.append("    ");
			else
				hexString.append(" ");
		}
		for (; i < data.length; i++){
			hexString.append(
				Integer.toHexString(0xFF & data[i] | 0x100).substring(1));
		}
		return hexString.toString().toUpperCase();
	}

	private String getThumbprintFromCertificate(X509Certificate certificate){
		String alg = certificate.getSigAlgName();
		try {
			if (alg.toLowerCase().indexOf("sha") > -1)
				alg = "SHA";
			else
				alg = "MD5";
			MessageDigest md = MessageDigest.getInstance(alg);
			md.reset();
			byte[] encCertInfo = certificate.getEncoded();
			byte[] digest = md.digest(encCertInfo);
			return toHexString(digest);
		} catch (Exception e) {
		}
		return "";
	}
	
	class CertificateDetails extends Dialog {
		private X509Certificate cert;
		
		public CertificateDetails(Shell shell){
			super(shell);
		}
		
		public void init(X509Certificate cer){
			this.cert = cer;
		}
		
		protected Control createDialogArea(Composite parent){ 
			getShell().setText(SecurityPlugin.getResourceString("STR_CERTIFICATE_PROP_TITLE"));
			Composite content = new Composite(parent, SWT.NONE);
			GridLayout layout = new GridLayout();
			GridData data = GridUtil.createFill();
			data.widthHint = 450;
			//data.heightHint = 300;
			layout.numColumns = 2;
			content.setLayout(layout);
			content.setLayoutData(data);
			
			// this should not happen
			if (this.cert == null){
				Label errorIcon = new Label(content, SWT.NONE);
				errorIcon.setImage(Display.getCurrent().getSystemImage(SWT.ICON_ERROR));
				data = new GridData();
				data.verticalAlignment = GridData.VERTICAL_ALIGN_BEGINNING;
				data.horizontalIndent = 5;
				errorIcon.setLayoutData(data);

				Label lblErrMsg = new Label(content, SWT.NONE);
				lblErrMsg.setText(SecurityPlugin.getResourceString("No certificate to show"));
				data = new GridData();
				data.verticalAlignment = GridData.VERTICAL_ALIGN_END;
				data.horizontalIndent = 5;
				lblErrMsg.setLayoutData(data);

				return content;
			}
			Label lblVersion = new Label(content, SWT.NONE);
			lblVersion.setText(SecurityPlugin.getResourceString("STR_CERTIF_VERSION_LBL"));
			data = new GridData();
			data.horizontalIndent = 5;		
			lblVersion.setLayoutData(data);

			Label lblVersionValue = new Label(content, SWT.NONE);

			data = new GridData(GridData.FILL_HORIZONTAL);
			data.horizontalIndent = 5;		
			lblVersionValue.setLayoutData(data);
					
			Label lblSubject = new Label(content, SWT.NONE);
			lblSubject.setText(SecurityPlugin.getResourceString("STR_SUBJECT_LBL"));
			data = new GridData();
			data.horizontalIndent = 5;		
			lblSubject.setLayoutData(data);

			Label lblSubjectValue = new Label(content, SWT.NONE);

			data = new GridData(GridData.FILL_HORIZONTAL);
			data.horizontalIndent = 5;		
			lblSubjectValue.setLayoutData(data);

			Label lblSigAlg = new Label(content, SWT.NONE);
			lblSigAlg.setText(SecurityPlugin.getResourceString("STR_SIGALG_LBL"));
			data = new GridData();
			data.horizontalIndent = 5;		
			lblSigAlg.setLayoutData(data);

			Label lblSigAlgValue = new Label(content, SWT.NONE);

			data = new GridData(GridData.FILL_HORIZONTAL);
			data.horizontalIndent = 5;		
			lblSigAlgValue.setLayoutData(data);

			Label lblKey = new Label(content, SWT.NONE);
			lblKey.setText(SecurityPlugin.getResourceString("STR_KEY_LBL"));
			data = new GridData();
			data.horizontalIndent = 5;		
			lblKey.setLayoutData(data);

			Label lblKeyValue = new Label(content, SWT.NONE);

			data = new GridData(GridData.FILL_HORIZONTAL);
			data.horizontalIndent = 5;		
			lblKeyValue.setLayoutData(data);

			Label lblPublicKey = new Label(content, SWT.NONE);
			lblPublicKey.setText(SecurityPlugin.getResourceString("STR_PUBLIC_KEY_LBL"));
			data = new GridData();
			data.verticalAlignment = GridData.BEGINNING;
			data.horizontalIndent = 5;		
			lblPublicKey.setLayoutData(data);
			
			Text lblPublicKeyValue = new Text(content, SWT.BORDER | SWT.READ_ONLY | SWT.WRAP | SWT.V_SCROLL) ;
			data = new GridData(GridData.FILL_HORIZONTAL);
			data.heightHint = 60;
			data.horizontalIndent = 5;		
			lblPublicKeyValue.setLayoutData(data);

			Label lblValidity = new Label(content, SWT.NONE);
			lblValidity.setText(SecurityPlugin.getResourceString("STR_VALIDITY_LBL"));
			data = new GridData();
			data.horizontalIndent = 5;		
			lblValidity.setLayoutData(data);
			
			Label lblValidityValue = new Label(content, SWT.NONE);
			data = new GridData(GridData.FILL_HORIZONTAL);
			data.horizontalIndent = 5;		
			lblValidityValue.setLayoutData(data);
					
			Label lblIssuedTo = new Label(content, SWT.NONE);
			lblIssuedTo.setText(SecurityPlugin.getResourceString("STR_ISSUED_TO_LBL"));
			data = new GridData();
			data.verticalAlignment = GridData.BEGINNING;
			data.horizontalIndent = 5;		
			lblIssuedTo.setLayoutData(data);
			
			Label lblIssuedToValue = new Label(content, SWT.NONE) ;
			data = new GridData(GridData.FILL_HORIZONTAL);
			data.horizontalIndent = 5;		
			lblIssuedToValue.setLayoutData(data);

			Label lblIssuedBy = new Label(content, SWT.NONE);
			lblIssuedBy.setText(SecurityPlugin.getResourceString("STR_ISSUED_BY_LBL"));
			data = new GridData();
			data.horizontalIndent = 5;
			data.verticalAlignment = GridData.BEGINNING;		
			lblIssuedBy.setLayoutData(data);
			
			Label lblIssuedByValue = new Label(content, SWT.NONE);
			data = new GridData(GridData.FILL_HORIZONTAL);
			data.horizontalIndent = 5;
			lblIssuedByValue.setLayoutData(data);

			Label lblSerialNum = new Label(content, SWT.NONE);
			lblSerialNum.setText(SecurityPlugin.getResourceString("STR_SERIAL_NUMBER_LBL"));
			data = new GridData();
			data.horizontalIndent = 5;		
			lblSerialNum.setLayoutData(data);

			Label lblSerialNumValue = new Label(content, SWT.NONE);

			data = new GridData(GridData.FILL_HORIZONTAL);
			data.horizontalIndent = 5;		
			lblSerialNumValue.setLayoutData(data);

			Label lblAlgorithm = new Label(content, SWT.NONE);
			lblAlgorithm.setText(SecurityPlugin.getResourceString("STR_ALGORITHM_LBL"));
			data = new GridData();
			data.horizontalIndent = 5;		
			lblAlgorithm.setLayoutData(data);

			Label lblAlgorithmValue = new Label(content, SWT.NONE);
			data = new GridData(GridData.FILL_HORIZONTAL);
			data.horizontalIndent = 5;		
			lblAlgorithmValue.setLayoutData(data);
			
			Label lblSignature = new Label(content, SWT.NONE);
			lblSignature.setText(SecurityPlugin.getResourceString("STR_SIGNATURE_LBL"));
			data = new GridData();
			data.verticalAlignment = GridData.BEGINNING;
			data.horizontalIndent = 5;		
			lblSignature.setLayoutData(data);
			
			Text lblSignatureValue = new Text(content, SWT.BORDER | SWT.READ_ONLY | SWT.WRAP | SWT.V_SCROLL) ;
			data = new GridData(GridData.FILL_HORIZONTAL);
			data.heightHint = 60;
			data.horizontalIndent = 5;		
			lblSignatureValue.setLayoutData(data);

			lblVersionValue.setText(cert.getType() + " V." + cert.getVersion());
			lblSubjectValue.setText(cert.getSubjectDN().getName());
			lblSigAlgValue.setText(cert.getSigAlgName()+", OID = "+cert.getSigAlgOID());
			lblKeyValue.setText(cert.getPublicKey().getAlgorithm());
			lblPublicKeyValue.setText(toHexString(cert.getPublicKey().getEncoded()));
			//lblPublicKeyValue.setText(cert.getPublicKey().toString());
			lblIssuedToValue.setText(cert.getSubjectDN().getName());
			lblIssuedByValue.setText(cert.getIssuerDN().getName());
			DateFormat df = DateFormat.getDateInstance(DateFormat.LONG);
			String validity = SecurityPlugin.getResourceString("STR_VALIDITY_PERIOD");
			validity = TString.change(validity, "%1", df.format(cert.getNotBefore()));
			validity = TString.change(validity, "%2", df.format(cert.getNotAfter()));
			lblValidityValue.setText(validity);
			lblSerialNumValue.setText(cert.getSerialNumber().toString());
			lblAlgorithmValue.setText(cert.getSigAlgName());
			//lblSignatureValue.setText(getThumbprintFromCertificate(cert));
			lblSignatureValue.setText(toHexString(cert.getSignature()));
									
			return content;
		}
		
		protected void createButtonsForButtonBar(Composite parent){
			createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);	
		}
	}

	class SaveCertificate extends Dialog {
		private X509Certificate cert = null;
		private String alias = null;
		private Text txtAliasValue = null;
		private Button okBtn = null;
		private String title = SecurityPlugin.getResourceString("STR_CERTIFICATE_PROP_TITLE");
		private Label lblErrMsg = null;
		private Label errorIcon = null;
		private KeyStore ks = null;
		
		public SaveCertificate(Shell shell){
			super(shell);
		}
		
		protected void init(X509Certificate cer){
			this.cert = cer;
			String storePath = SecurityPlugin.getKeyStoreLocation();
			String passw = SecurityPlugin.getWorkspaceName();

			try {
				ks = KeystoreHelper.createKeyStore(storePath, passw);
			} catch (KeyStoreException exc) {
				String text = SecurityPlugin.getResourceString("STR_INITIALIZE_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("STR_LOAD_KEYSTORE_ERROR_");
				Status err = new Status(Status.ERROR, ResourcesPlugin.PI_RESOURCES, IResourceStatus.ERROR, text, exc);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(), SecurityPlugin.getResourceString("SEC_MSG"), msg, err);
			} catch (NoSuchAlgorithmException e) {
				String text = SecurityPlugin.getResourceString("STR_ALGORITHM_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("STR_LOAD_KEYSTORE_ERROR_");
				
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,e);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			} catch (CertificateException exc) {
				String text = SecurityPlugin.getResourceString("STR_KEY_LOAD_ERROR_");
				text = TString.change(text, "%1", storePath);
				String msg = SecurityPlugin.getResourceString("STR_LOAD_KEYSTORE_ERROR_");
				
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			} catch (NoSuchProviderException e) {
				String text = SecurityPlugin.getResourceString("STR_SECURITY_PROVIDER_ERROR_");
				String msg = SecurityPlugin.getResourceString("STR_INITIALIZE_ERROR_");
				msg = TString.change(msg, "%1", SecurityPlugin.getKeyStoreLocation());
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,e);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			} catch (IOException e) {
				String text = SecurityPlugin.getResourceString("STR_LOAD_IO_EXC_");
				text = TString.change(text, "%1", storePath);
				String msg = SecurityPlugin.getResourceString("STR_LOAD_KEYSTORE_ERROR_");
				
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,e);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			}
		}
		
		protected Control createDialogArea(Composite parent){ 
			getShell().setText(title);
			Composite content = new Composite(parent, SWT.NONE);
			GridLayout layout = new GridLayout();
			GridData data = GridUtil.createFill();
			data.widthHint = 450;
			//data.heightHint = 300;
			layout.numColumns = 2;
			content.setLayout(layout);
			content.setLayoutData(data);
			
			errorIcon = new Label(content, SWT.NONE);
			errorIcon.setImage(Display.getCurrent().getSystemImage(SWT.ICON_ERROR));
			data = new GridData();
			data.horizontalIndent = 5;
			errorIcon.setLayoutData(data);
			errorIcon.setVisible(false);

			lblErrMsg = new Label(content, SWT.NONE);
			String text = SecurityPlugin.getResourceString("STR_ENTRY_IS_EMPTY");
			text = text.length() > SecurityPlugin.getResourceString("STR_ALIAS_REDUNDANT").length() ? text : SecurityPlugin.getResourceString("STR_ALIAS_REDUNDANT");
			lblErrMsg.setText(text);
			data = new GridData();
			data.horizontalIndent = 5;
			lblErrMsg.setLayoutData(data);
			lblErrMsg.setVisible(false);
			
			Label lblAlias = new Label(content, SWT.NONE);
			lblAlias.setText(SecurityPlugin.getResourceString("STR_PREF_ALIAS_NAME"));
			data = new GridData();
			data.verticalAlignment = GridData.BEGINNING;
			data.horizontalIndent = 5;		
			lblAlias.setLayoutData(data);
			
			txtAliasValue = new Text(content, SWT.BORDER) ;
			txtAliasValue.addModifyListener(new ModifyListener(){
				public void modifyText(ModifyEvent e) {
					String value = txtAliasValue.getText();
					if (value==null || value.trim().length()==0){
						SaveCertificate.this.lblErrMsg.setVisible(true);
						SaveCertificate.this.lblErrMsg.setText(SecurityPlugin.getResourceString("STR_ENTRY_IS_EMPTY"));
						SaveCertificate.this.errorIcon.setVisible(true);
						okBtn.setEnabled(false);
					} else {
						try {
							Enumeration enu = ks.aliases();
							while(enu.hasMoreElements()){
								String entry = (String) enu.nextElement();
								if (entry.equals(value.trim())){
									SaveCertificate.this.lblErrMsg.setVisible(true);
									SaveCertificate.this.lblErrMsg.setText(SecurityPlugin.getResourceString("STR_ALIAS_REDUNDANT"));
									SaveCertificate.this.errorIcon.setVisible(true);
									okBtn.setEnabled(false);
								}
							}
						} catch (Exception exp){
						}
						SaveCertificate.this.alias = value.trim();
						SaveCertificate.this.lblErrMsg.setVisible(false);
						SaveCertificate.this.errorIcon.setVisible(false);
						okBtn.setEnabled(true);
					}
				}
			});
			data = new GridData(GridData.FILL_HORIZONTAL);
			data.heightHint = 10;
			data.horizontalIndent = 5;		
			txtAliasValue.setLayoutData(data);
			return content;
		}

		protected void okPressed() {
			String storePath = SecurityPlugin.getKeyStoreLocation();
			String passw = SecurityPlugin.getWorkspaceName();
			try {
				ks.setCertificateEntry(alias, cert);
				KeystoreHelper.persistKeyStore(ks, storePath, passw);
			}
			catch(IOException e){

				String text = SecurityPlugin.getResourceString("STR_IO_SAVE_ERROR_");
				text = TString.change(text, "%1", storePath);

				text = TString.change(text, "%1", storePath);
				String msg = SecurityPlugin.getResourceString("STR_KEYSTORE_SAVE_ERROR_");
				
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,e);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			}
			catch(CertificateException exc){

				String text = SecurityPlugin.getResourceString("STR_CERTIFICATE_STORE_ERROR_");
				text = TString.change(text, "%1", storePath);
				String msg = SecurityPlugin.getResourceString("STR_KEYSTORE_SAVE_ERROR_");
				
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			}
			catch(KeyStoreException exc){
				String text = SecurityPlugin.getResourceString("STR_UNINIT_KEYSTORE_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("STR_KEYSTORE_SAVE_ERROR_");
				
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			}
			catch(NoSuchAlgorithmException exc2)
			{
				String text = SecurityPlugin.getResourceString("STR_ALGORITHM_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("STR_KEYSTORE_SAVE_ERROR_");
				
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR,text,exc2);
				ErrorDialog.openError(SecurityPlugin.getActiveWorkbenchShell(),SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			}
			setReturnCode(OK);
			close();
		}

		protected void createButtonsForButtonBar(Composite parent){
			okBtn = createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);
			okBtn.setEnabled(false);
			createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
		}
	}
	
	/*
	 * 
	 */
	public static boolean isLocalHost(String hostname)
	{
		String name = "localhost";
		if (name.equals(hostname))
			return true;
		
		try {
			  Node fnode = NodeFactory.createNode(name);
			  name = fnode.getInetAddress().getHostName();
		}
		catch(UnknownHostException exc)
		{
			return false;
		}

	    if(hostname.equals(name))
	      return true;
		
		return false;
	}
}
