/**********************************************************************
 * Copyright (c) 2005 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.filters.internal.util;

import java.text.Collator;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Vector;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.TimeBasedCorrelationQuery;
import org.eclipse.hyades.trace.ui.filters.IFilterViewer;
import org.eclipse.hyades.trace.ui.filters.internal.actions.FilterQueryProvider;
import org.eclipse.hyades.trace.ui.internal.filters.FilterTypesManager;
import org.eclipse.hyades.trace.ui.internal.filters.IFilterType;

public class FilterInformationManager {
//	private static Map _viewerTypes = null;
	
	private ScopeContainer _scopeContainer;
	private IFilterViewer _viewer;
	private String[] _scope;
	private FilterResourceHandler _resourceHandler;

	protected FilterInformationManager(IFilterViewer viewer, String[] scope)
	{
		_viewer = viewer;
		_scope = scope;
		Arrays.sort(_scope, new Comparator() {
			public int compare(Object a, Object b) {
				String str1 = (String)a;
				String str2 = (String)b;
				
				if (str1 == null)
					str1 = "";//$NON-NLS-1$
				if (str2 == null)
					str2 = "";//$NON-NLS-1$
				
				return Collator.getInstance().compare(str1, str2);				
			}
		});		

		_resourceHandler = FilterResourceHandler.instance();
		_scopeContainer = ScopeContainer.instance();
		
		if (_viewer != null)
		{
			_viewer.setFilterQueryProvider(new FilterQueryProvider(this));
			_scopeContainer.addFilterViewer(_viewer);
		}
	}
	
	private String getSelectedScope()
	{
		String selectedScope = null;
		if (_viewer != null)
			selectedScope = _scopeContainer.getSelectedScope(_viewer.getClass().getName());
		
		if (selectedScope == null && _scope.length > 0)
			return _scope[0];
		else
			return selectedScope;
	}

	public static FilterInformationManager instance(IFilterViewer viewer)
	{
		return new FilterInformationManager(viewer, viewer.getFilterScope());
	}
	
	public static FilterInformationManager instance(String scope)
	{
		String [] scopeArray = new String[1];
		scopeArray[0] = scope;
		
		return new FilterInformationManager(null, scopeArray);
	}	
	
	public IFilterType[] getFilterTypes()
	{
		return FilterTypesManager.instance().getFilterTypes(_scope);
	}
	
	
	private void updateViewers() {
		IFilterViewer[] viewers = _scopeContainer.getFilterViewers(getSelectedScope());
		
		for (int i = 0; i < viewers.length; i++)
		{
			_scopeContainer.setSelectedScope(viewers[i].getClass().getName(), getSelectedScope());			
			viewers[i].filterUpdated();
		}
	}
	
	public FilterQueries selectedFilter()
	{
		if (selectedFilterName() != null)
			return filter(selectedFilterName());
		
		return null;
	}
	
	public String selectedFilterName()
	{
		return _scopeContainer.getSelectedFilterName(getSelectedScope());
	}

	public void selectedFilterName(String selectedFilterName)
	{
		selectedFilterName(selectedFilterName, true);
	}
	
	public void selectedFilterName(String selectedFilterName, boolean updateViewer)
	{
		if (selectedFilterName != null)
		{
			String[] scopeApplied = _scopeContainer.setSelectedFilterName(selectedFilterName);
			String selectedScope = FilterTypesManager.firstScopeIntersection(_scope, scopeApplied);
			
			if (_viewer != null)
				_scopeContainer.setSelectedScope(_viewer.getClass().getName(), selectedScope);			
		}
		else
			_scopeContainer.setSelectedFilterName(getSelectedScope(), selectedFilterName);
		
		if (updateViewer)
			updateViewers();
	}	
	
	public int size()
	{
		return getFiltersArray().length;
	}
	
	public FilterQueries filter(int i)
	{
		return (FilterQueries)getFiltersArray()[i];
	}
	
	public Object[] getFiltersArray()
	{
		Object [] filters = _resourceHandler.toArray();
		
		Vector filtersOfTheseTypes = new Vector();
		
		for (int i = 0; i < filters.length; i++)
		{
			IFilterType filterType = FilterTypesManager.instance().getFilterType(((FilterQueries)filters[i]).type());
			if (filterType != null && hasScope(filterType.scope()))
				filtersOfTheseTypes.addElement(filters[i]);
		}
		
		Object [] sortedFilters = filtersOfTheseTypes.toArray();
		
		sortFilters(sortedFilters);
		
		return sortedFilters;
	}
	
	public void sortFilters(Object[] filters)
	{
		Arrays.sort(filters, new Comparator() {
			public int compare(Object a, Object b) {
				String name1 = null;
				if (a instanceof FilterQueries)
					name1 = ((FilterQueries)a).name();

				String name2 = null;
				if (b instanceof FilterQueries)
					name2 = ((FilterQueries)b).name();				
				
				if (name1 == null)
					name1 = "";//$NON-NLS-1$
				if (name2 == null)
					name2 = "";//$NON-NLS-1$
				
				return Collator.getInstance().compare(name1, name2);				
			}
		});		
	}
	
	public void restore()
	{
		_resourceHandler.load();
	}
	
	public void save()
	{
		_resourceHandler.save();
	}
	
	public FilterQueries createFilterQueries(String type)
	{
		return _resourceHandler.createFilterQueries(type);
	}
	
	public FilterQueries createFilterQueries(String type, SimpleSearchQuery standardQuery, SimpleSearchQuery advancedQuery)
	{
		return _resourceHandler.createFilterQueries(type, standardQuery, advancedQuery);
	}	
	
	public void addFilter(FilterQueries fq)
	{
		_resourceHandler.addFilter(fq);
	}
	
	public void removeFilter(FilterQueries fq)
	{
		_resourceHandler.removeFilter(fq);
	}
	
	public void removeFilters(int [] indeces)
	{
		Object[] filters = getFiltersArray();

		FilterQueries [] filterElementsToRemove = new FilterQueries[indeces.length];		
		int i;
		
		for (i = 0; i < indeces.length; i++)
		{
			filterElementsToRemove[i] = (FilterQueries) filters[indeces[i]];  
		}
		
		for (i = 0; i < filterElementsToRemove.length; i++)
		{
			removeFilter(filterElementsToRemove[i]);  
		}
	}
	
	public FilterQueries filter(String name)
	{
		return _resourceHandler.filter(name);
	}
	
	public boolean filterNameExists(String name)
	{
		return filter(name) != null;
	}		
	
	private boolean hasScope(String[] scope)
	{
		return FilterTypesManager.scopesHaveIntersection(_scope, scope);
	}
	
//	private static synchronized String[] getScope(IFilterViewer viewer) {
//
//		if (_viewerTypes == null)
//		{
//			_viewerTypes = new HashMap();
//			IAssociationMappingRegistry registry = (AssociationMappingRegistry)TraceAssociationManager.getTraceViewMappingRegistry();
//			IAssociationMapping mapping = registry.getAssociationMapping(HyadesUIPlugin.EP_ANALYZER_EXTENSIONS);
//			
//			String[] types = mapping.getTypes();
//			for(int i = 0, maxi = types.length; i < maxi; i++)
//			{
//				IAssociationDescriptor[] descriptors = mapping.getAssociationDescriptors(types[i]);
//				for (int j = 0; j < descriptors.length; j++)
//				{
//					if (IViewerRunnable.class.isAssignableFrom(descriptors[j].getImplementationClass()))
//					{
//						IViewerRunnable viewerRunnable = (IViewerRunnable)descriptors[j].createImplementationClassInstance();
//						_viewerTypes.put(viewerRunnable.fullyQualifiedViewerClassName(), descriptors[j].types());
//					}
//				}
//			}
//			
//		}
//
//		String [] types = (String []) _viewerTypes.get(viewer.getClass().getName());		
//		if (types != null)
//			return types;
//		else
//			return new String[0];
//	}
	
	
	public SimpleSearchQuery getCurrentFilter()
	{
		return createCombinedFilter(selectedFilter());
	}
	
	public SimpleSearchQuery createCombinedFilter(FilterQueries fq)
	{
		if (fq == null)
			return null;
		
		SimpleSearchQuery combinedQuery;
		if (fq.standard() instanceof TimeBasedCorrelationQuery)
			combinedQuery = ExtensionsFactory.eINSTANCE.createTimeBasedCorrelationQuery();
		else
			combinedQuery = ExtensionsFactory.eINSTANCE.createSimpleSearchQuery();
		
		combinedQuery.setName(fq.name());

		if (fq.standard().getOrderByExpresions().size() > 0)
		{
			combinedQuery.getOrderByExpresions().add(EcoreUtil.copy((EObject)fq.standard().getOrderByExpresions().get(0)));
			combinedQuery.setMaxElements(fq.standard().getMaxElements());
		}
		
		
		LogicalExpression standardLogicalExpression = null;
		LogicalExpression advancedLogicalExpression = null;

		if (fq.standard().getWhereExpression() != null && ((LogicalExpression)fq.standard().getWhereExpression()).getArguments().size() > 0)
			standardLogicalExpression = (LogicalExpression)EcoreUtil.copy(fq.standard().getWhereExpression());
		
		if (fq.advanced().getWhereExpression() != null && ((LogicalExpression)fq.advanced().getWhereExpression()).getArguments().size() > 0)
			advancedLogicalExpression = (LogicalExpression)EcoreUtil.copy(fq.advanced().getWhereExpression());
		
		if ((standardLogicalExpression != null && advancedLogicalExpression != null)
				|| (standardLogicalExpression == null && advancedLogicalExpression == null))
		{
			LogicalExpression logicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
			logicalExpression.setOperator(LogicalOperators.AND_LITERAL);
			
			if (standardLogicalExpression != null && advancedLogicalExpression != null)
			{
				logicalExpression.getArguments().add(standardLogicalExpression);
				logicalExpression.getArguments().add(advancedLogicalExpression);
			}
			
			combinedQuery.setWhereExpression(logicalExpression);			
		}
		else if (standardLogicalExpression != null)
			combinedQuery.setWhereExpression(standardLogicalExpression);
		else if (advancedLogicalExpression != null)
			combinedQuery.setWhereExpression(advancedLogicalExpression);		
				
		return combinedQuery;
	}
	
	public void viewerDisposed()
	{
		if (_viewer != null)
			_scopeContainer.removeFilterViewer(_viewer);
	}
	
}
