/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RefreshFromLocalAction.java,v 1.16 2005/03/19 00:47:46 curtispd Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.actions;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.IDeleteListener;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.navigator.SimpleLayout;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.SelectionProviderAction;
import org.eclipse.ui.dialogs.ListSelectionDialog;

/*
* CONTEXT_ID refl0000 for refresh from local action
*/

public class RefreshFromLocalAction extends SelectionProviderAction
									implements ISelectionChangedListener 
{
  private INavigator fViewer;

  public RefreshFromLocalAction(INavigator viewer)
  {
	super(viewer.getViewer(), UIPlugin.getResourceString("UNLOAD_MEMORY"));
	setDescription(UIPlugin.getResourceString("RELOAD_LOCAL_FILE"));
	fViewer = viewer;

	org.eclipse.ui.help.WorkbenchHelp.setHelp(
		this,
		UIPlugin.getPluginId()+".refl0000");
	
	setToolTipText(UIPlugin.getResourceString("RELOAD_LOCAL_FILE_TOOLTIP"));	
  }
  
  public void run()
  {
	List list = getProcessProxies(true);
	
	String preferenceKey = UIPlugin.getDefault().getPreferenceStore().getString(TraceConstants.RELOAD_INFO);
	IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
	Shell shell= window.getShell();
	boolean saveAndChange = false;
	
	if (UIPlugin.getDefault().getPreferenceStore().getBoolean(TraceConstants.RELOAD_INFO)) {
		MessageDialogWithToggle dialog = MessageDialogWithToggle.openOkCancelConfirm(shell, 
				UIPlugin.getResourceString("RELOAD_INFORMATION_TITLE"), //$NON-NLS-1$
				UIPlugin.getResourceString("RELOAD_INFORMATION"), //$NON-NLS-1$
				UIPlugin.getResourceString("ATTACH_MESSAGE_CHECKBOX_MSG"), //$NON-NLS-1$
				false, UIPlugin.getDefault().getPreferenceStore(),
				preferenceKey);
		
		saveAndChange = (dialog.getReturnCode()== IDialogConstants.OK_ID);
		
		if (dialog.getToggleState()) {
			UIPlugin.getDefault().getPreferenceStore().setValue(TraceConstants.RELOAD_INFO, Boolean.toString(!saveAndChange));
		}
		//Activate the shell if necessary so the prompt is visible
		if (shell.getMinimized()) {
			shell.setMinimized(false);
		}
		if (!saveAndChange)
			return;
	}
		
	if(!list.isEmpty())
	{
		final Object[] listArray = list.toArray();
		IStructuredContentProvider contentProvider = new IStructuredContentProvider(){

			public void dispose() {
			}

			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			}

			public Object[] getElements(Object inputElement) {
				return listArray;
			}
		};
		
		ListSelectionDialog dlg =
			new ListSelectionDialog(
					fViewer.getViewSite().getShell(),
					listArray,
					contentProvider,
					new SimpleLayout(null).getLabelProvider(),
					UIPlugin.getResourceString("FREE_MEMORY_QUESTION"));
		dlg.setInitialSelections(listArray);
		dlg.setTitle(UIPlugin.getResourceString("UNLOAD_MEMORY"));
		
		int result = dlg.open();	
		if(result == ListSelectionDialog.CANCEL)
			return;	
		
		if(!Arrays.asList(dlg.getResult()).isEmpty())
		{
			try{
				SaveAction saveLogs = new SaveAction(fViewer);
				saveLogs.run(result);
			}catch (Exception e) {
				HyadesUIPlugin.logError(e);
			}
		}else{
			String title = UIPlugin.getResourceString("TRACE_MSG");	
			String msg = UIPlugin.getResourceString("RELOAD_QUESTION");	
			
			if(! MessageDialog.openQuestion(fViewer.getViewSite().getShell(), title, msg))
				return;			
		}
	}		
	
	ProfileEvent event = new ProfileEvent();
	event.setSource(ResourcesPlugin.getWorkspace().getRoot());
	
	handleResources();
	
	if(!(fViewer instanceof PDProjectExplorer))	
	{	
		event.setType(ProfileEvent.REFRESH_LOG_NAVIGATOR);
		UIPlugin.getDefault().notifyProfileEventListener(event);		
	}
	
	event.setType(ProfileEvent.REFRESH_PD_EXPLORER);		
	UIPlugin.getDefault().notifyProfileEventListener(event);
	
  }  
  /**
   * @param list
   */
  protected List getProcessProxies(boolean onlyModified) 
  {
	List list = new ArrayList();
	for (Iterator iter = HierarchyResourceSetImpl.getInstance().getResources().iterator(); iter.hasNext();) 
	{
		Resource element = (Resource) iter.next();
		if(element.isLoaded() && (!onlyModified || element.isModified()))
		{
			if(element.getContents().size()>0 && element.getContents().get(0) instanceof TRCAgent)
			{
				TRCAgent agent = (TRCAgent)element.getContents().get(0);
				if(!agent.eIsProxy() && agent.getAgentProxy()!=null && agent.getAgentProxy().getProcessProxy()!=null)
					if(!list.contains(agent.getAgentProxy().getProcessProxy()))
						list.add(agent.getAgentProxy().getProcessProxy());
			}
		}
	}
	return list;
  }
  /**
  * 
  */

  public void selectionChanged(IStructuredSelection selection)
  {
	setEnabled(isEnabled());
  }

  /* (non-Javadoc)
   * @see org.eclipse.jface.action.Action#isEnabled()
   */
  public boolean isEnabled() {
	  return !getProcessProxies(false).isEmpty();
}
  
  public void dispose() {  	
	  super.dispose();
		
	  fViewer = null;
  }
  
  protected void handleResources(){
  	
  	PDContentProvider.resetMonitors();
  	
	Object[] resources = HierarchyResourceSetImpl.getInstance().getResources().toArray();
	for (int idx=0;idx < resources.length;idx++) 
	{
		Resource element = (Resource) resources[idx];
		if(element!=null && element.isLoaded()){
			Object[] elements = element.getContents().toArray();
			for(int i=0;i<elements.length;i++)
			{
				Object obj = elements[i];

				List deletelist = UIPlugin.getDefault().getDeleteListeners();
				for(int j=0; j<deletelist.size();j++)
				{
					((IDeleteListener)deletelist.get(j)).deregister(obj);
				}

				element.unload();
			}
		}
	}
  }
}
