/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceProfileOverviewUI.java,v 1.48 2005/04/18 17:51:35 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.core;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.security.util.TString;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.internal.util.AbstractChangeable;
import org.eclipse.hyades.trace.ui.internal.wizard.AddWizard;
import org.eclipse.hyades.trace.ui.internal.wizard.AddWizardDialog;
import org.eclipse.hyades.trace.ui.internal.wizard.EditWizard;
import org.eclipse.hyades.trace.ui.internal.wizard.EditWizardDialog;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSet;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnLayoutData;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.help.WorkbenchHelp;

/*
 * CONTEXT_ID pfpp0001 for Trace profilings table in Profile profilings ui page
 * CONTEXT_ID pfpp0002 for New button in Profile profilings ui page
 * CONTEXT_ID pfpp0003 for Edit button in Profile profilings ui page
 * CONTEXT_ID pfpp0004 for Delete button in Profile profilings ui page
 * CONTEXT_ID pfpp0005 for Up button in Profile profilings ui page
 * CONTEXT_ID pfpp0006 for Down button in Profile profilings ui page
 * CONTEXT_ID pfpp0007 for Package entry field in Profile profilings add/edit dialog page
 * CONTEXT_ID pfpp0008 for Method Name entry field in Profile profilings for Package entry field in Profile profilings add/edit dialog page
 * CONTEXT_ID pfpp0009 for Rule combobox in Profile profilings for Package entry field in Profile profilings add/edit dialog page
 * CONTEXT_ID pfps0001 for Trace profilings Set table in Profile profilings ui page
 * CONTEXT_ID pfps0002 for New Set button in Profile profilings ui page
 * CONTEXT_ID pfps0003 for Edit Set button in Profile profilings ui page
 * CONTEXT_ID pfps0004 for Delete Set button in Profile profilings ui page
 */
public class TraceProfileOverviewUI extends AbstractChangeable {
	private Composite _result;

	private List _profilingTypeList;
	
	private Table _table;
	private Table _profilingSetTable;
	private TableViewer _tableViewer;
	private CheckboxTableViewer _profilingSetTableViewer;

	private Button _addSetBtn;
	private Button _removeSetBtn;
	private Button _editBtn;

	private Label _contentLabel;
	private Label _profilingSetLabel;	
	
	private ProfilingSetsManagerCopy _managerCopy;
	
	private ILaunchConfiguration _conf;	

	private static class ProfilingSetLabelProvider extends LabelProvider {
		public String getText(Object element) {
			return ((IProfilingSet)element).getName();
		}

		public Image getImage(Object element) {
//			return null;
			return PDPluginImages.getImage(PDPluginImages.IMG_FILTER);
		}
	}

	private static class ProfilingSetContentProvider implements IStructuredContentProvider {
		public Object[] getElements(Object parent) {
			ProfilingSetsManagerCopy copy = (ProfilingSetsManagerCopy)parent;
			return copy.getProfilingSets().values().toArray();
		}

		public void inputChanged(Viewer viewer,	Object oldInput, Object newInput) {
		}

		public void dispose() {
		}
	}

	private class ListContentProvider implements IStructuredContentProvider {
		public Object[] getElements(Object input) {
			if (input != null && input instanceof List) {
				return ((List)input).toArray();
			}
			return new Object[0];
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}

		public void dispose() {
		}
	}
	
	
	public void dataChanged()
	{
		changed();
	}
	
	public Composite createControl(Composite parent) {
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.verticalSpacing = 0;

		_result = new Composite(parent, SWT.NONE);
		_result.setLayout(layout);
		_result.setLayoutData(GridUtil.createFill());
			
		createProfilingSetTable(_result);
		createProfilingTypeTable(_result);

		return _result;
	}
	
	private void createProfilingSetTable(Composite parent) {
		GridData data;
		GridLayout layout;
		
		Label label = new Label(parent, SWT.NONE);
		label.setText(UIPlugin.getResourceString("STR_SELECT_PROFILING_SET"));
		data = new GridData();
		data.horizontalSpan = 2;
		label.setLayoutData(data);
			
		Composite tableGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		data = GridUtil.createFill();
		data.heightHint = 140;
		tableGroup.setLayoutData(data);

		//_profilingSetTable = new Table(tableGroup, SWT.V_SCROLL | SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION | SWT.CHECK);
		_profilingSetTable = new Table(tableGroup, SWT.V_SCROLL	| SWT.H_SCROLL | SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION);
		_profilingSetTable.setLayoutData(GridUtil.createFill());
		_profilingSetTable.setLinesVisible(false);

		TableColumn column = new TableColumn(_profilingSetTable, SWT.NONE);
		column.setWidth(340);
		
		_profilingSetTableViewer = new CheckboxTableViewer(_profilingSetTable);
		_profilingSetTableViewer.setContentProvider(new ProfilingSetContentProvider());
		_profilingSetTableViewer.setLabelProvider(new ProfilingSetLabelProvider());
		_profilingSetTableViewer.setSorter(new ViewerSorter() {
			public int compare(Viewer viewer, Object e1, Object e2) {
				String a = ((IProfilingSet)e1).getName();
				String b = ((IProfilingSet)e2).getName();
				return a.compareTo(b);
			}
		});
		
		_profilingSetTableViewer.setInput(getManagerCopy());
//		_profilingSetTableViewer.addCheckStateListener(new ICheckStateListener() {
//			public void checkStateChanged(CheckStateChangedEvent event) {
//				IProfilingSet set = (IProfilingSet)event.getElement();
//				_profilingSetTableViewer.setSelection(new StructuredSelection(set));					
//			}
//		});

		Composite btnGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		btnGroup.setLayout(layout);
		btnGroup.setLayoutData(GridUtil.createVerticalFill());

		Composite addGroup = new Composite(btnGroup, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		addGroup.setLayout(layout);
		addGroup.setLayoutData(GridUtil.createVerticalFill());

		_addSetBtn = new Button(addGroup, SWT.PUSH);
		_addSetBtn.setText(UIPlugin.getResourceString("ADD_PROFILINGSET_TEXT"));
		_addSetBtn.setToolTipText(UIPlugin.getResourceString("ADD_FILTER_TOOLTIP_TEXT"));
		_addSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_removeSetBtn = new Button(addGroup, SWT.PUSH);
		_removeSetBtn.setText(UIPlugin.getResourceString("REMOVE_PROFILINGSET_TEXT"));
		_removeSetBtn.setToolTipText(UIPlugin.getResourceString("REMOVE_TOOLTIP_TEXT"));
		_removeSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_editBtn = new Button(addGroup, SWT.PUSH);
		_editBtn.setText(UIPlugin.getResourceString("EDIT_PROFILINGSETCONTENTS_TEXT"));
		_editBtn.setToolTipText(UIPlugin.getResourceString("EDIT_FILTER_TOOLTIP_TEXT"));
		_editBtn.setLayoutData(GridUtil.createHorizontalFill());
		
		
		_addSetBtn.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				addProfilingSet();
			}
		});

		_removeSetBtn.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				removeProfilingSet();
			}
		});

		_editBtn.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				IStructuredSelection selection = (IStructuredSelection)_profilingSetTableViewer.getSelection();
				if (!selection.isEmpty()) {
					IProfilingSet element = (IProfilingSet)selection.getFirstElement();
					showEditProfilingSetDialog(element);
					showProfilingTypeDetails(element);
				}
			}
		});

		_profilingSetTableViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				IStructuredSelection selection = (IStructuredSelection)event.getSelection();
				if (!selection.isEmpty()) {
					IProfilingSet set = (IProfilingSet)selection.getFirstElement();
					_profilingSetTableViewer.setAllChecked(false);
					_profilingSetTableViewer.setChecked(set, true);
					_profilingSetTableViewer.refresh();
					
					getManagerCopy().setDefaultSet(set);
					showProfilingTypeDetails(set);
					changed();
					
					_profilingSetLabel.setText(set.getDescription());
					_removeSetBtn.setEnabled(getManagerCopy().getProfilingSets().size() > 1);
				}
			}
		});
		
		_profilingSetTableViewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				IStructuredSelection selection = (IStructuredSelection)event.getSelection();
				if (!selection.isEmpty()) {
					IProfilingSet set = (IProfilingSet)selection.getFirstElement();
					showEditProfilingSetDialog(set);
					showProfilingTypeDetails(set);
				}
			}
		});
		

		WorkbenchHelp.setHelp(_profilingSetTable, UIPlugin.getPluginId() + ".pfps0001");
	}

	private void createProfilingTypeTable(Composite parent) {
		GridData data;
		GridLayout layout;

		createVerticalSpacer(parent,2);
		
		_contentLabel = new Label(parent, SWT.NONE);
		_contentLabel.setText(UIPlugin.getResourceString("STR_PROFILINGSET_CONTENTS"));
		data = new GridData();
		data.horizontalSpan = 2;
		data.horizontalIndent=5;
		_contentLabel.setLayoutData(data);

		Composite tableGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		data = GridUtil.createFill();
		data.heightHint = 125;
		tableGroup.setLayoutData(data);

		TableLayout tableLayout = new TableLayout();
		_table = new Table(tableGroup, SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER );
		_table.setEnabled(false);
		_table.setLinesVisible(false);
		_table.setLayoutData(GridUtil.createFill());
		_table.setLayout(tableLayout);
		_table.setHeaderVisible(true);

		String headers[] = {
			UIPlugin.getResourceString("HEADER_PROFILING_TYPE")
		};
		
		ColumnLayoutData[] layouts = {
			new ColumnPixelData(300, true),
		};

		TableColumn[] columns = new TableColumn[headers.length];

		for (int i = 0; i < headers.length; i++) {
			tableLayout.addColumnData(layouts[i]);
			TableColumn tc = new TableColumn(_table, SWT.NONE, i);
			tc.setResizable(layouts[i].resizable);
			tc.setText(headers[i]);
			columns[i] = tc;
		}

		_profilingTypeList = new ArrayList();
		_tableViewer = new TableViewer(_table);
		_tableViewer.setContentProvider(new ListContentProvider());
		_tableViewer.setLabelProvider(new LabelProvider());
		_tableViewer.setInput(_profilingTypeList);

		Composite btnGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		btnGroup.setLayout(layout);
		btnGroup.setLayoutData(GridUtil.createVerticalFill());

		Composite addGroup = new Composite(btnGroup, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		addGroup.setLayout(layout);
		addGroup.setLayoutData(GridUtil.createVerticalFill());
		
		Composite moveGroup = new Composite(btnGroup, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		moveGroup.setLayout(layout);
		moveGroup.setLayoutData(GridUtil.createHorizontalFill());

		Label label = new Label(parent, SWT.NONE);
		data = new GridData();
		data.horizontalSpan = 2;
		data.horizontalIndent = 5;
		label.setLayoutData(data);
		label.setText(UIPlugin.getResourceString("STR_DESCP"));
		setBold(label);
		
		_profilingSetLabel = new Label(parent, SWT.LEFT + SWT.WRAP);
		data = new GridData();
		data.horizontalSpan = 2;
		data.heightHint = 65;
		data.widthHint = 600;
		data.horizontalIndent = 5;
		_profilingSetLabel.setLayoutData(data);

		WorkbenchHelp.setHelp(_table, UIPlugin.getPluginId() + ".pfpp0002");
	}

	private void setBold(Label label) {
		FontData[] fontData = label.getFont().getFontData();
		for (int i = 0; i < fontData.length; i++) {
			fontData[i].setStyle(fontData[i].getStyle() | SWT.BOLD);
		}
		label.setFont(new Font(null,fontData));	
	}
	
	 // Create some empty space.
	protected void createVerticalSpacer(Composite comp, int colSpan) {
		Label label = new Label(comp, SWT.NONE);
		GridData gd = new GridData();
		gd.horizontalSpan = colSpan;
		label.setLayoutData(gd);
		label.setFont(comp.getFont());
	}	
	
	private void showProfilingTypeDetails(IProfilingSet set) {
		_profilingTypeList.clear();
		
		Collection types = new ArrayList();
		Iterator iter = set.getProfilingTypes().iterator();
		while(iter.hasNext()) {
			String type = (String)iter.next();
			IProfilingSetType pSetType = (IProfilingSetType)getManagerCopy().getProfilingTypes().get(type);
			types.add(pSetType);
		}
		
		/*
		 * The types must be sorted at this stage, not in the viewer.
		 * Viewers sort individual items, but in this case, we expand
		 * each item's description into potentially several new items,
		 * but the description must remain next to its type.
		 */
		IProfilingSetType[] sortedTypes = new IProfilingSetType[types.size()];
		types.toArray(sortedTypes);
		Arrays.sort(sortedTypes, new Comparator() {
			public int compare(Object o1, Object o2) {
				IProfilingSetType t1 = (IProfilingSetType)o1;
				IProfilingSetType t2 = (IProfilingSetType)o2;
				return t1.getName().compareTo(t2.getName());
			}
		});
		
		for(int i=0;i<sortedTypes.length;i++) {
			IProfilingSetType pSetType = sortedTypes[i];
			
			if(pSetType != null) {
				pSetType.setDescription(pSetType.getProfilingType().getDescription(getManagerCopy()));
				boolean descriptionExists = (pSetType.getDescription() != null) &&  (pSetType.getDescription().length() > 0);

				if (descriptionExists) {
					if (i != 0) {
						_profilingTypeList.add("");
					}
					_profilingTypeList.add(pSetType.getName());
					addDescription(_profilingTypeList, pSetType.getDescription());
					_profilingTypeList.add("");
				}
				else {
					_profilingTypeList.add(pSetType.getName());
				}
			}
		}

		_table.setRedraw(false);
		_tableViewer.refresh();
		_table.setRedraw(true);
	}
	
	/**
	 * adds the description of the profiling type to the List
	 * handles multi line descriptions
	 * the delimiter for the new line is \n
	 */
	private void addDescription(List profilingList, String description)
	{
		String delim = "\n";
		
		StringTokenizer tokens = new StringTokenizer(description, delim, false);
		while (tokens.hasMoreElements())
		{
			String token = tokens.nextToken();
			profilingList.add(token);
		}
	}

	private void showEditProfilingSetDialog(IProfilingSet set) {
		EditWizard wizard = new EditWizard(this);
		EditWizardDialog dialog = new EditWizardDialog(_editBtn.getShell(), wizard);

		wizard.initializeSet(set);
		
		dialog.create();
		dialog.open();
	}

	public TableViewer getTableViewer()
	{
		return _profilingSetTableViewer;
	}
	
	private void addProfilingSet() {
		AddWizard wizard = new AddWizard(this);
		AddWizardDialog dialog = new AddWizardDialog(new Shell(Display.getDefault()), wizard);

		wizard.initializeSet();

		dialog.create();
		dialog.open();		
	}
	
	private void removeProfilingSet() {
		IStructuredSelection selection = (IStructuredSelection) _profilingSetTableViewer
		.getSelection();
		if (!selection.isEmpty()) {
			IProfilingSet set = (IProfilingSet) selection.getFirstElement();
			try {
				boolean remove = true;
				String profilingSetId = set.getId();
		
				ILaunchConfiguration[] configurations = DebugPlugin
						.getDefault().getLaunchManager()
						.getLaunchConfigurations();
		
				String nl = System.getProperties()
						.getProperty("line.separator");
				String configurationNames = "";
				String confNamePrefix = "     " + UIPlugin.getResourceString("STR_REMOVE_PROFILINGSET_WARNING_MESSAGE_POINT")+" ";
		
				Vector configurationsV = new Vector();
				for (int i = 0; i < configurations.length; i++) {
					if (!_conf.getName().equals(configurations[i].getName())
							&& profilingSetId
									.equals(configurations[i]
											.getAttributes()
											.get(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET)))
					{
						configurationNames = configurationNames
								+ confNamePrefix + configurations[i].getName() + nl;
						configurationsV.addElement(configurations[i]);
					}
				}
		
				if (configurationNames.length() > 0) {
					
					String message = UIPlugin
							.getResourceString("STR_REMOVE_PROFILINGSET_WARNING_MESSAGE1")
							+ nl
							+ configurationNames
							+ TString.change(UIPlugin
									.getResourceString("STR_REMOVE_PROFILINGSET_WARNING_MESSAGE2"),
									"%1",
									ProfilingSetsManager.instance().getInitialDefaultSet(profilingSetId).getName().trim())
							+ nl + nl
							+ UIPlugin
									.getResourceString("STR_REMOVE_PROFILINGSET_WARNING_MESSAGE3");									
		
					String[] buttons = {
							UIPlugin
									.getResourceString("STR_REMOVE_PROFILINGSET_WARNING_YES_BUTTON"),
							UIPlugin
									.getResourceString("STR_REMOVE_PROFILINGSET_WARNING_NO_BUTTON") };
		
					MessageDialog messageDialog = new MessageDialog(
							_removeSetBtn.getShell(),
							UIPlugin
									.getResourceString("STR_REMOVE_PROFILINGSET_WARNING_TITLE"),
							null, message, MessageDialog.WARNING, buttons, 0);
					messageDialog.create();
					remove = messageDialog.open() == MessageDialog.OK;
		
				}
				if (remove) {
					getManagerCopy().getProfilingSets().remove(set.getId());
					getManagerCopy().applyChanges();
					_profilingSetTableViewer.refresh();
					
					IProfilingSet newSet = getManagerCopy().getInitialDefaultSet(null);					
					
					ILaunchConfiguration currentConf;
					ILaunchConfigurationWorkingCopy currentConfWorkingCopy;
					for (int i = 0; i < configurationsV.size(); i++)
					{
						currentConf = (ILaunchConfiguration)configurationsV.elementAt(i);
						currentConfWorkingCopy = currentConf.getWorkingCopy();
						currentConfWorkingCopy.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, newSet.getId());
						currentConfWorkingCopy.doSave();
					}
		
					_profilingSetTableViewer
							.setSelection(new StructuredSelection(newSet));
				}
			} catch (CoreException e) {
				HyadesUIPlugin.logError(e);
			}
		}
	}
	
	/**
	 * Initialize ui content using selected configuration attributes
	 * @param conf
	 */
	public void initializeFrom(ILaunchConfiguration conf) {
		_conf = conf;		
		getManagerCopy().initializeFrom(conf);
		
		IProfilingSet set = getManagerCopy().getDefaultSet();
		if(set != null)	{
			_profilingSetTableViewer.setSelection(new StructuredSelection(set));
		}
	}
	
	public void performApply(ILaunchConfigurationWorkingCopy wc) {
		_managerCopy.performApply(wc);
	}
	
	public boolean isValid(ILaunchConfiguration conf) {
		return true;
	}
	
	public ProfilingSetsManagerCopy getManagerCopy() {
		if(_managerCopy == null)
			_managerCopy = new ProfilingSetsManagerCopy();
		return _managerCopy;
	}
}
