/**********************************************************************
 * Copyright (c) 2005 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.filters;

import java.util.Vector;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.filters.IFilterAttribute;
import org.eclipse.hyades.trace.ui.internal.util.TString;

public class BasicFeatureFilterAttribute implements IFilterAttribute {

	public static String OR_SEPARATOR = "|";
	
	private String _id;
	private String _displayText;
	private RelationalOperators[] _operators; 
	
	private EStructuralFeature _feature;
	
	private boolean _useNotLikeOperator;
	
	public BasicFeatureFilterAttribute(String id, String displayText, EStructuralFeature feature, RelationalOperators[] operators, boolean useNotLikeOperator)
	{
		_id = id;
		_displayText = displayText;
		_operators = operators;
		
		_feature = feature;
		_useNotLikeOperator = useNotLikeOperator;
	}
	
	public EStructuralFeature feature()
	{
		return _feature;
	}
	
	public String id() {
		return _id;
	}

	public String displayText() {
		return _displayText;
	}

	public String validate(String operator, String value) {
		String values[] = getValues(value);
		
		for (int i = 0; i < values.length; i++)
		{
			if (_feature instanceof EAttribute)
			{
				try
				{
					Object valueFromString = EcoreUtil.createFromString(((EAttribute)_feature).getEAttributeType(), values[i]);
				}
				catch (Exception e)
				{
					String errorMsg = TString.change(UIPlugin.getResourceString("STR_ST_FILTER_INVALID_TYPE_FOR_ATTRIBUTE"), "%1", values[i]);
					errorMsg = TString.change(errorMsg, "%2", displayText());
					
					return  errorMsg;	
				}
			}
		}
			
		return null;
	}

	public String[] operators() {
		String[] operators = new String[_operators.length + (_useNotLikeOperator?1:0)];
		
		int j = 0;
		
		for (int i = 0; i < _operators.length; i++)
		{
			operators[j++] = getOperatorString(_operators[i]);
			if (RelationalOperators.get(RelationalOperators.LIKE).equals(_operators[i]) && _useNotLikeOperator)
				operators[j++] = UIPlugin.getResourceString("STR_FILTER_OPERATOR_NOT_LIKE");
		}
		
		if (_useNotLikeOperator && j < operators.length)
			operators[j] = UIPlugin.getResourceString("STR_FILTER_OPERATOR_NOT_LIKE");			
		
		return operators;
	}
	
	
	protected String getValueFromDisplayValue(String value)
	{
		return value;
	}

	private WhereExpression getSubWhereExpression(EStructuralFeature feature, RelationalOperators operator, String value, boolean notExpression)
	{
		if (notExpression)
		{
			LogicalExpression logicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
			logicalExpression.setOperator(LogicalOperators.NOT_LITERAL);
			
			logicalExpression.getArguments().add(ModelUtils.createBinaryExpression(feature, operator, getValueFromDisplayValue(value), false));
			
			return logicalExpression;
		}
		else
			return ModelUtils.createBinaryExpression(feature, operator, getValueFromDisplayValue(value), false);
	}
	
	public WhereExpression getWhereExpression(String operator, String value) {
		
		String[] values = getValues(value);
		if (values.length == 1)
		{
			RelationalOperators relationOperator = getOperator(operator);
			
			if (relationOperator != null)
				return getSubWhereExpression(_feature, relationOperator, values[0], false);
			else if (UIPlugin.getResourceString("STR_FILTER_OPERATOR_NOT_LIKE").equals(operator))
				return getSubWhereExpression(_feature, RelationalOperators.get(RelationalOperators.LIKE), values[0], true);
			else
				return null;
		}
		else
		{
			LogicalExpression logicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
			logicalExpression.setOperator(LogicalOperators.OR_LITERAL);				
			
			for (int j = 0; j < values.length; j++)
			{
				RelationalOperators relationOperator = getOperator(operator);
				
				if (relationOperator != null)
					logicalExpression.getArguments().add(getSubWhereExpression(_feature, relationOperator, values[j], false));
				else if (UIPlugin.getResourceString("STR_FILTER_OPERATOR_NOT_LIKE").equals(operator))
					logicalExpression.getArguments().add(getSubWhereExpression(_feature, RelationalOperators.get(RelationalOperators.LIKE), values[j], true));
			}
			
			if (logicalExpression.getArguments().size() > 0)
				return logicalExpression;
			else
				return null;
		}
	}

	protected String getDisplayValue(String value)
	{
		return value;
	}	
	
	public String getValue(WhereExpression whereExpression) {
		if (whereExpression instanceof BinaryExpression)
		{
		 	BinaryExpression be = (BinaryExpression)whereExpression;
		 	return getDisplayValue(((SimpleOperand)be.getRightOperands().get(0)).getRawValue()); 	
		}
		else if (whereExpression instanceof LogicalExpression)
		{
			String values = "";
			EList arguments = ((LogicalExpression)whereExpression).getArguments();
			
			for (int i = 0; i < (arguments.size() - 1); i++)
			 	values = values + getValue((WhereExpression)arguments.get(i)) + " " + OR_SEPARATOR + " ";
			
		 	values = values + getValue((WhereExpression)arguments.get(arguments.size()-1));		 	
		 	
			return values;
		}
		else
			return null;

	}

	public String getOperator(WhereExpression whereExpression) {
		if (whereExpression instanceof BinaryExpression)
		{
		 	BinaryExpression be = (BinaryExpression)whereExpression;
		 	return getOperatorString(be.getOperator());
		}
		else if (whereExpression instanceof LogicalExpression)
		{
			if (LogicalOperators.NOT_LITERAL.equals(((LogicalExpression)whereExpression).getOperator()))
				return UIPlugin.getResourceString("STR_FILTER_OPERATOR_NOT_LIKE");
			
			EList arguments = ((LogicalExpression)whereExpression).getArguments();
			if (arguments.size() > 0)
				return getOperator((WhereExpression)arguments.get(0));
			else
				return null;
		}
		else
			return null;
	}

	public String defaultValue() {
		return "";
	}
	
	protected String[] getValues(String valuesStr)
	{
		String[] values;
		Vector valuesV = new Vector();
		
		int idx = valuesStr.indexOf(OR_SEPARATOR);
		while (idx >= 0)
		{
			valuesV.add(valuesStr.substring(0, idx).trim());
			valuesStr = valuesStr.substring(idx+1);
			idx = valuesStr.indexOf(OR_SEPARATOR);
		}
		valuesV.add(valuesStr.trim());		
		
		values = new String[valuesV.size()];
		valuesV.toArray(values);
		return values;
	}	
	
	public static RelationalOperators getOperator(String operator)
	{
		if (UIPlugin.getResourceString("STR_FILTER_OPERATOR_LIKE").equals(operator))
			return RelationalOperators.get(RelationalOperators.LIKE);
		else if (UIPlugin.getResourceString("STR_FILTER_OPERATOR_EQ").equals(operator))
			return RelationalOperators.get(RelationalOperators.EQ);			
		else if (UIPlugin.getResourceString("STR_FILTER_OPERATOR_NEQ").equals(operator))
			return RelationalOperators.get(RelationalOperators.NEQ);			
		else if (UIPlugin.getResourceString("STR_FILTER_OPERATOR_GT").equals(operator))
			return RelationalOperators.get(RelationalOperators.GT);			
		else if (UIPlugin.getResourceString("STR_FILTER_OPERATOR_LT").equals(operator))
			return RelationalOperators.get(RelationalOperators.LT);			
		else if (UIPlugin.getResourceString("STR_FILTER_OPERATOR_GE").equals(operator))
			return RelationalOperators.get(RelationalOperators.GE);
		else if (UIPlugin.getResourceString("STR_FILTER_OPERATOR_LE").equals(operator))
			return RelationalOperators.get(RelationalOperators.LE);
		
		return null;
	}
	
	public static String getOperatorString(RelationalOperators operator)
	{
		if (RelationalOperators.get(RelationalOperators.LIKE).equals(operator))
			return UIPlugin.getResourceString("STR_FILTER_OPERATOR_LIKE");
		else if (RelationalOperators.get(RelationalOperators.EQ).equals(operator))
			return UIPlugin.getResourceString("STR_FILTER_OPERATOR_EQ");			
		else if (RelationalOperators.get(RelationalOperators.NEQ).equals(operator))
			return UIPlugin.getResourceString("STR_FILTER_OPERATOR_NEQ");
		else if (RelationalOperators.get(RelationalOperators.GT).equals(operator))
			return UIPlugin.getResourceString("STR_FILTER_OPERATOR_GT");
		else if (RelationalOperators.get(RelationalOperators.LT).equals(operator))
			return UIPlugin.getResourceString("STR_FILTER_OPERATOR_LT");
		else if (RelationalOperators.get(RelationalOperators.GE).equals(operator))
			return UIPlugin.getResourceString("STR_FILTER_OPERATOR_GE");
		else if (RelationalOperators.get(RelationalOperators.LE).equals(operator))
			return UIPlugin.getResourceString("STR_FILTER_OPERATOR_LE");
		
		return null;
	}	
}
