/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfileAttachDelegate.java,v 1.27 2005/04/19 23:24:25 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.ILaunchConfigurationDelegate;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

/**
 * @author znensi
 */
public class ProfileAttachDelegate implements ILaunchConfigurationDelegate
{
	public void launch(ILaunchConfiguration conf, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException {
		try {
			monitor.beginTask("", 4);
			monitor.subTask(UIPlugin.getResourceString("LAUNCHING_MESSAGES_ON_PROGRESS_VIEW"));
			boolean success = ProfileLaunchUtil.performProfilingTypesLaunch(conf);
			monitor.worked(1);
			
			if (!success) {
				monitor.setCanceled(true);
				return;
			}

			IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
			
			String hostName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, "localhost");
			int port = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getInt(HyadesConstants.LOCALHOST_PORT));
			String projectName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_PROJECT, store.getString(TraceConstants.TRACE_PROJECT_NAME));
			String monitorName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_MONITOR, store.getString(TraceConstants.TRACE_MONITOR_NAME));
			String profileFile = null;
			if (conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILE_TO_FILE, false))
				profileFile = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_FILE, (String)null);
				
			ProfilingSetsManager manager = ProfilingSetsManager.instance();
			ArrayList filters = manager.getFilters(conf);
			Vector options = manager.getOptions(conf);
			Vector agents = new Vector(getAgents(conf));
			monitor.worked(1);
			
			/*
			 * #59138: Check if the agents listed in the launch config
			 * are still valid. If not, display an error message.
			 */
			List agentList = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AGENTS, (List)null);
	
			// count the agents
			int agentCount = 0;
			if (agentList != null) {
				Iterator iter = agentList.iterator();
				while (iter.hasNext()) {
					String agentString = (String)iter.next();
					agentCount += new StringTokenizer(agentString, ":,").countTokens() - 1;
				}
			}
			
			if (agentCount != agents.size()) {
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						Shell parent = UIPlugin.getActiveWorkbenchShell();
						IStatus status = new Status(Status.WARNING, UIPlugin.getPluginId(), IResourceStatus.WARNING, UIPlugin.getResourceString("ERROR_DLG_INVALID_AGENTS_REASON"), null);
						ErrorDialog.openError(parent, UIPlugin.getResourceString("TRACE_MSG"), UIPlugin.getResourceString("LAUNCH_ERROR_"), status);
					}
				});
				monitor.setCanceled(true);
				return;
			}
			monitor.worked(1);
			
			PDCoreUtil.attach(hostName, agents, options, filters, projectName, monitorName, String.valueOf(port), profileFile, launch);
			
			Display.getDefault().asyncExec(new Runnable() {
	        	public void run() {
	        		try{
	        			if (UIPlugin.getDefault().getPreferenceStore().getBoolean(TraceConstants.PROFILE_TIPS)) {
		        			MessageDialogWithToggle.openInformation(UIPlugin.getActiveWorkbenchShell(), UIPlugin.getResourceString("PROFILING_TIP_TITLE"), UIPlugin.getResourceString("ATTACH_INFO_MESSAGE"), UIPlugin.getResourceString("ATTACH_MESSAGE_CHECKBOX_MSG"), false, UIPlugin.getDefault().getPreferenceStore(), TraceConstants.PROFILE_TIPS);
	        			}
	        		} catch (Exception exc) {
	        			exc.printStackTrace();
	        		}
	        	}
	        });
			monitor.worked(1);
		}
		catch (CoreException e) {
			monitor.setCanceled(true);
			throw e;
		}
	}
	
	static List getNodeConfiguration(ILaunchConfiguration conf, boolean showErrorMsgs)
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();

		try
		{
			String host = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, "localhost");
			int port = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getDefaultInt(HyadesConstants.LOCALHOST_PORT));
			Enumeration enum = PDCoreUtil.getNodeConfiguration(host, port, showErrorMsgs); 
			if (enum != null)
			{
				List list = new ArrayList();
				
				while (enum.hasMoreElements())
					list.add(enum.nextElement());
				
				return list;
			}
			else
			{
				return null;
			}
			
			
		}
		catch (CoreException e)
		{
			HyadesUIPlugin.logError(e);
			return null;
		}
	}
	
	static List getNodeConfiguration(ILaunchConfiguration conf)
	{
		// if the enum is null, an empty list will be returned
		List list = getNodeConfiguration(conf, true);
		
		if (list == null)
			list = new ArrayList();
		
		return list;
	}
	
	public static List getAttributeAgents(ILaunchConfiguration conf)
	{
		try
		{
			List agents = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AGENTS, (List)null);
			
			if (agents == null)
				return new ArrayList();
			
			Iterator iter = agents.iterator();

			ArrayList list = new ArrayList();			
			
			while (iter.hasNext())
				list.add(new AgentAttribute((String)iter.next()));
			
			return list;
		}
		catch (Exception e)
		{
			HyadesUIPlugin.logError(e);
		}
		
		return new ArrayList();
	}	

	public static List getAgents(ILaunchConfiguration conf) {
		return getAgents(conf, getNodeConfiguration(conf));
	}
	
	static List getAgents(ILaunchConfiguration conf, List nodes)
	{
		List agents = new ArrayList();

		if (nodes != null) {
			try
			{
				Iterator iter = nodes.iterator();
				List list = getAttributeAgents(conf);
				if (list != null)
				{
					Map map = createMap(list);
	
					Vector v = new Vector();
					while (iter.hasNext())
					{
						Process process = (Process)iter.next();
						String[] selected = (String[])map.get(process.getProcessId());
						if (selected != null)
						{
							for (int i = 0; i < selected.length; ++i)
								agents.add(process.getAgent(selected[i]));
						}
					}
				}
			}
			catch (Exception e)
			{
				HyadesUIPlugin.logError(e);
			}
		}

		return agents;
	}

	// parse and organize the agents in a more accessible data structure
	public static Map createMap(List list)
	{
		Map map = new HashMap();
		
		Iterator iter = list.iterator();
		while (iter.hasNext())
		{
			AgentAttribute agentAttribute = (AgentAttribute) iter.next();
			Vector agents = (Vector) map.get(agentAttribute.getPID());

			if (agents == null)
				agents = new Vector();
			
			agents.addElement(agentAttribute.getName());
			map.put(agentAttribute.getPID(), agents);
		}
		
		Iterator keys = map.keySet().iterator();
		String key = null;
		Vector agents = null;		
		String[] stringArray = null; 
		
		while (keys.hasNext())
		{
			key = (String)keys.next();
			agents = (Vector) map.get(key);
			stringArray = new String[agents.size()];
			agents.toArray(stringArray);

			map.put(key, stringArray);
		}
		
		return map;
	}
}

