/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfileJUnitPluginShortcut.java,v 1.7 2005/02/16 22:24:01 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugModelPresentation;
import org.eclipse.debug.ui.IDebugUIConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.internal.junit.launcher.JUnitBaseLaunchConfiguration;
import org.eclipse.jdt.internal.junit.ui.JUnitMessages;
import org.eclipse.jdt.internal.junit.ui.JUnitPlugin;
import org.eclipse.jdt.internal.junit.util.TestSearchEngine;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.jdt.ui.JavaElementLabelProvider;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.pde.internal.ui.launcher.JUnitWorkbenchShortcut;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

public class ProfileJUnitPluginShortcut extends JUnitWorkbenchShortcut {

	protected void launchType(Object[] search, String mode) {
		IType[] types= null;
		try {
			types= TestSearchEngine.findTests(search); 
		} catch (InterruptedException e) {
			JUnitPlugin.log(e);
			return;
		} catch (InvocationTargetException e) {
			JUnitPlugin.log(e);
			return;
		}
		IType type= null;
		if (types.length == 0) {
			MessageDialog.openInformation(getShell(), JUnitMessages.getString("LaunchTestAction.dialog.title"), JUnitMessages.getString("LaunchTestAction.message.notests")); //$NON-NLS-1$ //$NON-NLS-2$
		} else if (types.length > 1) {
			type= chooseType(types, mode);
		} else {
			type= types[0];
		}
		if (type != null) {
			launch(type, mode);
		}
	}

	private void launch(IType type, final String mode) {
		String fullyQualifiedName= type.getFullyQualifiedName();
		ILaunchConfiguration config = findLaunchConfiguration(
			mode, 
			type, 
			"",  //$NON-NLS-1$
			fullyQualifiedName, 
			"" //$NON-NLS-1$
		);
		if (config == null) {
			config= createConfiguration(
				type.getJavaProject(),
				type.getElementName(),
				fullyQualifiedName,
				"", //$NON-NLS-1$
				"" //$NON-NLS-1$
			);
		}
		
		final ILaunchConfiguration conf = config;
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				Shell shell = UIPlugin.getActiveWorkbenchShell();
				int result = DebugUITools.openLaunchConfigurationPropertiesDialog(shell, conf, IDebugUIConstants.ID_PROFILE_LAUNCH_GROUP, new Status(Status.OK, UIPlugin.getPluginId(), RunLaunchProfileStatusHandler.CODE, "", null));
				if (result == Window.OK) {
					DebugUITools.launch(conf, mode);
				}
			}
		});
	}

	/*
	 * @see org.eclipse.jdt.internal.junit.launcher.JUnitLaunchShortcut#launchConfiguration(java.lang.String, org.eclipse.debug.core.ILaunchConfiguration)
	 */
	protected void launchConfiguration(final String mode, final ILaunchConfiguration conf) {
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				Shell shell = UIPlugin.getActiveWorkbenchShell();
				int result = DebugUITools.openLaunchConfigurationPropertiesDialog(shell, conf, IDebugUIConstants.ID_PROFILE_LAUNCH_GROUP, new Status(Status.OK, UIPlugin.getPluginId(), RunLaunchProfileStatusHandler.CODE, "", null));
				if (result == Window.OK) {
					DebugUITools.launch(conf, mode);
				}
			}
		});
	}
	
	private ILaunchConfiguration findLaunchConfiguration(String mode, IJavaElement element, String container, String testClass, String testName) {
		ILaunchConfigurationType configType= getJUnitLaunchConfigType();
		List candidateConfigs= Collections.EMPTY_LIST;
		try {
			ILaunchConfiguration[] configs= getLaunchManager().getLaunchConfigurations(configType);
			candidateConfigs= new ArrayList(configs.length);
			for (int i= 0; i < configs.length; i++) {
				ILaunchConfiguration config= configs[i];
				if ((config.getAttribute(JUnitBaseLaunchConfiguration.LAUNCH_CONTAINER_ATTR, "").equals(container)) && //$NON-NLS-1$
					(config.getAttribute(IJavaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME, "").equals(testClass)) && //$NON-NLS-1$
					(config.getAttribute(JUnitBaseLaunchConfiguration.TESTNAME_ATTR,"").equals(testName)) &&   //$NON-NLS-1$
					(config.getAttribute(IJavaLaunchConfigurationConstants.ATTR_PROJECT_NAME, "").equals(element.getJavaProject().getElementName()))) {  //$NON-NLS-1$
						candidateConfigs.add(config);
				}
			}
		} catch (CoreException e) {
			JUnitPlugin.log(e);
		}
		
		// If there are no existing configs associated with the IType, create one.
		// If there is exactly one config associated with the IType, return it.
		// Otherwise, if there is more than one config associated with the IType, prompt the
		// user to choose one.
		int candidateCount= candidateConfigs.size();
		if (candidateCount < 1) {
			return null;
		} else if (candidateCount == 1) {
			return (ILaunchConfiguration) candidateConfigs.get(0);
		} else {
			// Prompt the user to choose a config.  A null result means the user
			// cancelled the dialog, in which case this method returns null,
			// since cancelling the dialog should also cancel launching anything.
			ILaunchConfiguration config= chooseConfiguration(candidateConfigs, mode);
			if (config != null) {
				return config;
			}
		}
		return null;
	}

	/**
	 * Prompts the user to select a type
	 * 
	 * @return the selected type or <code>null</code> if none.
	 */
	protected IType chooseType(IType[] types, String mode) {
		if (mode.equals(ILaunchManager.PROFILE_MODE)) {
			ElementListSelectionDialog dialog= new ElementListSelectionDialog(getShell(), new JavaElementLabelProvider(JavaElementLabelProvider.SHOW_POST_QUALIFIED));
			dialog.setElements(types);
			dialog.setTitle(UIPlugin.getResourceString("LAUNCH_SHORTCUT_SELECT_TEST_TITLE")); //$NON-NLS-1$
			dialog.setMessage(UIPlugin.getResourceString("LAUNCH_SHORTCUT_SELECT_TEST_MSG")); //$NON-NLS-1$
			dialog.setMultipleSelection(false);
			if (dialog.open() == Window.OK) {
				return (IType)dialog.getFirstResult();
			}
			return null;
		}
		else {
			return super.chooseType(types, mode);
		}
	}

	/**
	 * Show a selection dialog that allows the user to choose one of the specified
	 * launch configurations.  Return the chosen config, or <code>null</code> if the
	 * user cancelled the dialog.
	 */
	protected ILaunchConfiguration chooseConfiguration(List configList, String mode) {
		if (mode.equals(ILaunchManager.PROFILE_MODE)) {
			IDebugModelPresentation labelProvider = DebugUITools.newDebugModelPresentation();
			ElementListSelectionDialog dialog= new ElementListSelectionDialog(getShell(), labelProvider);
			dialog.setElements(configList.toArray());
			dialog.setTitle(UIPlugin.getResourceString("LAUNCH_SHORTCUT_SELECT_CONF_TITLE")); //$NON-NLS-1$
			dialog.setMessage(UIPlugin.getResourceString("LAUNCH_SHORTCUT_SELECT_CONF_MSG")); //$NON-NLS-1$
			dialog.setMultipleSelection(false);
			int result= dialog.open();
			labelProvider.dispose();
			if (result == ElementListSelectionDialog.OK) {
				return (ILaunchConfiguration)dialog.getFirstResult();
			}
			return null;
		}
		else {
			return super.chooseConfiguration(configList, mode);
		}
	}
}
