/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfileLaunchUtil.java,v 1.4 2005/05/06 16:26:41 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSet;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.IProfilingType;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;

/**
 * Utility methods shared by the profile launchers.
 */
public class ProfileLaunchUtil {
	
	private static final String[] EMPTY = new String[0];
	
	/**
	 * Retrieves the environment variables from the given launch
	 * configuration, and returns them in the form of Strings
	 * in the form "name=value".
	 */
	public static String[] getEnvironmentVariables(ILaunchConfiguration conf) throws CoreException {
		Map envMap = conf.getAttribute(ILaunchManager.ATTR_ENVIRONMENT_VARIABLES, (Map)null);
		if (envMap != null) {
			List list = new ArrayList();
			Iterator iter = envMap.keySet().iterator();
			while (iter.hasNext()) {
				String key = (String)iter.next();
				list.add(key + "=" + envMap.get(key));
			}
			String[] array = new String[list.size()];
			list.toArray(array);
			return array;
		}
		else {
			return EMPTY;
		}
	}
	
	/**
	 * Notifies this configuration's profiling types that the
	 * launch is about to begin.
	 */
	public static boolean performProfilingTypesLaunch(ILaunchConfiguration config) {
		boolean success = true;

		ProfilingSetsManager manager = ProfilingSetsManager.instance();
		IProfilingSet set = null;
		try {
			String setID = config.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, (String)null);
			set = (IProfilingSet)manager.getProfilingSets().get(setID);
		}
		catch (CoreException e) {
			HyadesUIPlugin.logError(e);
		}

		if (set == null) {
			return true;
		}

		IProfilingSetType[] profilingTypes = new ProfilingSetType[manager.getProfilingTypes().size()];
		profilingTypes = (IProfilingSetType[]) manager.getProfilingTypes().values().toArray(profilingTypes);

		List types = set.getProfilingTypes();
		for (int i = 0; i < profilingTypes.length; i++) {
			if (types.contains(profilingTypes[i].getId())) {
				IProfilingType type = profilingTypes[i].getProfilingType();
				final String msg = type.launch(config);
				if (msg != null) {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							MessageDialog.openError(Display.getDefault().getActiveShell(), UIPlugin.getResourceString("ERROR_DLG_CONFLICTS_TITLE"), msg);
						}
					});
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Converts the given array into a delimited String with
	 * the given delimiter character.
	 */
	public static String convertToDelimitedString(String[] array, char delimiter) {
		return convertToDelimitedString(array, delimiter, true);
	}	
	
	/**
	 * Converts the given array into a delimited String with
	 * the given delimiter character.
	 */
	public static String convertToDelimitedString(String[] array, char delimiter, boolean addQuotes) {
		StringBuffer buf = new StringBuffer();
		if (array.length > 0) {
			for (int i=0;i<array.length;++i) {
				if (i != 0) {
					buf.append(delimiter);
				}

				String arg = array[i].trim();
				if (array[i].startsWith("\"") && array[i].endsWith("\"")) {
					buf.append(array[i]);
				}
				else if (!addQuotes || array[i].indexOf(' ') == -1) {
					buf.append(array[i]);
				}
				else {
					buf.append('"');
					buf.append(array[i]);
					buf.append('"');
				}
			}
		}
		return buf.toString();
	}
}
