/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.trace.ui.internal.navigator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;


/**
 * A content provider wrapper that processes the output of that content provider
 * and filters/skips elements. Filtering means removing the element and all its
 * ancestors, and skipping means filtering only the immediate element, but its
 * children will still be shown.
 * 
 * @author Curtis d'Entremont
 * @since 3.3
 */
public abstract class AbstractFilteredContentProvider implements ITreeContentProvider {

	private ITreeContentProvider _content;
	
	/**
	 * Constructs a new wrapper for the given content provider.
	 *  
	 * @param content The underlying content provider that has all the unfiltered content.
	 */
	public AbstractFilteredContentProvider(ITreeContentProvider content) {
		_content = content;
	}
	
	/**
	 * Disposes the content provider.
	 * 
	 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
	 */
	public void dispose() {
		_content.dispose();
	}
	
	/**
	 * Returns the filtered list of children.
	 * 
	 * @param parentElement The parent that contains the children.
	 * @return The children, filtered.
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object parentElement) {
		return filter(_content.getChildren(parentElement));
	}
	
	/**
	 * Returns the filtered list of top-level elements.
	 * 
	 * @param inputElement The input of the content provider.
	 * @return The filtered top-level elements.
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement) {
		return filter(_content.getElements(inputElement));
	}
	
	/**
	 * Returns the parent of the given element. If the parent was skipped,
	 * returns the closest ancestor that was not skipped.
	 * 
	 * @param element The element whose parent to get.
	 * @return The given element's parent.
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
	 */
	public Object getParent(Object element) {
		Object parent = _content.getParent(element);
		if (isSkipped(parent)) {
			return getParent(parent);
		}
		return parent;
	}
	
	/**
	 * Returns whether or not the given element has any children. This
	 * overrides the underlying implementation because it does not know about
	 * the filtering, and we do not know about its internal details. So the
	 * only thing we can do is query for all the children, pass them through
	 * the filter, and see if there are any left.
	 * 
	 * @param element The element being queried for children.
	 * @return Whether or not the given element has any children.
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
	 */
	public boolean hasChildren(Object element) {
		Object[] children = getChildren(element);
		return (children != null && children.length > 0);
	}
	
	/**
	 * A signal sent when the content provider's input changes.
	 * 
	 * @param viewer The viewer that has this content provider.
	 * @param oldInput The previous input.
	 * @param newInput The new input.
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		_content.inputChanged(viewer, oldInput, newInput);
	}
	
	/**
	 * Returns whether or not the given element and all its descendants should
	 * be filtered out entirely.
	 * 
	 * @param element The element to check.
	 * @return Whether or not to filter the given element and its descendants.
	 */
	public abstract boolean isFiltered(Object element);
	
	/**
	 * Returns whether or not to skip the given element. This has no effect on
	 * the children, and simply "skips" the given node in the tree, so that its
	 * children (if not skipped) will appear in its place.
	 * 
	 * @param element The element to check.
	 * @return Whether or not to skip the given element.
	 */
	public abstract boolean isSkipped(Object element);
	
	/**
	 * Filters the appropriate notes from the given list of elements. By
	 * filtering, it returns the elements' children.
	 * 
	 * @param elements The original list of unfiltered elements.
	 * @return The filtered list.
	 */
	private Object[] filter(Object[] elements) {
		List list = new ArrayList();
		for (int i=0;i<elements.length;++i) {
			if (!isFiltered(elements[i])) {
				if (isSkipped(elements[i])) {
					Object[] children = getChildren(elements[i]);
					if (children != null && children.length > 0) {
						list.addAll(Arrays.asList(children));
					}
				}
				else {
					list.add(elements[i]);
				}
			}
		}
		return list.toArray();
	}
}
