/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CTree.java,v 1.10 2005/04/19 23:24:25 ewchan Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.lang.reflect.InvocationTargetException;
import java.util.Vector;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.DeferredProcessAdapter;
import org.eclipse.hyades.trace.ui.internal.launcher.ProcessWrapperRunnable;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.model.IWorkbenchAdapter;
import org.eclipse.ui.progress.DeferredTreeContentManager;
import org.eclipse.ui.progress.WorkbenchJob;

public class CTree {

	private Label _label;

	private Tree _tree;
	
	private Listener _uiListener;
	
	private Composite groupWidget;

	private TreeViewer _viewer;

	private CTreeContentProvider _contentProvider;

	private CTreeLabelProvider _labelProvider;

	private final static Object[] EMPTY_ARRAY = new Object[0];
	
	private ProfilingAgentList _agentList;

	public class CTreeContentProvider implements ITreeContentProvider {

		private ILaunchConfiguration _conf;

		private DeferredTreeContentManager tManager;

		public void setLaunchConfiguration(ILaunchConfiguration conf) {
			_conf = conf;
		}

		public ILaunchConfiguration getLaunchConfiguration() {
			return _conf;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
		 */
		public Object[] getChildren(Object parentElement) {
			if (parentElement instanceof ProcessTreeItem) {
				ProcessTreeItem pti = (ProcessTreeItem) parentElement;

				if (pti.getData() instanceof Process && _agentList != null)
				{
					return getChildrenInTree(pti);
				}
				else
				{
					if (tManager != null) {
	
						pti.setAdapter(new DeferredProcessAdapter(this, pti,
								getTreeViewer()));
						Object[] children = tManager.getChildren(pti);
						if (children != null)
							return children;
					}
					return fetchChildren(pti);
				}
			} else if (parentElement instanceof ProfilingAgentList) {
				return ((ProfilingAgentList) parentElement).getProcessList();
			}

			return EMPTY_ARRAY;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
		 */
		public Object getParent(Object element) {
			if (element instanceof ProcessTreeItem)
			{
				return ((ProcessTreeItem)element).getParent();
			}
			
			return null;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
		 */
		public boolean hasChildren(Object element) {

			if (element instanceof ProcessTreeItem) {
				ProcessTreeItem pti = (ProcessTreeItem) element;

				if (pti.getData() == null || pti.getData() instanceof Process)
					return true;
				else
					return false;
			} else if (element instanceof ProfilingAgentList) {
				return ((ProfilingAgentList) element).getAgentCount() > 0;
			}

			return false; // the "Update ..." placeholder has no children
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		 */
		public Object[] getElements(Object inputElement) {
			return getChildren(inputElement);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose() {
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer,
		 *      java.lang.Object, java.lang.Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			if (viewer instanceof AbstractTreeViewer) {
				tManager = new DeferredTreeContentManager(this,
						(AbstractTreeViewer) viewer);
			}
		}

		public Object[] fetchChildren(Object parent) {
			IRunnableContext context = UIPlugin.getActiveWorkbenchWindow();
			ProcessWrapperRunnable runnable = new ProcessWrapperRunnable(
					parent, _conf);
			try {
				context.run(true, true, runnable);
			} catch (InvocationTargetException e) {
				e.printStackTrace();
				return EMPTY_ARRAY;
			} catch (InterruptedException e) {
				e.printStackTrace();
				return EMPTY_ARRAY;
			}

			return runnable.getChildren();
		}
		
		public void updateTrees()
		{
			
			WorkbenchJob clearJob = new WorkbenchJob("CTree.updateTrees") {//$NON-NLS-1$
				/*
				 * (non-Javadoc)
				 * 
				 * @see org.eclipse.ui.progress.UIJob#runInUIThread(org.eclipse.core.runtime.IProgressMonitor)
				 */
				public IStatus runInUIThread(IProgressMonitor monitor) {
					_uiListener.handleEvent(null);
					return Status.OK_STATUS;
				}
			};
			clearJob.setSystem(true);
			clearJob.schedule();
		}
	}

	public class CTreeLabelProvider extends LabelProvider {

		public String getText(Object element) {
			if (element instanceof ProcessTreeItem)
				return ((ProcessTreeItem) element).getLabelText();
			else if (isPendingUpdate(element))
				return UIPlugin.getResourceString("STR_PENDING_ITEM");

			return "";
		}

		private boolean isPendingUpdate(Object element) {
			return element instanceof IWorkbenchAdapter;
		}

		public Image getImage(Object element) {
			if (element instanceof ProcessTreeItem)
				return ((ProcessTreeItem) element).getImage();

			return super.getImage(element);
		}
	}

	public CTree(Composite parent, String text) {
		// Create the grid.
		groupWidget = new Composite(parent, SWT.NULL);

		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		groupWidget.setLayout(layout);
		GridData data = GridUtil.createFill();
		data.heightHint = 200;
		data.widthHint = 250;
		groupWidget.setLayoutData(data);

		_label = new Label(groupWidget, SWT.NULL);
		_label.setText(text);
		_tree = new Tree(groupWidget, SWT.MULTI | SWT.BORDER | SWT.H_SCROLL
				| SWT.V_SCROLL);

		_viewer = new TreeViewer(_tree);

		_viewer.setContentProvider(getContentProvider());
		_viewer.setLabelProvider(getLabelProvider());

		_tree.setLayoutData(GridUtil.createFill());
		
		_agentList = null;
	}

	protected IContentProvider getContentProvider() {
		if (_contentProvider == null)
			_contentProvider = new CTreeContentProvider();
		return _contentProvider;
	}

	protected ILabelProvider getLabelProvider() {
		if (_labelProvider == null)
			_labelProvider = new CTreeLabelProvider();
		return _labelProvider;
	}

	protected TreeViewer getTreeViewer() {
		return _viewer;
	}

	public void initializeTree(ILaunchConfiguration conf) {
		_contentProvider.setLaunchConfiguration(conf);
		_agentList = null;
		getTreeViewer().setInput(new ProcessTreeItem(null, null));
	}

	public void resetTree(ProfilingAgentList agentList) {
		_agentList = agentList;
		getTreeViewer().setInput(agentList);
	}

	private Tree getTree() {
		return getTreeViewer().getTree();
	}

	public void setRedraw(boolean redraw) {
		getTree().setRedraw(redraw);
	}

	public void enable(boolean enable) {
		getTree().setEnabled(enable);
		_label.setEnabled(enable);
	}

	public Object[] getSelectedItems() {
		Vector selectedItemsV = new Vector();
		
		ISelection iSelection = getTreeViewer().getSelection();
		
		if (!(iSelection instanceof StructuredSelection))
			return new Object[0];
		
		Object [] selection = ((StructuredSelection)iSelection).toArray();
		ProcessTreeItem pti;
		
		Object[] items = getItems();
		
		for (int i = 0; i < items.length; i++)
		{
			if (items[i] instanceof ProcessTreeItem)
			{
				if (isItemSelected((ProcessTreeItem)items[i], selection))
					selectedItemsV.addElement(items[i]);
			}
		}

		return selectedItemsV.toArray();
	}
	
	private boolean isItemSelected(ProcessTreeItem pti, Object[] selection)
	{
		for (int i = 0; i < selection.length; i++)
		{
			if (pti == selection[i])
				return true;
			else if (pti.getParent() == selection[i])
				return true;
		}
		return false;
	}

	public Object[] getItems() {
		TreeItem []items =  getTree().getItems();
		
		Vector itemsV = new Vector();
		
		for(int i = 0; i < items.length; i++)
		{
			if (items[i].getData() instanceof ProcessTreeItem)
			{
				if (((ProcessTreeItem)items[i].getData()).getData() instanceof Process)
				{
					Object[] agentItems = getChildrenInTree((ProcessTreeItem)items[i].getData());
					for (int j = 0; j < agentItems.length; j++)
						itemsV.addElement(agentItems[j]);
				}
				else if (((ProcessTreeItem)items[i].getData()).getData() instanceof ErrorItem)
					itemsV.addElement(items[i].getData());
			}
			else
			{
				itemsV.addElement(items[i].getData());
			}

		}
		return itemsV.toArray();
	}

	public void clearSelection() {
		getTree().deselectAll();
	}

	public void setSelectionFromData(Object[] data) {
		StructuredSelection sSelection = new StructuredSelection(data);

		getTreeViewer().setSelection(sSelection);
	}

	public void addSelectionListener(SelectionListener listener) {
		getTree().addSelectionListener(listener);
	}

	public void addPaintListener(PaintListener listener) {
		getTree().addPaintListener(listener);
	}

	public boolean hasWidget(Widget widget) {
		return getTree() == widget;
	}

	public int getItemCount() {
		return getItems().length;
	}
	
	public void setUIListener(Listener uiListener)
	{
		_uiListener = uiListener;
	}
	
	private Object[] getChildrenInTree(ProcessTreeItem parent)
	{
		if (_agentList == null)
			return parent.getChildren();
		
		Vector childrenV = new Vector();
		
		ProcessTreeItem[] ptiItems = parent.getChildren();
		
		for (int i = 0; i < ptiItems.length; i++)
		{
			if (_agentList.agentInList(ptiItems[i]))
				childrenV.addElement(ptiItems[i]);
		}
		
		return childrenV.toArray();
	}
}