/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DoubleCTree.java,v 1.12 2005/04/19 23:24:25 ewchan Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.util.ArrayList;
import java.util.Vector;

import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.AgentAttribute;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfileAttachDelegate;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.ui.model.IWorkbenchAdapter;

public class DoubleCTree extends AbstractChangeable {

	Composite _group;

	Composite groupWidget;

	private CTree sourceList;

	private CTree targetList;

	private ProfilingAgentList sourceAgentList;

	private ProfilingAgentList targetAgentList;

	private boolean _needToResetAgentLists = false;

	private boolean _treesLoaded = false;

	public Label targetListLabel;

	public Label sourceListLabel;

	public Button add;

	public Button remove;

	public Button addAll;

	public Button removeAll;

	private WizardPage _wizardPage;

	private ILaunchConfiguration _conf;

	class TreeUpdatedListener implements Listener
	{

		/* (non-Javadoc)
		 * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
		 */
		public void handleEvent(Event event) {
			resetAgentLists();
		}
	}

	class TreeSelListener implements SelectionListener {
		public void widgetSelected(SelectionEvent e) {
			sourceAgentList.initializeFromAgents(sourceList.getItems());

			//allow only one list to have selection
			if (sourceList.hasWidget(e.widget)
					&& sourceList.getSelectedItems().length != 0)
				targetList.clearSelection();
			else if (targetList.hasWidget(e.widget)
					&& targetList.getSelectedItems().length != 0)
				sourceList.clearSelection();

			changed();
		}

		public void widgetDefaultSelected(SelectionEvent e) {
			sourceAgentList.initializeFromAgents(sourceList.getItems());

			if (sourceList.hasWidget(e.widget)) {
				Object[] items = sourceList.getSelectedItems();
				if (items.length > 0) {
					addAgentsToTargetList(items);
					changed();
				}
			} else if (targetList.hasWidget(e.widget)) {
				Object[] items = targetList.getSelectedItems();
				if (items.length > 0) {
					removeAgentsFromTargetList(items);
					changed();
				}
			}
		}
	}

	class ButtonListener implements Listener {

		public void handleEvent(Event e) {
			sourceAgentList.initializeFromAgents(sourceList.getItems());

			if (e.widget == add) {
				addAgentsToTargetList(sourceList.getSelectedItems());
			} else if (e.widget == addAll) {
				addAgentsToTargetList(sourceList.getItems());
			} else if (e.widget == remove) {
				removeAgentsFromTargetList(targetList.getSelectedItems());
			} else if (e.widget == removeAll) {
				removeAgentsFromTargetList(targetList.getItems());
			}

			changed();
		}
	}
	

	private boolean isRetrievingList()
	{
		Object[] items = sourceList.getItems();
		
		if (items.length == 0)
			return false;
		
		if (items[0] instanceof IWorkbenchAdapter)
			return true;	
		
		return false;
	}

	protected void addAgentsToTargetList(Object[] items) {

		if (isRetrievingList())
			return;
		
		
		if (sourceList.getItemCount() > 0)
		{
			ProcessTreeItem pti = (ProcessTreeItem)sourceList.getItems()[0];
			if (pti.getData() instanceof ErrorItem)
				return;
		}		
		
		targetList.setRedraw(false);
		sourceList.setRedraw(false);

		Vector dataToBeSelected = new Vector();

		for (int i = 0; i < items.length; i++) {
			if (!vaildItemtoAdd(items[i]))
				continue;

			dataToBeSelected.addElement(items[i]);
			targetAgentList.addAgent((ProcessTreeItem)items[i]);
			sourceAgentList.removeAgent((ProcessTreeItem)items[i]);
		}

		sourceList.resetTree(sourceAgentList);
		targetList.resetTree(targetAgentList);

		sourceList.clearSelection();

		if (dataToBeSelected.size() > 0)
			targetList.setSelectionFromData(dataToBeSelected.toArray());

		targetList.setRedraw(true);
		sourceList.setRedraw(true);

		if ((_wizardPage != null) && getSelectedItems().length > 0)
			_wizardPage.setPageComplete(true);
	}

	protected void removeAgentsFromTargetList(Object[] items) {
		if (isRetrievingList())
			return;
		
		targetList.setRedraw(false);
		sourceList.setRedraw(false);

		for (int i = 0; i < items.length; i++) {
			sourceAgentList.addAgent((ProcessTreeItem)items[i]);
			targetAgentList.removeAgent((ProcessTreeItem)items[i]);			
		}

		targetList.clearSelection();
		if ((_wizardPage != null) && getSelectedItems().length > 0)
			_wizardPage.setPageComplete(true);

		sourceList.resetTree(sourceAgentList);
		targetList.resetTree(targetAgentList);

		targetList.setRedraw(true);
		sourceList.setRedraw(true);

	}

	private boolean vaildItemtoAdd(Object item) {
		if (item == null || item instanceof IWorkbenchAdapter)
			return false;
		else
			return true;
	}

	public Object[] getTargetItems() {
		return targetList.getItems();
	}

	public void removeAll() {
		sourceAgentList.initializeFromAgents(new Object[0]);
		targetAgentList.initializeFromAgents(new Object[0]);

		sourceList.setRedraw(false);
		targetList.setRedraw(false);

		sourceList.resetTree(sourceAgentList);
		targetList.resetTree(targetAgentList);

		sourceList.setRedraw(true);
		targetList.setRedraw(true);

	}

	public void setWizardPage(WizardPage wizardPage) {
		_wizardPage = wizardPage;
	}

	public DoubleCTree(Composite parent, int style, String groupText,
			String sourceTextID, String targetTextID) {
		_treesLoaded = false;

		sourceAgentList = new ProfilingAgentList();
		targetAgentList = new ProfilingAgentList();

		//_group = new Group(parent, style);
		_group = new Composite(parent, style);
		//_group.setText(groupText);

		GridLayout layout = new GridLayout();
		layout.verticalSpacing = -10;
		layout.horizontalSpacing = 0;
		_group.setLayout(layout);
		_group.setLayoutData(GridUtil.createFill());

		createDoubleTree(_group, sourceTextID, targetTextID);

		//add listeners
		add.addListener(SWT.Selection, new ButtonListener());
		remove.addListener(SWT.Selection, new ButtonListener());
		addAll.addListener(SWT.Selection, new ButtonListener());
		removeAll.addListener(SWT.Selection, new ButtonListener());
		//double-click action
		sourceList.addSelectionListener(new TreeSelListener());
		targetList.addSelectionListener(new TreeSelListener());
		///////////////////
	}

	/**
	 * Create double list components
	 */
	public void createDoubleTree(Composite parent, String sourceTextID,
			String targetTextID) {
		GridData spec;
		_conf = null;

		// Create the grid.
		groupWidget = parent;

		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		groupWidget.setLayout(layout);

		sourceList = new CTree(groupWidget, sourceTextID);

		// Create the grid.
		Composite buttonWidget = new Composite(groupWidget, SWT.NULL);
		GridLayout blayout = new GridLayout();
		blayout.numColumns = 1;
		buttonWidget.setLayout(blayout);

		Label label = new Label(buttonWidget, SWT.PUSH);
		label.setText("");
		spec = new GridData();
		spec.grabExcessVerticalSpace = true;
		spec.verticalAlignment = GridData.FILL;
		label.setLayoutData(spec);

		add = new Button(buttonWidget, SWT.PUSH);
		add.setText(UIPlugin.getResourceString("LEFT_TO_RIGHT"));
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		add.setLayoutData(spec);

		remove = new Button(buttonWidget, SWT.PUSH);
		remove.setText(UIPlugin.getResourceString("RIGHT_TO_LEFT"));
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		remove.setLayoutData(spec);

		new Label(buttonWidget, SWT.NONE);

		addAll = new Button(buttonWidget, SWT.PUSH);
		addAll.setText(UIPlugin.getResourceString("LEFT_ALL_TO_RIGHT"));
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		addAll.setLayoutData(spec);

		removeAll = new Button(buttonWidget, SWT.PUSH);
		removeAll.setText(UIPlugin.getResourceString("RIGHT_ALL_TO_LEFT"));
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		removeAll.setLayoutData(spec);

		Label label1 = new Label(buttonWidget, SWT.PUSH);
		label1.setText("");
		spec = new GridData();
		spec.grabExcessVerticalSpace = true;
		spec.verticalAlignment = GridData.FILL;
		label.setLayoutData(spec);

		targetList = new CTree(groupWidget, targetTextID);
		
		sourceList.setUIListener(new TreeUpdatedListener());
		targetList.setUIListener(new TreeUpdatedListener());
	}

	/**
	 * Insert the method's description here. Creation date: (11/08/2000 2:31:59
	 * PM)
	 * 
	 * @return com.ibm.swt.widgets.Control
	 */
	public Control getControl() {
		return _group;
	}

	public Object[] getSelectedItems() {
		return targetList.getSelectedItems();
	}

	public Object[] getSelectedItemsForProfiling() {
		return targetList.getItems();
	}	

	public Composite getWidget() {
		return groupWidget;
	}

	public void initializeLaunchConfiguration(ILaunchConfiguration conf) {
		_conf = conf;
	}

	public void initialize() {
		sourceList.initializeTree(_conf);

		sourceAgentList.initializeFromAgents(sourceList.getItems());
		targetAgentList.initializeFromAgents(targetList.getItems());
		_needToResetAgentLists = true;
	}

	public java.util.List getAttributeAgents() {
		if (_conf != null) {
			try {
				return ProfileAttachDelegate.getAttributeAgents(_conf);
			} catch (Exception e) {
				
			}
		}

		return new ArrayList();
	}
	
	public static java.util.List getAttributeAgents(Object[] agentItems)
	{
		java.util.List list = new ArrayList();
		
		for (int i = 0; i < agentItems.length; ++i)
		{
			ProcessTreeItem pti = (ProcessTreeItem) agentItems[i];
			String pid = pti.getProcessId();
			
			if (pid != null)
				list.add(new AgentAttribute(pid, (Agent) pti.getData()));
		}		
		
		return list;
	}


	private AgentAttribute[] getTargetAgentList() {
		java.util.List attrAgList;
		
		if (treesLoaded())
			attrAgList = getAttributeAgents(getTargetItems());
		else
			attrAgList = getAttributeAgents();
		
		AgentAttribute[] attrAgents = new AgentAttribute[attrAgList.size()];
		attrAgList.toArray(attrAgents);
		
		return attrAgents;
	}

	private AgentAttribute[] getTargetSelectedAgentList() {
		java.util.List attrAgList;		
		
		if (treesLoaded()) {
			attrAgList = getAttributeAgents(getSelectedItems());
		} else
			attrAgList = getAttributeAgents();
		
		AgentAttribute[] attrAgents = new AgentAttribute[attrAgList.size()];
		attrAgList.toArray(attrAgents);
		
		return attrAgents;
	}

	public boolean treesLoaded() {
		return _treesLoaded;
	}
	
	public void treesLoaded(boolean treesLoaded) {
		_treesLoaded = treesLoaded;
	}

	public void resetAgentLists() {
		if (getControl().isDisposed())
			return;
		
		if (_needToResetAgentLists
				&& (sourceList.getItemCount() == 0 || !isRetrievingList())) {
			
			_needToResetAgentLists = false;			
			

			sourceAgentList.initializeFromAgents(sourceList.getItems());
			
			if (sourceList.getItemCount() > 0)
			{
				ProcessTreeItem pti = (ProcessTreeItem)sourceList.getItems()[0];
				if (pti.getData() instanceof ErrorItem)
				{
					targetAgentList.initializeFromAgents(new Object[0]);
					targetList.resetTree(targetAgentList);
					return;
				}
			}

			AgentAttribute[] targetAgentAttributeList = getTargetAgentList();
			AgentAttribute[] selectedTargetAgentAttributeList = getTargetSelectedAgentList();

			if (targetAgentAttributeList.length > 0) {
				//EC: refresh button was clicked with selected agent(s)
				//EC: this need special care as refresh button shouldn't
				// 'reset' the 2 list, but update it instead.

				Vector agentsInTarget = new Vector();
				Vector dataToBeSelected = new Vector();

				for (int i = 0; i < targetAgentAttributeList.length; i++) {
					ProcessTreeItem agentFromSource = sourceAgentList.getAgentForAgentAttribute(targetAgentAttributeList[i]);
							
					if (agentFromSource != null) {
						agentsInTarget.addElement(agentFromSource);
						sourceAgentList.removeAgent((ProcessTreeItem)agentFromSource);

						if (agentAttributeListContains(selectedTargetAgentAttributeList, agentFromSource))
							dataToBeSelected.addElement(agentFromSource);
					}
				}

				targetAgentList.initializeFromAgents(agentsInTarget.toArray());

				sourceList.setRedraw(false);
				targetList.setRedraw(false);

				sourceList.resetTree(sourceAgentList);
				targetList.resetTree(targetAgentList);

				if (dataToBeSelected.size() > 0)
					targetList.setSelectionFromData(dataToBeSelected.toArray());

				sourceList.setRedraw(true);
				targetList.setRedraw(true);
			}
			else
			{
				targetAgentList.initializeFromAgents(new Object[0]);
				targetList.resetTree(targetAgentList);				
			}

			_treesLoaded = true;
			changed();
		}
	}
	
	private boolean agentAttributeListContains(AgentAttribute[] attributes, ProcessTreeItem pti)
	{
		String pid = pti.getProcessId();
		String name = ((Agent)pti.getData()).getName();
		String type = ((Agent)pti.getData()).getType();
		
		for (int i = 0; i < attributes.length; i++)
		{
			if (attributes[i].getPID().equals(pid)
				 &&	attributes[i].getName().equals(name)
				 && attributes[i].getType().equals(type))
				return true;
			
		}
		
		return false;
	}
}