/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProcessAdapter.java,v 1.11 2005/04/29 19:43:31 ewchan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.IStreamListener;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.debug.core.model.IStreamMonitor;
import org.eclipse.debug.core.model.IStreamsProxy;
import org.eclipse.debug.core.model.ITerminate;
import org.eclipse.hyades.execution.core.IDataProcessor;
import org.eclipse.hyades.internal.execution.local.common.Console;
import org.eclipse.hyades.internal.execution.local.common.DataProcessor;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessListener;

/**
 * An adapter from <code>Process</code> (Hyades) to <code>IProcess</code>
 * (Eclipse). Used to add a process to an <code>ILaunch</code> by the launch
 * delegate, in order to provide I/O to the console view, get the process's
 * status (running, terminated), and to terminate the process.
 */
public class ProcessAdapter implements IProcess {
	private Map _attributes = new HashMap();
	private boolean _isAlive = false;
	private String _label = null;
	private Process _process;
	private IOProxy _proxy;
	private ILaunch _launch;
	protected int launchMode = 1;

	/**
	 * Creates an adapter using the given <code>Process</code> and <code>
	 * ILaunch</code>. Also sets the whether the process is currently
	 * running and its launch mode.
	 */
	public ProcessAdapter(Process process, ILaunch launch, boolean isAlive, int pLaunchMode) {
		this (process, launch, isAlive);
		
		launchMode = pLaunchMode;
	}	
	
	/**
	 * Creates an adapter using the given <code>Process</code> and <code>
	 * ILaunch</code>.
	 */
	public ProcessAdapter(Process process, ILaunch launch) {
		DateFormat format = DateFormat.getDateTimeInstance(DateFormat.LONG,DateFormat.MEDIUM,Locale.getDefault());
		_label = process.getExecutable() + " (" + format.format(new Date(System.currentTimeMillis())) + ")"; //$NON-NLS-1$
		_process = process;
		_launch = launch;
		_process.addProcessListener(new ProcessListener() {
			public void processLaunched(Process p) {
				_isAlive = true;
			}
			public void processExited(Process p) {
				_isAlive = false;
			}
		});
	}

	/**
	 * Creates an adapter using the given <code>Process</code> and <code>
	 * ILaunch</code>. Also sets the whether the process is currently
	 * running.
	 */
	public ProcessAdapter(Process process, ILaunch launch, boolean isAlive) {
		this(process, launch);
		_isAlive = isAlive;
	}
	
	/**
	 * @see IAdaptable#getAdapter()
	 */
	public Object getAdapter(Class adapter) {
		if (DataProcessor.class.equals(adapter))
			return _proxy;
		return null;
	}
	/**
	 * @see IProcess#getAttribute()
	 */
	public String getAttribute(String key) {
		return (String) _attributes.get(key);
	}
	/**
	 * @see IProcess#setAttribute()
	 */
	public void setAttribute(String key, String value) {
		_attributes.put(key, value);
	}
	/**
	 * @see IProcess#getExitValue()
	 */
	public int getExitValue() throws DebugException {
		return 0;
	}
	/**
	 * @see IProcess#getLabel()
	 */
	public String getLabel() {
		return _label;
	}
	/**
	 * @see IProcess#getLaunch()
	 */
	public ILaunch getLaunch() {
		return _launch;
	}
	/**
	 * @see IProcess#getStreamsProxy()
	 */
	public IStreamsProxy getStreamsProxy() {
		if (_proxy == null)
			_proxy = new IOProxy(_process);
		return _proxy;
	}
	/**
	 * @see ITerminate#canTerminate()
	 */
	public boolean canTerminate() {
		return _isAlive && getLaunchMode() != 0;
	}
	/**
	 * @see ITerminate#isTerminated()
	 */
	public boolean isTerminated() {
		return !_isAlive;
	}
	/**
	 * @see ITerminate#terminate()
	 */
	public void terminate() throws DebugException {
		
		//#88654; only processes launched should be terminated
		if (canTerminate())
		{
			try {
				_process.kill(0);
			} catch (Exception e) {
				e.printStackTrace();
			}
			_isAlive = false;
		}
	}
	public int getLaunchMode() {
		return launchMode;
	}
	public void setLaunchMode(int launchMode) {
		this.launchMode = launchMode;
	}
}
/**
 * A proxy to the streams provided by the <code>Process</code>. Needed by
 * the console view.
 */
class IOProxy implements IStreamsProxy, IStreamMonitor, DataProcessor {
	private List _listeners = new ArrayList();
	private StringBuffer _buffer = new StringBuffer();
	private Console _console;
	/**
	 * Creates a new proxy.
	 */
	public IOProxy(Process process) {
		_console = process.getConsole();
		_console.setDataProcessor(this);
	}
	/**
	 * @see IStreamsProxy#getErrorStreamMonitor()
	 */
	public IStreamMonitor getErrorStreamMonitor() {
		return null;
	}
	/**
	 * @see IStreamsProxy#getOutputStreamMonitor()
	 */
	public IStreamMonitor getOutputStreamMonitor() {
		return this;
	}
	/**
	 * @see IStreamsProxy#write()
	 */
	public void write(String input) throws IOException {
		_console.write(input);
	}
	/**
	 * @see IStreamMonitor#addListener()
	 */
	public void addListener(IStreamListener listener) {
		synchronized (_listeners) {
			_listeners.add(listener);
		}
	}
	/**
	 * @see IStreamMonitor#removeListener()
	 */
	public void removeListener(IStreamListener listener) {
		synchronized (_listeners) {
			_listeners.remove(listener);
		}
	}
	/**
	 * @see IStreamMonitor#getContents()
	 */
	public String getContents() {
		return _buffer.toString();
	}
	/**
	 * @see IDataProcessor#incommingData()
	 */
	public void incommingData(byte[] buffer, int length, InetAddress peer) {
		
		try {
			String str = new String(buffer, 0, length,"UTF8");
			_buffer.append(str);
			synchronized (_listeners) {
				Iterator i = _listeners.iterator();
				while (i.hasNext())
					((IStreamListener) i.next()).streamAppended(str, this);
			}
		} catch (UnsupportedEncodingException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	/**
	 * @see IDataProcessor#incommingData()
	 */
	public void incommingData(char[] buffer, int length, InetAddress peer) {
		byte[] buf = new String(buffer).getBytes();
		incommingData(buf, buf.length, peer);
	}
	/**
	 * @see IDataProcessor#invalidDataType()
	 */
	public void invalidDataType(byte[] arg0, int arg1, InetAddress arg2) {
	}
	/**
	 * @see IDataProcessor#waitingForData()
	 */
	public void waitingForData() {
	}
}
