/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: OpenLogAgentPage.java,v 1.7 2005/04/06 22:30:18 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.wizard;

import java.text.DateFormat;

import org.eclipse.core.resources.IResource;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.core.*;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

/*
* CONTEXT_ID olap0000 for open Logging Agent page
*/

public class OpenLogAgentPage  extends WizardPage implements ModifyListener 
{
	protected TraceLocationUI _locationUI;

	protected Text _name;

	protected INavigator fViewer;

	protected Text _hour;

	protected Text _min;

	protected Text _sec;

	protected Text _micsec;

	protected Group deltaTime;
	protected int hourValue = 0;				
	protected int minValue = 0;				
	protected int secValue = 0;
	protected int micsecValue = 0;
	

	public OpenLogAgentPage(String name, INavigator viewer)

	{
		super(name);
		setTitle(UIPlugin.getResourceString("LOG_AGENT_TITLE"));
		fViewer = viewer;

		setDescription("");
	}

	/**
	 * Insert the method's description here. Creation date: (08/22/2000 4:48:53
	 * PM)
	 * 
	 * @param show
	 *            boolean
	 */
	public void createControl(Composite parent) {
		Label label;

		Composite content = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		layout.verticalSpacing = 5;
		content.setLayout(layout);
		GridData gd = GridUtil.createFill();
		gd.widthHint = 500;
		gd.heightHint = 400;
		content.setLayoutData(gd);

		_locationUI = new TraceLocationUI();
		_locationUI.createControl(content);
		_locationUI.setEnabled(false);

		Composite nameGrp = new Composite(content, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 2;
		layout.verticalSpacing = 5;
		nameGrp.setLayout(layout);
		nameGrp.setLayoutData(GridUtil.createHorizontalFill());

		Label nameLabel = new Label(nameGrp, SWT.NONE);
		nameLabel.setText(UIPlugin.getResourceString("STR_AGENT_NAME"));
		_name = new Text(nameGrp, SWT.BORDER);
		_name.setLayoutData(GridUtil.createHorizontalFill());
		_name.setEnabled(false);

		deltaTime = new Group(content, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 8;
		layout.verticalSpacing = 5;
		deltaTime.setLayout(layout);
		deltaTime.setLayoutData(GridUtil.createHorizontalFill());
		deltaTime.setText(UIPlugin.getResourceString("STR_SYNCH_NODE_TIME"));

		label = new Label(deltaTime, SWT.NONE);
		label.setText(UIPlugin.getResourceString("STR_DELTA_TIME"));

		_hour = new Text(deltaTime, SWT.BORDER);
		gd = new GridData();
		gd.widthHint = 40;
		_hour.setLayoutData(gd);
		_hour.setTextLimit(4);
		label = new Label(deltaTime, SWT.NONE);
		label.setText(":");

		_min = new Text(deltaTime, SWT.BORDER);
		gd = new GridData();
		gd.widthHint = 30;
		_min.setLayoutData(gd);
		_min.setTextLimit(3);
		label = new Label(deltaTime, SWT.NONE);
		label.setText(":");

		_sec = new Text(deltaTime, SWT.BORDER);
		gd = new GridData();
		gd.widthHint = 30;
		_sec.setLayoutData(gd);
		_sec.setTextLimit(3);
		label.setText(":");

		_micsec = new Text(deltaTime, SWT.BORDER);
		gd = new GridData();
		gd.widthHint = 70;
		_micsec.setLayoutData(gd);
		_micsec.setTextLimit(7);

		label = new Label(deltaTime, SWT.NONE);
		label.setText(UIPlugin.getResourceString("STR_DELTA_HMS"));
		
		label = new Label(deltaTime, SWT.NONE);
		label.setText(UIPlugin.getResourceString("STR_DELTA_MICROSEC_EXAMPLE"));
		gd = new GridData();
		gd.horizontalSpan = 8;
		label.setLayoutData(gd);
		

		_hour.addModifyListener(this);
		_min.addModifyListener(this);
		_sec.addModifyListener(this);
		_micsec.addModifyListener(this);

		setControl(content);

		populateData();

		org.eclipse.ui.help.WorkbenchHelp.setHelp(parent, UIPlugin
				.getPluginId()
				+ ".olap0000");

	}

	public boolean finish() {
		TRCAgentProxy agent = ((OpenLogAgentWizard) getWizard()).getAgent();

		if (agent != null)
			agent.setDeltaTime(getDeltaTime());

		return true;
	}

	public double getDeltaTime() {
		int hours = 0;
		try {
			hours = Integer.parseInt(_hour.getText());
		} catch (Exception exc) {
		}

		int min = 0;
		try {
			min = Integer.parseInt(_min.getText());
		} catch (Exception exc) {
		}
		int sec = 0;
		try {
			sec = Integer.parseInt(_sec.getText());
		} catch (Exception exc) {
		}
		int micsec = 0;
		try {
			micsec = Integer.parseInt(_micsec.getText());
		} catch (Exception exc) {
		}
		
		if(hours<0 || min<0 || sec<0 || micsec<0){
			return  -((double) (Math.abs(hours) * 3600 + Math.abs(min) * 60 + Math.abs(sec)) * 1000000 + Math.abs(micsec));
		}

		return ((double) hours * 3600 + min * 60 + sec) * 1000000 + micsec;

	}

	public void setDeltaTime(double delta) {
		DateFormat df = DateFormat.getTimeInstance();

		try {
			int micsec = (int) (Math.abs(delta) % 1000000);
			double absDelta = Math.abs(delta) / 1000000;
			int hour = (int) (absDelta / 3600);
			int min = (int) ((absDelta % 3600) / 60);
			int sec = (int) ((absDelta % 3600) % 60);
			
			if(delta>=0){
				hourValue = hour;
				minValue = min;
				secValue = sec;
				micsecValue = micsec;
			}else{
				hourValue = hour>0 ? -hour : hour;
				if(hourValue!=0){
					minValue = min;
				}else{ 
					if(min!=0){				
						minValue = -min;
					}else{
						minValue = 0;
					}
				}
				
				if(hourValue!=0 || minValue!=0){
					secValue = sec;
				}else{
					if(sec!=0){
						secValue = -sec;
					}else{
						secValue = sec;
					}
				}
				
				if(hourValue!=0 || minValue!=0 || secValue!=0){
					micsecValue = micsec;
				}else{
					if(micsec!=0){
						micsecValue = -micsec;
					}else{
						micsecValue = micsec;
					}
				}

			}
			_hour.setText(hourValue == 0 ? "" : String.valueOf(hourValue));
			_min.setText(minValue == 0 ? "" : String.valueOf(minValue));
			_sec.setText(secValue == 0 ? "" : String.valueOf(secValue));
			_micsec.setText(micsecValue == 0 ? "" : String.valueOf(micsecValue));
		} catch (Exception exc) {
			exc.printStackTrace();
		}
	}

	/**
	 * Insert the method's description here. Creation date: (10/05/2000 3:17:53
	 * PM)
	 */
	protected void populateData() {
		IResource sel = HyadesUtil.getSelectionFolder(fViewer);
		if (sel != null)
			_locationUI.setLocation(sel.getFullPath().toOSString());

		TRCAgentProxy agent = ((OpenLogAgentWizard) getWizard()).getAgent();

		if (agent == null)
			return;

		_locationUI.setMonitor(agent.getProcessProxy().getNode().getMonitor()
				.getName());

		_name.setText(agent.getName());
		if (agent.getDeltaTime() != 0) {
			setDeltaTime(agent.getDeltaTime());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
	 */
	public void modifyText(ModifyEvent event) {
		if (event.widget == _hour) {
			try {
				hourValue = Integer.parseInt(_hour.getText().trim());
				setErrorMessage(null);					
				setPageComplete(canCompletePage());
			} catch (NumberFormatException e) {
				if (_hour.getText().trim().equals("")) {
					hourValue = 0;
					setPageComplete(canCompletePage());
					return;
				}
				String message = UIPlugin
						.getResourceString("INVALID_DELTA_FORMAT");				
				setErrorMessage(message);
				setPageComplete(false);
			}
		} else 
		if (event.widget == _min) {
			try {				
				minValue = Integer.parseInt(_min.getText().trim());
				setPageComplete(canCompletePage());
			} catch (NumberFormatException e) {
				if (_min.getText().trim().equals("")) {
					minValue = 0;
					setPageComplete(canCompletePage());					
					return;
				}
				String message = UIPlugin
				.getResourceString("INVALID_DELTA_FORMAT");				
				setErrorMessage(message);

				setPageComplete(false);
			}
		} else if (event.widget == _sec) {
			try {
				secValue = Integer.parseInt(_sec.getText().trim());
				setPageComplete(canCompletePage());
			} catch (NumberFormatException e) {
				if (_sec.getText().trim().equals("")) {
					secValue = 0;
					setPageComplete(canCompletePage());
					return;
				}
				String message = UIPlugin
				.getResourceString("INVALID_DELTA_FORMAT");				
				setErrorMessage(message);

				setPageComplete(false);				
			}
		} else if (event.widget == _micsec) {
			try {
				micsecValue = Integer.parseInt(_micsec.getText().trim());
				setPageComplete(canCompletePage());
			} catch (NumberFormatException e) {
				if (_micsec.getText().trim().equals("")) {
					micsecValue = 0;
					setPageComplete(canCompletePage());
					return;
				}
				String message = UIPlugin
				.getResourceString("INVALID_DELTA_FORMAT");				
				setErrorMessage(message);

				setPageComplete(false);
			}
		}

	}

	protected boolean canCompletePage() {

		String hour = _hour.getText().trim();
		String min = _min.getText().trim();
		String sec = _sec.getText().trim();
		String micsec = _micsec.getText().trim();

		return isHourValid() && isMinValid() && isSecValid() && isMicsecValid();
		
	}
	
	protected boolean isHourValid(){
		int hourValue = 0;
		String hour = _hour.getText().trim();
		try {
			hourValue = Integer.parseInt(hour);
		} catch (Exception e) {
			if(!hour.equals("")){
				setErrorMessage(UIPlugin
						.getResourceString("INVALID_DELTA_FORMAT"));				
				return false;
			}
		}
		setErrorMessage(null);
		return true;
	}
	
	protected boolean isMinValid(){
		int minValue = 0;
		String hour = _hour.getText().trim();
		String min = _min.getText().trim();
		try {
			minValue = Integer.parseInt(min);
		} catch (Exception e) {
			if (!_min.getText().trim().equals("")) {
				setErrorMessage(UIPlugin
						.getResourceString("INVALID_DELTA_FORMAT"));
				return false;
			}

		}
		if(!hour.equals("")	&& hourValue != 0
				&& !(minValue >= 0 && minValue < 60)){
			setErrorMessage(UIPlugin
					.getResourceString("INVALID_DELTA_TIME"));
			return false;			
		}
		setErrorMessage(null);
		return true;
		
	}
	
	protected boolean isSecValid(){
		int secValue = 0;
		String hour = _hour.getText().trim();
		String min = _min.getText().trim();
		String sec = _sec.getText().trim(); 
		try {
			secValue = Integer.parseInt(sec);
		} catch (Exception e) {
			if (!_sec.getText().trim().equals("")) {
				setErrorMessage(UIPlugin
						.getResourceString("INVALID_DELTA_FORMAT"));
				return false;
			}

		}
		if(((!hour.equals("")&& hourValue != 0) || (!min.equals("") && minValue!=0))
				&& !(secValue >= 0 && secValue < 60)){
			setErrorMessage(UIPlugin
					.getResourceString("INVALID_DELTA_TIME"));
			return false;			
		}
		setErrorMessage(null);
		return true;
	}
	
	protected boolean isMicsecValid(){
		int microsecValue = 0;
		String hour = _hour.getText().trim();
		String min = _min.getText().trim();
		String sec = _sec.getText().trim();
		String micsec = _micsec.getText().trim();
		try {
			micsecValue = Integer.parseInt(micsec);
		} catch (Exception e) {
			if (!_micsec.getText().trim().equals("")) {
				setErrorMessage(UIPlugin
						.getResourceString("INVALID_DELTA_FORMAT"));
				return false;
			}

		}
		if((((!hour.equals("")&& hourValue != 0) || (!min.equals("") && minValue!=0))||
				(!sec.equals("")&& secValue != 0))
				&& !(micsecValue >= 0 && micsecValue < 1000000)){
			setErrorMessage(UIPlugin
					.getResourceString("INVALID_DELTA_TIME"));
			return false;			
		}
		setErrorMessage(null);
		return true;
	}	
}
