/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfileSetWizard.java,v 1.4 2005/04/04 18:12:25 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.wizard;

import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceProfileOverviewUI;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSet;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.IWorkbench;

public abstract class ProfileSetWizard extends Wizard {
	
	private ProfileSetWizardPage1 wizPage1;
	private ProfileSetWizardPage2 wizPage2;
	private ProfileSetWizardPage3 wizPage3;
	private IStructuredSelection selection;
	private IWorkbench workbench;
	private ProfilingSetsManagerCopy original;
	private ProfilingSetsManagerCopy copy;
	
	private String _name;
	private IProfilingSet set;
	
	private TraceProfileOverviewUI _overviewUI;
	
	public ProfileSetWizard(TraceProfileOverviewUI overviewUI)
	{
		super();
		original = overviewUI.getManagerCopy();
		copy = original.createCopy();
		_overviewUI = overviewUI;			
	}
	
	public void initializeSet(IProfilingSet profilingSet) {
	    set = profilingSet;
		setProfileSName(set.getName());
	}
	
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		this.workbench = workbench;
		this.selection = selection;
	}
	
	protected abstract String getSetId();
	
	private void setProfileSName(String name)
	{
		_name = name;
		
		if (getSetId() == null)
			set.setId(_name);
		else
			set.setId(getSetId());
		
		set.setName(_name);
		copy.getProfilingSets().put(set.getId(), set);
		copy.setDefaultSet(set);		
	}
	
	public boolean profileNameExists(String name)
	{
		if (_name != null && name.equals(_name.trim()))
			return false;
		else
		{
			Iterator iter = copy.getProfilingSets().values().iterator();
			
			String otherSetName = null;
			
			while (iter.hasNext())
			{
				if (name.equals(((IProfilingSet)iter.next()).getName().trim()))
					return true;
			}
			
			return false;
		}
	}
	
	public void setProfileSetName (String name) {
		copy.getProfilingSets().remove(set.getId());
		
		setProfileSName(name);
	}
	
	public void setProfileSetDescription (String description) {	
		set.setDescription(description);
	}	
	
	public void setNameToPages()
	{
		wizPage2.setProfilingSetName(set.getName());
	}
	
	public boolean performFinish() {
		if (!validateFilterPatterns(copy.getDefaultFilterSet().getChildren()))
			return false;
		
		original.resetFrom(copy);
		original.applyChanges();

		_overviewUI.dataChanged();
		_overviewUI.getTableViewer().refresh(true);
		_overviewUI.getTableViewer().setSelection(new StructuredSelection(set));
		
		return true;
	}
	
	public ProfilingSetsManagerCopy getProfilingSetsManagerCopy() {
		return copy;
	}	
	
	public void addPages() {
		super.addPages();
		
		wizPage1 = new ProfileSetWizardPage1("name", set.getName(), set.getDescription(), copy);			
		wizPage1.setTitle(UIPlugin.getResourceString("STR_ADD_PROFILINGSET_NAME_DLG"));
		wizPage1.setDescription(UIPlugin.getResourceString("STR_ADD_PROFILINGSET_NAME_DESC"));
		addPage(wizPage1);
		
		wizPage2 = new ProfileSetWizardPage2("profilingTypes", set.getName(), copy, _overviewUI);			
		wizPage2.setTitle(UIPlugin.getResourceString("STR_EDIT_PROFILINGSET_DLG"));
		wizPage2.setDescription(UIPlugin.getResourceString("STR_EDIT_PROFILINGSET_DESC"));
		addPage(wizPage2);		
		
		wizPage3 = new ProfileSetWizardPage3("filters", _overviewUI);
		wizPage3.setTitle(UIPlugin.getResourceString("STR_EDIT_FILTERSET_DLG"));
		wizPage3.setDescription(UIPlugin.getResourceString("STR_EDIT_FILTERSET_DESC"));
		addPage(wizPage3);
	}	
	
	private boolean isBadFilterPattern(String filterStr)
	{
		if (filterStr.length() <= 1)
			return false;
		else if (filterStr.indexOf('*') != filterStr.lastIndexOf('*'))
			return true;		
		else if (filterStr.substring(1, filterStr.length()-1).indexOf('*') >= 0)
			return true;
		
		return false;
	}
	
	private boolean validateFilterPatterns(List filterList)
	{
		String badFilters = "";
		String nl = System.getProperty("line.separator"); 
		
		if (filterList != null)
		{
			for (int i = 0; i < filterList.size(); i++)
			{
				FilterTableElement fte = (FilterTableElement)filterList.get(i);
				
				if (isBadFilterPattern(fte.getText()))
					badFilters = badFilters + fte.getText() + nl;
				
				if (isBadFilterPattern(fte.getMethod()))
					badFilters = badFilters + fte.getMethod() + nl;
				
			}
		}
		
		if ("".equals(badFilters))
			return true;
		else
		{
			String message = UIPlugin.getResourceString("STR_ST_FILTER_PATTERN_WARNNG");
			message = message + nl + nl+ badFilters;
			message = message + nl + UIPlugin.getResourceString("STR_ST_FILTER_PATTERN_WARNNG_CONTINUE");
			
			String[] buttons = {
					UIPlugin
							.getResourceString("STR_ST_FILTER_PATTERN_WARNNG_YES_BUTTON"),
					UIPlugin
							.getResourceString("STR_ST_FILTER_PATTERN_WARNNG_NO_BUTTON") };
			
			MessageDialog messageDialog = new MessageDialog(
					getShell(),
					UIPlugin.getResourceString("STR_ST_FILTER_PATTERN_WARNNG_TITLE"),
					null, message, MessageDialog.WARNING, buttons, 0);
			
			messageDialog.create();

			return messageDialog.open() == MessageDialog.OK;
		}		
	}	
}
