/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IHyadesEditorPart.java,v 1.5 2005/02/16 22:24:04 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.editor;

import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IPropertyListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;

import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * Interface for the Hyades editor part.  Each page may contain 
 * an arbitrary SWT control or an IEditorPart.
 * 
 * <p>The Hyades editor part could be considered a "pseudo" editor.  
 * Basically it is a mechanism to allow the definition of one or more editors 
 * (or, more specifically, IEditorExtensions) for the same resource or for the 
 * same resource file extension. When this editor is opened it checks the content 
 * of the resource and calculates which would be the appropriate editor extension 
 * to be used.  
 * 
 * <p>This interface is not intended to be implemented by clients.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public interface IHyadesEditorPart
extends IDisposable, ISelectionProvider
{
	/**
	 * The property id for <code>getTitle</code> and <code>getTitleImage</code>.
	 */
	public static final int PROP_TITLE = IWorkbenchPart.PROP_TITLE;

	/**
	 * The property id for <code>isDirty</code>.
	 */
	public static final int PROP_DIRTY = IEditorPart.PROP_DIRTY;

	/**
	 * The property id for <code>getEditorInput</code>.
	 */
	public static final int PROP_INPUT = IEditorPart.PROP_INPUT;

	/**
	 * The property id for <code>getEditorObject</code>.
	 */
	public static final int PROP_OBJECT = PROP_TITLE+PROP_DIRTY+PROP_INPUT+1;

	/**
	 * Adds a listener for changes to properties of this workbench part.
	 * Has no effect if an identical listener is already registered.
	 * 
	 * <p>The properties ids are as follows:
	 * <ul>
	 *   <li><code>IHyadesEditorPart.PROP_TITLE</code> </li>
	 *   <li><code>IHyadesEditorPart.PROP_INPUT</code> </li>
	 *   <li><code>IHyadesEditorPart.PROP_DIRTY</code> </li>
	 *   <li><code>IHyadesEditorPart.PROP_OBJECT</code> </li>
	 * </ul>
	 *
	 * @param listener a property listener
	 */
	public void addPropertyListener(IPropertyListener listener);

	/**
	 * Removes the given property listener from this workbench part.
	 * Has no affect if an identical listener is not registered.
	 * @param listener a property listener
	 */
	public void removePropertyListener(IPropertyListener listener);

	/**
	 * Adds a listener for selection changes in this selection provider.
	 * Has no effect if an identical listener is already registered.
	 * @param listener a selection changed listener
	 */
	public void addSelectionChangedListener(ISelectionChangedListener listener);

	/**
	 * Removes the given selection change listener from this selection provider.
	 * Has no affect if an identical listener is not registered.
	 * @param listener a selection changed listener
	 */
	public void removeSelectionChangedListener(ISelectionChangedListener listener);

	/**
	 * Fires a property changed event.
	 * @param propertyId the id of the property that changed
	 */
	public void firePropertyChange(int propertyId);

	/**
	 * Returns whether the contents of this part have changed since the last save
	 * operation. If this value changes the part must fire a property listener 
	 * event with <code>PROP_DIRTY</code>.
	 * 
	 * @return <code>true</code> if the contents have been modified and need
	 *   saving, and <code>false</code> if they have not changed since the last
	 *   save
	 */
	public boolean isDirty();

	/**
	 * Returns whether the content of this editor is read only.
	 * @return boolean
	 */
	public boolean isReadOnly();

	/**
	 * Returns the file extension.
	 * @return int
	 */
	public String getFileExtension();

	/**
	 * Returns the title of this Hyades editor part. If this value changes 
	 * the part must fire a property listener event with <code>PROP_TITLE</code>.
	 * 
	 * <p>The title is used to populate the title bar of this part's visual
	 * container.  
	 *
	 * @return the Hyades editor part title
	 */
	public String getTitle();

	/**
	 * Sets or clears the title of this Hyades editor part.
	 * @param title the title, or <code>null</code> to clear
	 */
	public void setTitle(String title);

	/**
	 * Returns the title image of this workbench part.  If this value changes 
	 * the part must fire a property listener event with <code>PROP_TITLE</code>.
	 * 
	 * <p>The title image is usually used to populate the title bar of this part's
	 * visual container. Since this image is managed by the part itself, callers
	 * must <b>not</b> dispose the returned image.
	 *
	 * @return the title image
	 */
	public Image getTitleImage();

	/**
	 * Sets or clears the title image of this part.
	 * @param titleImage the title image, or <code>null</code> to clear
	 */
	public void setTitleImage(Image titleImage);

	/**
	 * Returns the title tool tip text of this workbench part. If this value 
	 * changes the part must fire a property listener event with 
	 * <code>PROP_TITLE</code>.
	 * 
	 * <p>The tool tip text is used to populate the title bar of this part's 
	 * visual container.  
	 *
	 * @return the workbench part title tool tip
	 */
	public String getTitleToolTip();
	
	/**
	 * Sets or clears the title tool tip text of this part.
	 * @param text the new tool tip text
	 */
	public void setTitleToolTip(String text);	

	
	/**
	 * Sets or clears the object edited by this Hyades editor part.
	 * @param editorObject the object, or <code>null</code> to clear
	 */
	public void setEditorObject(Object editorObject);
	
	/**
	 * Returns the object that is edited by this editor.  If this value changes the 
	 * part must fire a property listener event with <code>PROP_OBJECT</code>.
	 * @return the edited object
	 */
	public Object getEditorObject();
	
	/**
	 * Gets the editor part associated with this instance.
	 * @return IEditorPart
	 */
	public IEditorPart getEditorPart();

	/**
	 * Returns the composite control containing this editor part's pages.
	 * This should be used as the parent when creating controls for the individual 
	 * pages. That is, when calling <code>addPage(Object)</code>, the container 
	 * returned by this method should be the parent for any SWT control related to 
	 * the object.
	 * 
	 * <p>Warning: Clients should not assume that the container is any particular 
	 * subclass of Composite.  The actual class used may change in order to improve 
	 * the look and feel of the editor part.  Any code making assumptions on the 
	 * particular subclass would thus be broken.
	 *
	 * @return the composite, or <code>null</code> if <code>createPartControl</code>
	 * has not been called yet
	 */
	public Composite getContainer();

	/**
	 * Creates and adds the given control as new page to this HyadesEditorPart.  The 
	 * control may be <code>null</code>, allowing it to be created and set later 
	 * using <code>setControl</code>.
	 * @param control the control, or <code>null</code>
	 * @return the index of the new page
	 * @see #setControl
	 */
	public int addPage(Control control);

	/**
	 * Sets the control for the given page index.  The page index must be valid.
	 * @param pageIndex the index of the page
	 * @param control the control for the specified page, or <code>null</code> to clear the control
	 */
	public void setControl(int pageIndex, Control control);
	
	/**
	 * Returns the control for the given page index, or <code>null</code>
	 * if no control has been set for the page.  The page index must be valid.
	 * @param pageIndex the index of the page
	 * @return the control for the specified page, or <code>null</code> if none has been set
	 */
	public Control getControl(int pageIndex);

	/**
	 * Creates and adds a new page containing the given editor to this page  
	 * container. This also hooks a property change listener on the nested editor.
	 * @param editor the nested editor
	 * @param input the input for the nested editor
	 * @return the index of the new page
	 * @exception PartInitException if a new page could not be created
	 * @see #handlePropertyChange the handler for property change events from the nested editor
	 */
	public int addPage(IEditorPart editor, IEditorInput input) throws PartInitException;
	
	/**
	 * Returns the editor for the given page index. The page index must be valid.
	 * @param pageIndex the index of the page
	 * @return the editor for the specified page, or <code>null</code> if the
	 * specified page was not created with
	 * <code>addPage(IEditorPart,IEditorInput)</code>
	 */
	public IEditorPart getEditor(int pageIndex);
	
	/**
	 * Removes the page with the given index from this editor part.  The controls for 
	 * the page are disposed of. The page index must be valid.
	 * @param pageIndex the index of the page
	 * @see #addPage
	 */
	public void removePage(int pageIndex);
	
	/**
	 * Returns the number of pages in this editor part.
	 * @return the number of pages
	 */
	public int getPageCount();

	/**
	 * Sets the currently active page.
	 * @param pageIndex the index of the page to be activated; the index must be 
	 * valid
	 */
	public void setActivePage(int pageIndex);
	
	/**
	 * Returns the index of the currently active page, or -1 if there is no 
	 * active page.
	 * @return the index of the active page, or -1 if there is no active page
	 */
	public int getActivePage();

	/**
	 * Sets the image for the page with the given index, or <code>null</code>
	 * to clear the image for the page. The page index must be valid.
	 * @param pageIndex the index of the page
	 * @param image the image, or <code>null</code>
	 */
	public void setPageImage(int pageIndex, Image image);
	
	/**
	 * Returns the image for the page with the given index, or <code>null</code> if 
	 * no image has been set for the page.  The page index must be valid.
	 * @param pageIndex the index of the page
	 * @return the image, or <code>null</code> if none
	 */
	public Image getPageImage(int pageIndex);

	/**
	 * Sets the text label for the page with the given index.  The page index must 
	 * be valid. The text label must not be null.
	 * @param pageIndex the index of the page
	 * @param text the text label
	 */
	public void setPageText(int pageIndex, String text);
	
	/**
	 * Returns the text label for the page with the given index or an empty string 
	 * if no text label has been set for the page. The page index must be valid.
	 * @param pageIndex the index of the page
	 * @return the text label for the page
	 */
	public String getPageText(int pageIndex);
	
	/**
	 * Returns the editor extension definition that is being used by this
	 * instance.
	 * @return IAssociationDescriptor
	 */
	public IAssociationDescriptor getEditorDescriptor();	
}
