/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IAssociationMapping.java,v 1.4 2005/02/16 22:24:04 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.ui.extension;

import org.eclipse.hyades.ui.util.IDisposable;

/**
 * 
 * <p>This interface is not intended to be implemented by clients.  Instances can 
 * be obtained by 
 * {@link org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#retrieveAssociationMapping(String)}.
 * 
 * @author popescu
 * @author marcelop
 * @since 0.0.1
 */
public interface IAssociationMapping
extends IDisposable
{
	/**
	 * Returns the association mapping registry that contains this 
	 * mapping
	 * @return IAssociationMappingRegistry
	 */
	public IAssociationMappingRegistry getAssociationMappingRegistry();
	
	/**
	 * Returns the extension point name that defines the association descriptors
	 * of this mapping. 
	 * @return String
	 */
	public String getExtensionPoint();
	
	/**
	 * Returns the id of the plugin that declares the association descriptors of this
	 * mapping.
	 * @return String
	 */
	public String getPluginId();

	/**
	 * Returns the types that are used with this mapping.
	 * @return String[]
	 */
	public String[] getTypes();
	
	/**
	 * Returns <b>all</b> the association descriptors of this mapping.  In other
	 * words, this methods returns the association descriptors of all the types
	 * that are used by this map.
	 * 
	 * <p>This method doesn't follow the "get" naming convention to reduce the
	 * change of being used in the wrong scenario.
	 * 
	 * @return IAssociationDescriptor[]
	 */
	public IAssociationDescriptor[] associationDescriptors();
	
	/**
	 * Returns the association descriptors of this mapping that are 
	 * not associated to any type.
	 * @return IAssociationDescriptor[]
	 */
	public IAssociationDescriptor[] getAssociationDescriptors();

	/**
	 * Returns the association descriptors of this mapping that are associated to
	 * - or that applies to - a type.
	 * 
	 * <p>The <code>type</code> argument is, for example, a type of "Monitor" or a
	 * type of "Test Suite".
	 * 
	 * @param type
	 * @return IAssociationDescriptor[]
	 * @throws IllegalArgumentException if the type is null.
	 */
	public IAssociationDescriptor[] getAssociationDescriptors(String type)
	throws IllegalArgumentException;
	
	/**
	 * Returns the association descriptor that has the specified id and that
	 * is not associated to any type.
	 * @param id
	 * @return IAssociationDescriptor
	 */
	public IAssociationDescriptor getAssociationDescriptor(String id);

	/**
	 * Returns the association descriptor that has the specified id and that
	 * is a specific type.
	 * @param id
	 * @param type
	 * @return IAssociationDescriptor
	 * @throws IllegalArgumentException if the type is null.
	 */
	public IAssociationDescriptor getAssociationDescriptor(String type, String id)
	throws IllegalArgumentException;

	/**
	 * Returns the default association descriptor from the set of 
	 * descriptors that are not associated to a type.
	 * @return IAssociationDescriptor
	 */
	public IAssociationDescriptor getDefaultAssociationDescriptor();

	/**
	 * Returns the default association descriptor associated to a type. 
	 * @param type
	 * @return IAssociationDescriptor
	 * @throws IllegalArgumentException if the type is null.
	 */
	public IAssociationDescriptor getDefaultAssociationDescriptor(String type)
	throws IllegalArgumentException;

	/**
	 * Sets the default association descriptor for the set of descriptors that
	 * are not associated to a type.
	 * @param associationDescriptor
	 * @return boolean <code>true</code> if the default was set or <code>false</code>
	 * otherwise.
	 * @throws IllegalArgumentException<OL>
	 * <LI>if the associationDescriptor is not owned by this mapping <b>or</b></LI>
	 * <LI>if the association descriptor doesn't apply to all the types.</LI>
	 * </OL>
	 */
	public boolean setDefaultAssociationDescriptor(IAssociationDescriptor associationDescriptor)
	throws IllegalArgumentException;

	/**
	 * Sets the default association descriptor for a given type.
	 * @param type
	 * @param associationDescriptor
	 * @return boolean <code>true</code> if the default was set or <code>false</code>
	 * otherwise.
	 * @throws IllegalArgumentException<OL>
	 * <LI>if the associationDescriptor is not owned by this mapping <b>or</b></LI>
	 * <LI>if the association descriptor doesn't apply to the type <b>or</b></LI>
	 * <LI>the type is <code>null</code>.</LI>
	 * </OL>
	 */
	public boolean setDefaultAssociationDescriptor(String type, IAssociationDescriptor associationDescriptor)
	throws IllegalArgumentException;
	
	/**
	 * Adds an association descriptor to the "avoided set" of a given type.  The
	 * avoided set has the descriptors that could be used by a type but won't due to
	 * user preferences. 
	 * @param type
	 * @param associationDescriptor
	 * @return <code>true</code> if the association was added to the avoid set or 
	 * <code>false</code> otherwise.
	 */
	public boolean addToAvoidedSet(String type, IAssociationDescriptor associationDescriptor);

	/**
	 * Removes an association descriptor from the "avoided set" of a given type.  The
	 * avoided set has the descriptors that could be used by a type but won't due to
	 * user preferences. 
	 * @param type
	 * @param associationDescriptor
	 * @return <code>true</code> if the association was removed from the avoid set or 
	 * <code>false</code> otherwise.
	 */
	public boolean removeFromAvoidedSet(String type, IAssociationDescriptor associationDescriptor);
	
	/**
	 * Removes all the association descriptors from the "avoided set" of a given type.  
	 * The avoided set has the descriptors that could be used by a type but won't due to
	 * user preferences. 
	 * @param type
	 * @param associationDescriptor
	 * @return <code>true</code> if the association was removed from the avoid set or 
	 * <code>false</code> otherwise.
	 */	
	public boolean removeAllFromAvoidSet(String type);

	/**
	 * Returns all the association descriptors from the "avoided set" of a given type.  
	 * The avoided set has the descriptors that could be used by a type but won't due to
	 * user preferences. 
	 * @param type
	 * @return IAssociationDescriptor[]
	 */	
	public IAssociationDescriptor[] getAvoidedAssociationDescriptors(String type);
	
	/**
	 * Creates an association descriptor owned by this mapping.  The descriptor
	 * is not associated to any particular type.
	 * 
	 * <p>Although the descriptor created by this method is added to the list
	 * of untyped descriptors of this mapping, there is no validation against
	 * the {@link IAssociationDescriptor} that may be registered to this 
	 * mapping's registry.  It is up to the invoker of this method that the
	 * descriptor has all the required attributes.
	 * 
	 * @return IAssociationDescriptor
	 */
	public IAssociationDescriptor createAssociationDescriptor();
}
