/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AssociationMapping.java,v 1.5 2005/02/16 22:24:04 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.extension;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.resource.ImageRegistry;

import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationDescriptorFilter;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.extension.IAssociationMappingRegistry;
import org.eclipse.hyades.ui.internal.util.CoreUtil;

/**
 * Implementation of the {@link org.eclipse.hyades.ui.extension.IAssociationMapping}
 * interface.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class AssociationMapping 
implements IAssociationMapping
{	
	private AssociationMappingRegistry associationMappingRegistry;
	private String pluginId;
	private List commonAssociationDescriptors;
	private Map entryByType;
	private String extensionPoint;
	private boolean pluginRegistryWasLoaded = false;
	
	/**
	 * Constructor for AssociationMapping
	 * @param associationMappingRegistry the registry that owns this mapping.
	 * @param extensionPoint the extension point that declares the descriptors
	 * of this mapping.
	 * @param pluginId the plugin id where the extension was defined.
	 */
	public AssociationMapping(AssociationMappingRegistry associationMappingRegistry, String extensionPoint, String pluginId)
	{
		this.associationMappingRegistry = associationMappingRegistry;
		this.extensionPoint = extensionPoint;
		this.pluginId = pluginId;
		
		commonAssociationDescriptors = new ArrayList();
		entryByType = new HashMap();
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		CoreUtil.dispose(commonAssociationDescriptors);
		CoreUtil.dispose(entryByType);			
		associationMappingRegistry = null;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#getPluginId()
	 */
	public String getPluginId()
	{
		return pluginId;
	}

	/**
	 * Returns the image registry that is used to store the image
	 * of the descriptors.
	 * @return
	 */
	protected ImageRegistry getImageRegistry()
	{
		return associationMappingRegistry.getImageRegistry();	
	}
	
	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#getAssociationMappingRegistry()
	 */
	public IAssociationMappingRegistry getAssociationMappingRegistry()
	{
		return associationMappingRegistry;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#getExtensionPoint()
	 */
	public String getExtensionPoint()
	{
		return extensionPoint;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#getTypes()
	 */
	public String[] getTypes()
	{
		loadPluginRegistry();
		return (String[])entryByType.keySet().toArray(new String[entryByType.keySet().size()]);
	}

	/**
	 * Returns whether this association mapping has descriptors.  This method
	 * <b>doesn't</b> load the descriptors from the plugin registry. 
	 * @return boolean
	 */
	protected boolean hasAssociationDescriptors()
	{
		return !(commonAssociationDescriptors.isEmpty() && entryByType.isEmpty());
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#associationDescriptors()
	 */
	public IAssociationDescriptor[] associationDescriptors()
	{
		loadPluginRegistry();
		
		Set set = new HashSet(commonAssociationDescriptors);
		for (Iterator i = entryByType.values().iterator(); i.hasNext();)
			set.addAll(((AssociationDescriptorEntry)i.next()).getTypedAssociationDescriptor());
		
		return (IAssociationDescriptor[])set.toArray(new IAssociationDescriptor[set.size()]);
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#getAssociationDescriptors()
	 */
	public IAssociationDescriptor[] getAssociationDescriptors()
	{
		loadPluginRegistry();
		return (IAssociationDescriptor[])commonAssociationDescriptors.toArray(new IAssociationDescriptor[commonAssociationDescriptors.size()]);
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAvgssociationMapping#getAssociationDescriptor(java.lang.String)
	 */
	public IAssociationDescriptor[] getAssociationDescriptors(String type)
	throws IllegalArgumentException
	{
		if(type == null)
			throw new IllegalArgumentException(HyadesUIPlugin.getString("_ERROR_NULL_TYPE"));

		loadPluginRegistry();
		return retrieveEntry(type).getAssociationDescriptors();
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#getAssociationDescriptor(java.lang.String)
	 */
	public IAssociationDescriptor getAssociationDescriptor(String id)
	{
		if(id == null)
			return null;
			
		IAssociationDescriptor[] descriptors = getAssociationDescriptors();
		for (int i = 0, maxi = descriptors.length; i < maxi; i++)
		{
			if(id.equals(descriptors[i].getId()))
				return descriptors[i];			
		}	
		
		return null;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#getAssociationDescriptor(java.lang.String, java.lang.String)
	 */
	public IAssociationDescriptor getAssociationDescriptor(String type, String id) 
	throws IllegalArgumentException
	{
		if(id == null)
			return null;

		IAssociationDescriptor[] descriptors = getAssociationDescriptors(type);
		for (int i = 0, maxi = descriptors.length; i < maxi; i++)
		{
			if(id.equals(descriptors[i].getId()))
				return descriptors[i];			
		}	
		
		return null;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#getDefaultAssociationDescriptor()
	 */
	public IAssociationDescriptor getDefaultAssociationDescriptor()
	{
		loadPluginRegistry();
		if(commonAssociationDescriptors.isEmpty())
			return null;
		return (IAssociationDescriptor)commonAssociationDescriptors.get(0);
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#getDefaultAssociationDescriptor(java.lang.String)
	 */
	public IAssociationDescriptor getDefaultAssociationDescriptor(String type)
	throws IllegalArgumentException
	{
		if(type == null)
			throw new IllegalArgumentException(HyadesUIPlugin.getString("_ERROR_NULL_TYPE"));

		loadPluginRegistry();
		return retrieveEntry(type).getDefault();
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#setDefaultAssociationDescriptor(org.eclipse.hyades.ui.extension.IAssociationDescriptor)
	 */
	public synchronized boolean setDefaultAssociationDescriptor(IAssociationDescriptor associationDescriptor) throws IllegalArgumentException
	{
		if(associationDescriptor == null)
			return false;
			
		if(associationDescriptor.getAssociationMapping() != this)
			throw new IllegalArgumentException(HyadesUIPlugin.getString("_ERROR_INV_ASSOC"));

		int index = commonAssociationDescriptors.indexOf(associationDescriptor);
		if(index == 0)
			return true;
			
		if(index > 0)
			commonAssociationDescriptors.remove(index);
		commonAssociationDescriptors.add(0, associationDescriptor);
		
		return (associationDescriptor == commonAssociationDescriptors.get(0));
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#setDefaultAssociationDescriptor(java.lang.String, org.eclipse.hyades.ui.extension.IAssociationDescriptor)
	 */
	public synchronized boolean setDefaultAssociationDescriptor(String type, IAssociationDescriptor associationDescriptor)
	throws IllegalArgumentException
	{
		if(associationDescriptor == null)
			return false;

		if(type == null)
			throw new IllegalArgumentException(HyadesUIPlugin.getString("_ERROR_NULL_TYPE"));
			
		if(associationDescriptor.getAssociationMapping() != this)
			throw new IllegalArgumentException(HyadesUIPlugin.getString("_ERROR_INV_ASSOC"));

		if(!associationDescriptor.applyTo(type))
			throw new IllegalArgumentException(HyadesUIPlugin.getString("_ERROR_INV_ASSOC"));

		AssociationDescriptorEntry entry = retrieveEntry(type);
		entry.setDefault(associationDescriptor);
		return (entry.getDefault() == associationDescriptor);
	}
	
	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#addToAvoidedSet(java.lang.String, org.eclipse.hyades.ui.extension.IAssociationDescriptor)
	 */
	public boolean addToAvoidedSet(String type, IAssociationDescriptor associationDescriptor)
	{
		if((type == null) || (associationDescriptor == null) || (!associationDescriptor.applyTo(type)))
			return false;
			
		retrieveEntry(type).addToAvoidedSet(associationDescriptor);
		return true;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#getAvoidedAssociationDescriptors(java.lang.String)
	 */
	public IAssociationDescriptor[] getAvoidedAssociationDescriptors(String type)
	{
		if(type != null)
		{
			Set ret = retrieveEntry(type).avoidedAssociationDescriptors();
			return (IAssociationDescriptor[])ret.toArray(new IAssociationDescriptor[ret.size()]); 
		}
			
		return new IAssociationDescriptor[0];
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#removeAllFromAvoidSet(java.lang.String)
	 */
	public boolean removeAllFromAvoidSet(String type)
	{
		if(type == null)
			return false;
			
		retrieveEntry(type).removeAllFromAvoidedSet();
		return true;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#removeFromAvoidedSet(java.lang.String, org.eclipse.hyades.ui.extension.IAssociationDescriptor)
	 */
	public boolean removeFromAvoidedSet(String type, IAssociationDescriptor associationDescriptor)
	{
		if(type == null)
			return false;
			
		return retrieveEntry(type).removeFromAvoidedSet(associationDescriptor);
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMapping#createAssociationDescriptor()
	 */
	public IAssociationDescriptor createAssociationDescriptor()
	{
		AssociationDescriptor associationDescriptor = new AssociationDescriptor(this);
		commonAssociationDescriptors.add(associationDescriptor);
		return associationDescriptor;
	}

	/**
	 * Loads the association descriptors from the plugin registry.
	 */
	protected synchronized void loadPluginRegistry()
	{
		if(pluginRegistryWasLoaded)
			return;
		
		if(pluginId != null)
		{
			IConfigurationElement[] configurationElements = Platform.getExtensionRegistry().getConfigurationElementsFor(pluginId, getExtensionPoint());
			for (int i = 0, maxi = configurationElements.length; i < maxi; i++)
			{
				AssociationDescriptor associationDescriptor = new AssociationDescriptor(this);
				associationDescriptor.loadFromConfigurationElement(configurationElements[i]);
			}
		}
		
		pluginRegistryWasLoaded = true;
	}
	
	/**
	 * Returns an entry for the specified type creating a new one if
	 * required.
	 * @param type
	 * @return AssociationDescriptorEntry
	 */
	protected synchronized AssociationDescriptorEntry retrieveEntry(String type)
	{
		if(type == null)
			return null;
			
		AssociationDescriptorEntry entry = (AssociationDescriptorEntry)entryByType.get(type);
		if(entry == null)
		{
			entry = new AssociationDescriptorEntry(commonAssociationDescriptors, type);
			entryByType.put(type, entry);
		}
		return entry;
	}

	/**
	 * Registers an association descriptor to the common set of types.
	 * @param associationDescriptor
	 */
	protected synchronized void register(IAssociationDescriptor associationDescriptor)
	{
		if(!isValid(associationDescriptor))
			return;
			
		if(!commonAssociationDescriptors.contains(associationDescriptor))
			commonAssociationDescriptors.add(associationDescriptor);
	}
		
	/**
	 * Registers an association descriptor to a given type
	 * @param associationDescriptor
	 * @param type
	 * @return <code>true</code> if the association descriptor was registered or
	 * <code>false</code> otherwise.
	 */
	protected boolean register(String type, IAssociationDescriptor associationDescriptor)
	{
		if(type == null)
			return false;
			
		if(!isValid(associationDescriptor))
			return false;
			
		retrieveEntry(type).addTypedAssociationDescriptor(associationDescriptor);
		return true;
	}
	
	/**
	 * Deregisters an association descriptor from the common set of types.
	 * @param associationDescriptor
	 */
	protected void deregister(IAssociationDescriptor associationDescriptor)
	{
		commonAssociationDescriptors.remove(associationDescriptor);
	}
	
	/**
	 * Deregisters an association descriptor from a given type
	 * @param type
	 * @param associationDescriptor
	 */
	protected synchronized void deregister(String type, IAssociationDescriptor associationDescriptor)
	{
		if(type == null)
			return;
			
		AssociationDescriptorEntry entry = (AssociationDescriptorEntry)entryByType.get(type);
		if(entry == null)
			return;
			
		entry.removeTypedAssociationDescriptor(associationDescriptor);
	}
	
	/**
	 * Checks if the association descriptor is valid according to this
	 * mapping's filter.
	 * @param associationDescriptor
	 * @return <code>true</code> if the association descriptor is valid or 
	 * <code>false</code> otherwise.
	 */
	protected boolean isValid(IAssociationDescriptor associationDescriptor)
	{
		IAssociationDescriptorFilter filter = associationMappingRegistry.getFilter();
		if(filter == null)
			return true;
		
		return filter.isValid(this, associationDescriptor);
	}
}
