/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NavigatorViewerSorter.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.navigator;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.jface.viewers.Viewer;

import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.util.HyadesViewerSorter;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.hyades.ui.util.INamedElement;

/**
 * Sorter for viewers that display items of type <code>IResource</code>,
 * <code>INamedElement</code>, <code>LogicalFolder</code> and <code>IAdaptable</code>
 * that have <code>IResource</code> as a valid adapter.
 * 
 * <p>The sorter supports two sort criteria:
 * <p><code>NAME</code>: Folders are given order precedence, followed by files.
 * Within these two groups resources are ordered by name.  All name comparisons
 * are case-insensitive.
 * 
 * <p><code>TYPE</code>: Folders are given order precedence, followed by files.
 * Within these two groups resources are ordered by extension.
 * 
 * <p>All extension comparisons are case-insensitive.
 * 
 * <p>This implementation is based on the 
 * {@link org.eclipse.ui.views.navigator.ResourceSorter} class.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class NavigatorViewerSorter 
extends HyadesViewerSorter implements IDisposable
{
	/**
	 * Change type to be passed to the navigator's <code>handleChange(int)</code>
	 * method.
	 */
	public static final int SORTER_TYPE = NavigatorViewerSorter.class.hashCode();
	
	/**
	 * Constructor argument value that indicates to sort items by name.
	 */
	public final static int NAME = 1;

	/**
	 * Constructor argument value that indicates to sort items by extension.
	 */
	public final static int TYPE = 2;

	private INavigator navigator;

	private int criteria;
	
	/**
	 * Constructor for NavigatorViewerSorter
	 */
	public NavigatorViewerSorter(INavigator navigator, int criteria)
	{
		super();
		
		this.navigator = navigator;
		
		if((criteria != NAME) && (criteria != TYPE))
			this.criteria = NAME;
		else
			this.criteria = criteria;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		navigator = null;
	}
	
	/**
	 * Returns this sorter's navigator
	 * @return INavigator
	 */
	public INavigator getNavigator()
	{
		return navigator;
	}
	
	/**
	 * Sets the sort criteria of this this sorter.
	 * @param the sort criterion: one of <code>NAME</code> or <code>TYPE</code>.  
	 * Any other value is not considered.
	 */
	public void setCriteria(int newCriteria)
	{
		if(criteria == newCriteria)
			return;
			
		if((newCriteria != NAME) && (newCriteria != TYPE))
			return;
			
		criteria = newCriteria;
		if(navigator.getViewer() != null)
			navigator.getViewer().refresh();
		
		navigator.handleChange(SORTER_TYPE);
	}	
	
	/**
	 * Returns the sort criteria of this this sorter.
	 * @return the sort criterion: one of <code>NAME</code> or <code>TYPE</code>
	 */
	public int getCriteria()
	{
		return criteria;
	}

	/**
	 * @see org.eclipse.jface.viewers.ViewerSorter#category(java.lang.Object)
	 */
	public int category(Object element)
	{
		if(element instanceof IProject)
			return 1000;
			
		if(element instanceof IFolder)
			return 400;

		if(element instanceof LogicalFolder)
			return 200;			
		
		return 600;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.util.HyadesViewerSorter#getValue(org.eclipse.jface.viewers.Viewer, java.lang.Object)
	 */
	protected String getValue(Viewer viewer, Object e)
	{
		String value = null;

		if(e instanceof LogicalFolder)
		{
			value = getLogicalFolderValue(viewer, (LogicalFolder)e);
			if(value != null)
				return value;
		}

		if(e instanceof IResource)
		{
			value = getResourceValue(viewer, (IResource)e);
			if(value != null)
				return value;
		}
		
		if(e instanceof IAdaptable)
		{
			value = getAdaptableValue(viewer, (IAdaptable)e);
			if(value != null)
				return value;
		}
		
		if(e instanceof INamedElement)
		{
			value = getNamedElementValue(viewer, (INamedElement)e);
			if(value != null)
				return value;
		}
		
		return super.getValue(viewer, e);
	}
	
	/**
	 * Returns the value for a LogicFolder or <code>null</code> if no value
	 * can be assigned.  This implementation returns always the same value
	 * so the initial order should be preserved.
	 * @param viewer
	 * @param logicalFolder
	 * @return String
	 */
	protected String getLogicalFolderValue(Viewer viewer, LogicalFolder logicalFolder)
	{
		return "0";
	}

	/**
	 * Returns the value for a IResource or <code>null</code> if no value
	 * can be assigned. 
	 * @param viewer
	 * @param resource
	 * @return String
	 */
	protected String getResourceValue(Viewer viewer, IResource resource)
	{
		if(resource.getFileExtension() == null)
			return resource.getName();
			
		if(NAME == getCriteria())
			return resource.getName() + "_+_" + resource.getFileExtension();
		return resource.getFileExtension() + "_+_" + resource.getName();
	}

	/**
	 * Returns the value for an adaptable or <code>null</code> if no value
	 * can be assigned. 
	 * @param viewer
	 * @param adaptable
	 * @return String
	 */
	protected String getAdaptableValue(Viewer viewer, IAdaptable adaptable)
	{
		IResource resource = (IResource)adaptable.getAdapter(IResource.class);
		if(resource == null)
			return null;
		return getResourceValue(viewer, resource);
	}
	
	/**
	 * Returns the value for an namedElement or <code>null</code> if no value
	 * can be assigned. 
	 * @param viewer
	 * @param namedElement
	 * @return String
	 */
	protected String getNamedElementValue(Viewer viewer, INamedElement namedElement)
	{
		return namedElement.getName();
	}	
}
