/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ContentProviderPicker.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.provider;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;

/**
 * This content provider is a type of provider decorator that allows clients to 
 * select a provider to handle the requests.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class ContentProviderPicker 
implements ITreeContentProvider
{
	private List providers;
	private int currentProvider;
	
	/**
	 * Constructor for ContentProviderPicker
	 */
	public ContentProviderPicker()
	{
		providers = new ArrayList();
		currentProvider = -1;
	}

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
	 */
	public void dispose()
	{
		currentProvider = -1;
		for (Iterator i = providers.iterator(); i.hasNext();)
		{
			ITreeContentProvider treeContentProvider = (ITreeContentProvider)i.next();
			treeContentProvider.dispose();
		}
		providers.clear();
	}

	/**
	 * Adds a provider.  The provider are kept in the
	 * order they are added.
	 * 
	 * <p>This method sets the specified provider as the current if no other
	 * provider is already defined as current.
	 * 
	 * @param contentProvider
 	 */
	public void addProvider(ITreeContentProvider contentProvider)
	{
		if(providers.add(contentProvider) && (currentProvider < 0))
			currentProvider = providers.size()-1;		
	}
	
	/**
	 * Removes all the providers.
	 */
	public void clearProviders()
	{
		currentProvider = -1;
		providers.clear();
	}
	
	/**
	 * Removes the provider associated with a given index.  The first 
	 * provider added to this picker has index 0.
	 * 
	 * @param index
     * @return 
	 */
	public void removeProvider(int index)
	{
		if(index == currentProvider)
			currentProvider = -1;
		else if(index < currentProvider)
			currentProvider--;
			
		providers.remove(index);
	}
	
	/**
	 * Sets the current provider.  If the index is not in used the current
	 * selection is kept,
	 * @param index
	 * @return <code>true</code> if the current provider was set or <code>false</code> 
	 * otherwise.
	 */
	public boolean setCurrentProvider(int index)
	{
		if((index < 0) || (index >= providers.size()))
			return false;
		if(index == currentProvider)
			return true;

		currentProvider = index;
		return true;
	}
	
	/**
	 * Returns the current provider
	 * @return ITreeContentProvider
	 */
	public ITreeContentProvider getCurrentProvider()
	{
		if(currentProvider < 0)
			return null;
		return (ITreeContentProvider)providers.get(currentProvider);
	}
	
	/**
	 * Returns the index of the current provider.
	 * @return int
	 */
	public int getCurrentProviderIndex()
	{
		return currentProvider;
	}
	
	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object parentElement)
	{
		if(getCurrentProviderIndex() < 0)
			return new Object[0];
			
		return getCurrentProvider().getChildren(parentElement);
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
	 */
	public Object getParent(Object element)
	{
		if(getCurrentProviderIndex() < 0)
			return null;
			
		return getCurrentProvider().getParent(element);
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
	 */
	public boolean hasChildren(Object element)
	{
		if(getCurrentProviderIndex() < 0)
			return false;
			
		return getCurrentProvider().hasChildren(element);
	}

	/**
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement)
	{
		if(getCurrentProviderIndex() < 0)
			return new Object[0];
			
		return getCurrentProvider().getElements(inputElement);
	}

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
	{
		if(getCurrentProviderIndex() < 0)
			return;
			
		getCurrentProvider().inputChanged(viewer, oldInput, newInput);
	}
}
