/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available a
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ReportManager.java,v 1.6 2005/02/16 22:24:05 qiyanli Exp $
 *
 * # %Z% %I% %W% %G% %U% [%H% %T%]
 * 
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.internal.report;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.util.XMLUtil;
import org.eclipse.hyades.ui.report.IReportDescription;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;


/**
 * This manages all report generators.
 */
public class ReportManager implements IDisposable {
	
	private static String prefKey = "reportDescriptions";
	private List pluginReportDescription;
	private List preferenceReportDescription;
	
	private AbstractUIPlugin plugin;
	private String extensionPoint;
	
	/**
	 * Constructor for ReportManager
	 */
	public ReportManager(AbstractUIPlugin plugin, String extensionPoint)
	{
		super();
		preferenceReportDescription = new ArrayList();
		this.plugin = plugin;
		this.extensionPoint = extensionPoint;
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		if(pluginReportDescription != null)
			pluginReportDescription.clear();
		preferenceReportDescription.clear();
		
		plugin = null;
	}
	
	protected IPreferenceStore getPreferenceStore()
	{
		return plugin.getPreferenceStore();
	}

	public void addReportDefinition(IReportDescription reportDescription)
	{
		preferenceReportDescription.add(reportDescription);
	}
	
	public void removeReportDefinition(IReportDescription reportDescription)
	{
		preferenceReportDescription.remove(reportDescription);
	}
	
	public IReportDescription[] getPreferenceStoreReportDescriptions()
	{
		return (IReportDescription[])preferenceReportDescription.toArray(new ReportDescription[preferenceReportDescription.size()]);
	}
	
	public IReportDescription[] getReportDescriptions()
	{
		List ret = null;
		if(pluginReportDescription != null)
		{
			ret = new ArrayList(pluginReportDescription);
			ret.addAll(preferenceReportDescription);
		}
		else
			ret = preferenceReportDescription;
			
		return (IReportDescription[])ret.toArray(new ReportDescription[ret.size()]);
	}
	
	public void loadFromPluginRegistry()
	{
		if(pluginReportDescription != null)
			return;
			
		pluginReportDescription = new ArrayList();
		
		//code to load from plugin	
		if(plugin != null)
		{
			IConfigurationElement[] configurationElements = Platform.getExtensionRegistry().getConfigurationElementsFor(HyadesUIPlugin.getID(), extensionPoint);
			for (int i=0, maxi = configurationElements.length; i < maxi; i++)
			{
				if("reportDescription".equals(configurationElements[i].getName()))
				{
						String name = configurationElements[i].getAttribute("name");
						String className = configurationElements[i].getAttribute("class");
						pluginReportDescription.add(new ReportDescription(name, className, "", configurationElements[i]));
				}
			}
		}
	}
	
	public void loadFromPreferenceStore()
	{
		IPreferenceStore preferenceStore = getPreferenceStore();		
			
		String value = preferenceStore.getString(prefKey);
		if((value == null) || (value.length() == 0))
			return;
			
		XMLUtil.setLogger(HyadesUIPlugin.getLogger());
		Element registryElement = XMLUtil.loadDom(value, "reportDescriptions");
		if(registryElement == null)
			return;
			
		NodeList nodeList = XMLUtil.getChildrenByName(registryElement, "reportDescription");
		for(int i=0, maxi=nodeList.getLength(); i<maxi; i++)
		{
			if(nodeList.item(i) instanceof Element)
				loadDescription((Element)nodeList.item(i));
		}
	}
	
	/**
	 * Loads the preference setting located in the XML element.
	 * @param mappingElement
	 */
	protected void loadDescription(Element descriptionElement)
	{
		String name = XMLUtil.getValue(descriptionElement, "name");
		String clas = XMLUtil.getValue(descriptionElement, "class");
		if(name == null || clas == null)
			return;
		
		String classpath = XMLUtil.getValue(descriptionElement, "classpath");
		classpath = XMLUtil.removeXMLSymbols(classpath);
		
		ReportDescription report = new ReportDescription(plugin.getClass().getClassLoader(), name, clas, classpath);
		preferenceReportDescription.add(report);
		
	}
	
	
	public void saveIntoPreferenceStore()
	{
		IPreferenceStore preferenceStore = getPreferenceStore();		
		StringBuffer xml = new StringBuffer();
		
		//Header
		xml.append("<?xml");
		xml.append(XMLUtil.createXMLAttribute("version","1.0", false)).append(XMLUtil.createXMLAttribute("encoding","UTF-8", false));
		xml.append("?>");
		
		//Body
		xml.append("<reportDescriptions>");
		for(Iterator i = preferenceReportDescription.iterator(); i.hasNext();)
		{
			IReportDescription reportGeneratorDefinition = (IReportDescription)i.next();			
			storeDescription(xml, reportGeneratorDefinition);
		}
		xml.append("</reportDescriptions>");
		
		String xmlString = xml.toString();
		if(xmlString.equals(preferenceStore.getString(prefKey)))
			return;
			
		preferenceStore.setValue(prefKey, xmlString);
	}

	/**
	 * Writes the preferences of the specified report definition in an XML format
	 * @param xml
	 * @param reportDefinition
	 */
	protected void storeDescription(StringBuffer xml, IReportDescription reportDefinition)
	{
		StringBuffer tempXML = new StringBuffer();
		tempXML.append("<reportDescription");
		tempXML.append(XMLUtil.createXMLAttribute("name",reportDefinition.getName(), false));
		tempXML.append(XMLUtil.createXMLAttribute("class",reportDefinition.getClassName(), false));
		tempXML.append(XMLUtil.createXMLAttribute("classpath",reportDefinition.getClassPath(), false));		
		tempXML.append("/>");
			
		xml.append(tempXML);
	}
}
