/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: OpenJavaSource.java,v 1.9 2005/03/07 22:17:04 curtispd Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.util.Vector;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IParent;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.Signature;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.jdt.core.search.SearchEngine;
import org.eclipse.jdt.core.search.SearchMatch;
import org.eclipse.jdt.core.search.SearchParticipant;
import org.eclipse.jdt.core.search.SearchPattern;
import org.eclipse.jdt.core.search.SearchRequestor;
import org.eclipse.jdt.internal.core.SourceType;
import org.eclipse.jdt.ui.actions.OpenAction;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.hyades.ui.HyadesUIPlugin;

/**
 * A utility class to open a Java file selecting a specific element in the file.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class OpenJavaSource
{
	/**
	 * Eclipse's Java perspective ID.
	 */
	private final static String JAVA_PERSPECTIVE_ID = "org.eclipse.jdt.ui.JavaPerspective";
	

	/**
	 * Opens the source for a selected object in the text editor of the Java Perspective.
	 * @param String pattern - the object that we wish to display the source for.  The pattern should be the full
	 * package name of the class or method.  For a method, the signature is also part of the pattern.  If the method
	 * is a constructor, the class name is followed by the signature.  
	 * Eg., a typical method would be com.ibm.etools.pd.core.util.OpenSourceUtil.openSource(String, int) void
	 * and a constructor would be com.ibm.ibm.etools.pd.core.util.OpenSourceUtil()
	 * @param javaType The java type identifies whether the pattern represents a Type, a method, or a constructor.
	 * The values are defined in org.eclipse.jdt.core.search.IJavaSearchConstants and can be TYPE, METHOD, or CONSTRUCTOR 
	 * respectively.	
	 * @param searchScope The search scope for the source to be opened.  If null, the WorkspaceScope is be used;
	 * @param switchToJavaPerspective If <code>true</code> the Java perspective is be opened.
	 * @return <code>true</code> if the editor was opened or <code>false</code> otherwise
	 */
	public static boolean openSource(String pattern, int javaType, IJavaSearchScope searchScope, boolean switchToJavaPerspective)
	{
		OpenSourceJavaSearchResultRequestor requestor = getSearchResults(pattern, javaType, searchScope);
		if (requestor != null && requestor.getNumberOfMatches() > 0) {
			return openWbSource(pattern, requestor, switchToJavaPerspective);
		}
		return false;
	}

	public static IFile getSourceFile(String pattern, int javaType, IJavaSearchScope searchScope) {
		OpenSourceJavaSearchResultRequestor requestor = getSearchResults(pattern, javaType, searchScope);
		if (requestor != null) {
			return OpenSourceJavaSearchResultRequestor.getFile(requestor.getMatchForPattern(pattern));
		}
		return null;
	}

	/**
	 * Adjusts the search pattern for the case where the user is opening the source
	 * on an inner class.
	 * 
	 * @param pattern the original pattern.
	 * @param requestor the originator of the search.
	 * @param javaType the type of object this is (e.g. class, method).
	 * @return the revised pattern.
	 */
	private static String fixPatternForInnerClasses(String pattern, OpenSourceJavaSearchResultRequestor requestor, int javaType) {
		int bIdx = pattern.indexOf("(");
		int idx;
		if (bIdx >= 0) {
			idx = pattern.substring(0, bIdx).indexOf("$");
		}
		else {
			idx = pattern.indexOf("$");
		}
		
		if (idx >= 0) {
			requestor.setInnerClassType(new InnerClassType(pattern.substring(idx+1)));
			pattern = pattern.substring(0, idx);			
		}
		return pattern;		
	}

	/**
	 * Returns the SearchPattern instance constructed from the given parameters.
	 * 
	 * @param pattern the java pattern to search for.
	 * @param javaType the type of java object (e.g. method, class).
	 * @param requestor the search originator.
	 * @return the SearchPattern instance to use when searching.
	 */
	private static SearchPattern getSearchPattern(String pattern, int javaType, OpenSourceJavaSearchResultRequestor requestor) {
		if (pattern != null && !"".equals(pattern))	{
			pattern = fixPatternForInnerClasses(pattern, requestor, javaType);
			return SearchPattern.createPattern(pattern, requestor.hasInnerClassType()?IJavaSearchConstants.TYPE:javaType, IJavaSearchConstants.DECLARATIONS,
					SearchPattern.R_EXACT_MATCH | SearchPattern.R_CASE_SENSITIVE);
		}
		return null;
	}
	
	/**
	 * Returns a valid search scope if the given one is not valid (i.e. null).
	 * 
	 * @param searchScope the original search scope.
	 * @return a valid search scope.
	 */
	private static IJavaSearchScope getJavaSearchScope(IJavaSearchScope searchScope) {
		if (searchScope == null) {
			IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
			IJavaElement[] javaElements = new IJavaElement[projects.length];
			for (int i = 0; i < projects.length; i++) {
				javaElements[i] = JavaCore.create(projects[i]);
			}			
			searchScope = SearchEngine.createJavaSearchScope(javaElements, false);
		}
		return searchScope;
	}
	
	private static OpenSourceJavaSearchResultRequestor getSearchResults(String pattern, int javaType, IJavaSearchScope searchScope) {
		OpenSourceJavaSearchResultRequestor requestor = new OpenSourceJavaSearchResultRequestor();		

		SearchPattern searchPattern = getSearchPattern(pattern, javaType, requestor);
		
		if (searchPattern != null)
		{
			// Create a search engine and search the workspace for the selected object.
			SearchEngine searchEngine = new SearchEngine();
			
			// Perform the search
			try
			{
				searchEngine.search(searchPattern, new SearchParticipant[] {SearchEngine.getDefaultSearchParticipant()},
						getJavaSearchScope(searchScope), requestor, new NullProgressMonitor());				
			}
			catch (CoreException e) {
				// This error will result in falling into the openSourceFailed code below
			}
			return requestor;
		}
		return null;
	}
	
	/**
	  * OpenSourceJavaSearchResultCollector is responsible for handling the search results for finding 
	  * the source and source range for the given selected object.  An instance of this class is created and
	  * passed to the SearchEngine.
	  */
	private static class OpenSourceJavaSearchResultRequestor extends SearchRequestor
	{
		private Vector _matches;
		private InnerClassType _innerClassType;		

		public OpenSourceJavaSearchResultRequestor()
		{
			super();
			_matches = new Vector();
			_innerClassType = null;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jdt.core.search.SearchRequestor#acceptSearchMatch(org.eclipse.jdt.core.search.SearchMatch)
		 */
		public void acceptSearchMatch(SearchMatch match) throws CoreException {
			_matches.addElement(match);
		}
		
		public int getNumberOfMatches()
		{
			return _matches.size();
		}
		
		public SearchMatch getMatch(int i)
		{
			return (SearchMatch)_matches.elementAt(i);
		}
		
		public SearchMatch getFirstMatch()
		{
			if (getNumberOfMatches() > 0)
				return getMatch(0);
			
			return null;
		}
		
		public SearchMatch getMatchForPattern(String pattern)
		{
			if (getNumberOfMatches() > 1)
			{
				try
				{
					IJavaElement je = null;
					IMethod me = null;		
					
					for (int i = 0; i < getNumberOfMatches(); i++)
					{
						je = (IJavaElement)getMatch(i).getElement();
						
						if (je instanceof IMethod)
						{
							me = (IMethod)je;
							
							if (!me.isConstructor() && pattern.indexOf(me.getDeclaringType().getFullyQualifiedName()) >= 0)
								return getMatch(i);
						}
					}
				}
				catch (Exception e)
				{
					HyadesUIPlugin.logError(e);
				}
			}
			
			return getFirstMatch();
		}		
		
		public static IFile getFile(SearchMatch match)
		{
			if (match != null && match.getResource() instanceof IFile)
				return (IFile) match.getResource();
			
			return null;
		}
		
		public boolean hasInnerClassType()
		{
			return _innerClassType != null;
		}

		public InnerClassType setInnerClassType(InnerClassType innerClassType)
		{
			return _innerClassType = innerClassType;
		}			
		
		public InnerClassType getInnerClassType()
		{
			return _innerClassType;
		}		
	}

	/**
	 * Represents an inner class (named or unnamed/anonymous).
	 */
	private static class InnerClassType	{
		
		private String _name;
		private String _method;
		private int _anonymousNumber;
		private InnerClassType _child;
		
		/**
		 * Constructs a new inner class type from the given class name, which
		 * contains a '$' and either the inner class name, or number for anonymous
		 * inner classes.
		 * 
		 * @param innerClassTypeStr the full class name, including '$'.
		 */
		public InnerClassType(String innerClassTypeStr)	{
			super();
			
			_child = null;
			_method = null;
			_anonymousNumber = -1;
			
			int idx = innerClassTypeStr.indexOf("$");
			int bidx = innerClassTypeStr.indexOf("(");					
			
			if ((bidx < 0 || bidx > idx) && idx >= 0) {
				_name = innerClassTypeStr.substring(0, idx);
				_child = new InnerClassType(innerClassTypeStr.substring(idx+1));
			}
			else {
				idx = innerClassTypeStr.indexOf(".");
				
				if ((bidx < 0 || bidx > idx) && idx >= 0) {
					_name = innerClassTypeStr.substring(0, idx);
					_method = innerClassTypeStr.substring(idx+1);
				}
				else if (bidx >= 0)	{
					_name = innerClassTypeStr.substring(0, bidx);
					
					int bidx2 = innerClassTypeStr.indexOf(")");
					String parms = innerClassTypeStr.substring(bidx+1, bidx2);
					int idx2 = parms.indexOf(",");
					
					if (idx2 >= 0) {
						parms = parms.substring(idx2+1).trim();
					}
					else {
						parms = "";
					}
					_method = _name + "(" + parms + ")"; 
				}
				else {
					_name = innerClassTypeStr;
				}
			}
			
			try {
				int number = Integer.parseInt(_name);
				_anonymousNumber = number;
			}
			catch (NumberFormatException ne) {
				_anonymousNumber = -1;
			}
		}
		
		/**
		 * Returns whether or not this inner class is anonymous (unnamed).
		 * 
		 * @return true iff the inner class has no name.
		 */
		public boolean isAnonymous() {
			return _anonymousNumber > -1;
		}
		
		/**
		 * Returns which anonymous inner class this one is. These are numbered
		 * starting from 1 based on where they are in the source code.
		 * 
		 * @return the inner class number, starting from 1.
		 */
		public int getAnonymousNumber() {
			return _anonymousNumber;
		}		
		
		/**
		 * Returns whether or not the inner class has a method.
		 * 
		 * @return true iff the inner class has a method.
		 */
		public boolean hasMethod() {
			return _method != null;
		}
		
		/**
		 * Returns the method name, if a method is present.
		 * 
		 * @return the method name if present.
		 */
		public String getMethod() {
			return _method;
		}
		
		/**
		 * Returns the inner class's name, or null if it has none.
		 * 
		 * @return the inner class name.
		 */
		public String getName() {
			return _name;
		}

		/**
		 * Returns whether this inner class has yet another inner class within it.
		 * 
		 * @return whether this inner class has yet another inner class within it.
		 */
		public boolean hasChildClass() {
			return _child != null;
		}
		
		/**
		 * Returns the inner class contained in this inner class, if one exists.
		 * Otherwise, returns null.
		 * 
		 * @return the inner class within this inner class, or null if doesn't exist.
		 */
		public InnerClassType getChildClass() {
			return _child;
		}
	}
	
	private static IJavaElement getJavaElementToOpen(String pattern, OpenSourceJavaSearchResultRequestor requestor)
	{
		SearchMatch match = requestor.getMatchForPattern(pattern);
		
		IJavaElement je = (IJavaElement)match.getElement();
		
		if (requestor.hasInnerClassType())
		{
			InnerClassType innerClassType = requestor.getInnerClassType();
			
			IType innerClass = getInnerType(je, innerClassType);
			IMethod method = null;
			
			while (innerClassType.hasChildClass()) 
			{
				innerClassType = innerClassType.getChildClass();
				innerClass = getInnerType(innerClass, innerClassType);
			}
			
			if (innerClassType.hasMethod())
				method = getMethodFromType(innerClass, innerClassType.getMethod());
			
			if (method != null)
				je = method;
			else if (innerClass != null)
				je = innerClass;
			else
				je = null;
		}
		

		return je;
	}
	

	/**
	 * openWbSource is used to open the specified source in the text editor of the Java Perspective.
	 * In addition, it moves the source to the start of the specified source region and highlights
	 * the region.
	 * @param match  org.eclipse.jdt.core.search.SearchMatch - A SearchMatch object that this method will open the source on.
	 * @return boolean - true if the source was opened successfully, false otherwise.
	 */
	private static boolean openWbSource(String pattern, OpenSourceJavaSearchResultRequestor requestor, boolean switchToJavaPerspective)
	{
		IJavaElement je = getJavaElementToOpen(pattern, requestor);
		
		if (je == null)
			return false;
		
		IWorkbenchPage page = null;
	
		if(switchToJavaPerspective)
			page = getJavaActivePage();
		else
			page = UIUtil.getActiveWorkbenchPage();
		
		if(page == null)
			return false;
		
		try
		{
			StructuredSelection ss = new StructuredSelection(je);	
			
			OpenAction openAction = new OpenAction(page.getActivePart().getSite());
			openAction.run(ss);
			
			return true;
		}catch (Exception e)
		{
			HyadesUIPlugin.logError(e);
		}
				
		return false;
	}
	
	/**
	 * Returns the parameters from the given method signature.
	 * 
	 * @param signature the method signature containing the parameters.
	 * @return an array of parameter types.
	 */
	private static String[] getParamatersFromMethodSignature(String signature)
	{
		int bIdx1 = signature.indexOf('(');
		int bIdx2 = signature.indexOf(')');		
		
		String parms = signature.substring(bIdx1+1, bIdx2);
		if (parms.length() == 0)
			return new String [0];
		
		Vector pTypes = new Vector();
		
		int cIdx = parms.indexOf(',');
		
		while (cIdx >= 0)
		{
			pTypes.addElement(parms.substring(0, cIdx).trim());
			parms = parms.substring(cIdx+1);
			cIdx = parms.indexOf(',');
		}
		pTypes.addElement(parms.trim());
		
		String [] result = new String[pTypes.size()];
		
		for (int i = 0; i < result.length; i++)
			result[i] = pTypes.get(i).toString();
		
		
		return result;
	}
	
	private static IMethod getMethodFromType(IType type, String signature)
	{
		String name = signature.substring(0, signature.indexOf("("));
		String [] parameterTypes = getParamatersFromMethodSignature(signature);
		
		try
		{
			IMethod[] methods = type.getMethods();
			for (int i = 0; i < methods.length; i++)
			{
				if (methods[i].getElementName().equals(name)
						&& parametersAreEqual(methods[i].getParameterTypes(), parameterTypes))
				{
					return methods[i];
				}
	
			}
		}catch (Exception e)
		{
			HyadesUIPlugin.logError(e);
		}
		return null;
		
	}
	
	/**
	 * Returns whether or not the given sets of parameters are the equal.
	 * 
	 * @param parameters1 the first set of parameters.
	 * @param parameters2 the second set of parameters.
	 * @return whether or not the parameter sets are equal.
	 */
	private static boolean parametersAreEqual(String[] parameters1, String[] parameters2) {
		if (parameters1.length != parameters2.length)
			return false;
		
		boolean parametersMatch = true;
		for (int i = 0; i < parameters1.length; i++)
		{
			if (!Signature.toString(parameters1[i]).equals(parameters2[i]))
				parametersMatch = false;
		}
		
		if (parametersMatch)
			return true;
		
		for (int i = 0; i < parameters1.length; i++)
		{
			if (!Signature.toString(parameters1[i]).equals(Signature.getSimpleName(parameters2[i])))
				return false;
		}		
		
		return true;
	}
	
	/**
	 * Returns the specified inner class from the given java element.
	 * 
	 * @param je the java element containing the inner class.
	 * @param innerClassType the inner class to get.
	 * @return an IType representing the inner class.
	 */
	private static IType getInnerType(IJavaElement je, InnerClassType innerClassType) {
		try {
			if (je instanceof SourceType) {
				SourceType type = (SourceType) je;
				if (type.isAnonymous() == innerClassType.isAnonymous()
						&& ((innerClassType.isAnonymous() && innerClassType.getAnonymousNumber() == type.occurrenceCount)
								|| (!innerClassType.isAnonymous() && type.getElementName().equals(innerClassType.getName()))))
					return type;
			}
			
			if (je instanceof IParent) {
				IParent parentElement = (IParent)je;
				IJavaElement[] children = parentElement.getChildren();
				
				for (int i = 0; i < children.length; i++) {
					IType result = getInnerType(children[i], innerClassType);
					if (result != null)
						return result;
				}
			}
		}
		catch (Exception e) {
			HyadesUIPlugin.logError(e);
		}
		return null;
	}

	/**
	 * getJavaActivePage is used to locate and return the workbench page for the Java Perspective.
	 */
	private static IWorkbenchPage getJavaActivePage()
	{
		IWorkbenchPage page = UIUtil.getActiveWorkbenchPage();
		if (page == null || !page.getPerspective().getId().equals(JAVA_PERSPECTIVE_ID))
		{
			final IWorkbenchWindow dwindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
			page = null;

			try
			{
				IWorkbenchPage[] persps = dwindow.getPages();
				for (int idx = 0; idx < persps.length; idx++)
				{
					if (persps[idx].getPerspective().getId().equals(JAVA_PERSPECTIVE_ID))
					{
						//trace perspective
						page = persps[idx];

						dwindow.setActivePage(page);
						break;
					}
				}

				if (page == null)
				{
					IAdaptable element = ResourcesPlugin.getWorkspace().getRoot();
					IWorkbench workBench = dwindow.getWorkbench();
					if (workBench != null && element != null)
						page = workBench.showPerspective(JAVA_PERSPECTIVE_ID, dwindow, element);
				}
			}
			catch (Exception exc)
			{
				HyadesUIPlugin.logError(exc);
			}
		}

		return page;
	}	
}