/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: UIUtil.java,v 1.10 2005/03/19 00:47:44 curtispd Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.FileEditorInput;
import org.osgi.framework.Bundle;

/**
 * Contains UI utility methods.
 * 
 * @author marcelop
 * @author popescu
 * @since 0.0.1
 */
public class UIUtil
{
	/**
	 * Returns an ImageDescriptor whose path relative to the plugin described 
	 * by <code>pluginDescriptor</code> is <code>subdirectoryAndFilename</code>.
	 * Returns <code>null</code>if no image could be found.
	 *
	 * <p>This method is convenience and only intended for use by the workbench 
	 * because it explicitly uses the workbench's registry for caching/retrieving 
	 * images from other extensions -- other plugins must user their own registry. 
	 * This convenience method is subject to removal.
	 *
	 * <p>Note: subdirectoryAndFilename must not have any leading "." or path 
	 * separators / or \ ISV's should use icons/mysample.gif and not 
	 * ./icons/mysample.gif.
	 *
	 * <p>Note:  This consults the plugin for extension and obtains its installation 
	 * location.  All requested images are assumed to be in a directory below and 
	 * relative to that plugins installation directory.
	 */
	public static ImageDescriptor getImageDescriptorFromPlugin(Bundle pluginBundle, String subdirectoryAndFilename)
	{
		URL path = pluginBundle.getEntry("/");
		URL fullPathString = null;
		try
		{
			fullPathString = new URL(path,subdirectoryAndFilename);
			return ImageDescriptor.createFromURL(fullPathString);
		}
		catch (MalformedURLException e)
		{
			HyadesUIPlugin.logError(e);
		}
		
		return null;
	}
	
	/**
	 * Fires the current selection of a given structured viewer.
	 * @param structuredViewer
	 * @return <code>true</code> if the selection was fired or <code>false</code>
	 * otherwise. 
	 */
	public static boolean fireCurrentSelection(StructuredViewer structuredViewer)
	{
		SelectionChangedEvent event = new SelectionChangedEvent(structuredViewer, structuredViewer.getSelection());
		try
		{
			CoreUtil.invokeObjectMethod(structuredViewer, "fireSelectionChanged", new Object[]{event}, true);
		}
		catch(RuntimeException e)
		{
			HyadesUIPlugin.logError(e);
			return false;
		}
		
		return true;
	}
	
	/**
	 * Calculates the size of one line of text in a composite. 
	 * @param parent
	 * @return int
	 */
	public static int availableRows(Composite parent)
	{
		int fontHeight = (parent.getFont().getFontData())[0].getHeight();
		int displayHeight = parent.getDisplay().getClientArea().height;

		return displayHeight / fontHeight;
	}
	
	/**
	 * Returns the opened editor associated with a given IFile.
	 * @param inputFile
	 * @return IEditorPart
	 */
	public static IEditorPart getOpenEditor(IFile inputFile)
	{
		IWorkbenchPage activePage = getActiveWorkbenchPage();
		if(activePage == null)
			return null;
			
		IEditorReference[] editorReferences = activePage.getEditorReferences();
		for(int i=0, length=editorReferences.length; i<length; i++)
		{
			IEditorPart editorPart = editorReferences[i].getEditor(false);
			if(editorPart == null)
				continue;
				
			Object input = editorPart.getEditorInput();
			if((input != null) && (input instanceof IFileEditorInput))
			{
				IFile file = ((IFileEditorInput)input).getFile();
				if((file != null) && (inputFile != null) && (file.equals(inputFile)))
					return editorPart;
				
				if((file == null) && (inputFile == null))
					return editorPart;
			}
		}

		return null;
	}
	
	/**
	 * Returns an active workbench page.
	 * @return IWorkbenchPage
	 */
	public static IWorkbenchPage getActiveWorkbenchPage()
	{
		IWorkbench workbench = PlatformUI.getWorkbench();
		IWorkbenchPage activePage = null;
		if(workbench.getActiveWorkbenchWindow() != null)
		{
			activePage = workbench.getActiveWorkbenchWindow().getActivePage();
		}
		else if(workbench.getWorkbenchWindows().length > 0)
		{
			activePage = workbench.getWorkbenchWindows()[0].getActivePage();
		}
		
		return activePage;
	}
	
	/**
	 * Returns all the nodes that are currently visible in the given tree
	 * viewer. An element is visible if its parent and all ancestors are
	 * expanded. This is useful for maintaining the expanded state of a tree
	 * after performing an operation that collapses it.
	 * 
	 * @param viewer The viewer containing the tree of elements.
	 * @return All the visible elements showing in the tree.
	 */
	public static Set getVisibleViewerNodes(TreeViewer viewer) {
		Set set = new HashSet();
		ITreeContentProvider content = (ITreeContentProvider)viewer.getContentProvider();
		
		// Recurse down all the top-level elements
		Object[] top = content.getElements(viewer.getInput());
		for (int i=0;i<top.length;++i) {
			collectVisibleElements(viewer, content, top[i], set);
		}
		return set;
	}

	/**
	 * Opens the editor for the specified file.
	 * 
	 * <p>If <code>editorId</code> is null the default editor is opened.  
	 * If substring is true then the <code>editorId</code> value is considered to be 
	 * first part of the editor id.
	 * 
	 * @param file;
	 * @param editorId;
	 * @param substring;
	 * @return The opened editor part or <code>null</code> if this method was not
	 * able to open the editor.
	 */
	public static IEditorPart openEditor(IFile file, String editorId, boolean substring)
	{
		if(file == null)
			return null;
			
		IWorkbenchPage activePage = getActiveWorkbenchPage();
		if(activePage == null)
			return null;
		
		if("".equals(editorId))
			editorId = null;
		
		try
		{
			if(editorId == null)
			{
				return IDE.openEditor(activePage, file, true);
			}
			else
			{
				IWorkbench workbench = PlatformUI.getWorkbench();
				IEditorDescriptor[] editors = workbench.getEditorRegistry().getEditors(file.getName());
				if(editors != null)
				{
					boolean found = false;
					for(int i=0, length=editors.length; i<length; i++)
					{
						if(substring)
							found = editors[i].getId().startsWith(editorId);
						else
							found = editorId.equals(editors[i].getId());
							
						if(found)
							return IDE.openEditor(activePage, new FileEditorInput(file), editors[i].getId());
					}
				}				
			}
		}
		catch(Exception e)
		{
			HyadesUIPlugin.logError(e);
		}
	
		return null;
	}
	
	/**
	 * Shows an error dialog to present to the user why a file saving attempt
	 * has failed.  
	 * @param filePath
	 * @param throwable
	 */
	public static void openSaveFileErrorDialog(Shell shell, String filePath, Throwable throwable)
	{
		if(filePath == null)
			filePath = "";
		
		
		String error = "";
		if(throwable != null)
		{
			if(throwable.getLocalizedMessage() != null)
				error = throwable.getLocalizedMessage();
			else if(throwable.getMessage() != null)
				error = throwable.getMessage();
		}
		
		String message = HyadesUIPlugin.getString("_ERROR_SAVING", new String[]{filePath, error});
		MessageDialog.openError(shell, HyadesUIPlugin.getString("W_ERROR"), message);
	}
	
	
	/**
	 * Applies all the filters of the specified structured viewer to the element
	 * array.
	 * @param structuredViewer
	 * @param elements
	 * @return Object[]
	 */
	public static Object[] applyFilters(StructuredViewer structuredViewer, Object parent, Object[] elements)
	{
		if(elements.length == 0)
			return elements;
			
		ViewerFilter[] filters = structuredViewer.getFilters();
		for(int i = 0, maxi = filters.length; i < maxi; i++)
		{
			elements = filters[i].filter(structuredViewer, parent, elements);
			if(elements.length == 0)
				return elements;
		}
		return elements;
	}
	
	/**
	 * Returns whether the two selections have the same elements.
	 * @param selection1
	 * @param selection2
	 * @return boolean
	 */
	public static boolean areEquals(ISelection selection1, ISelection selection2)
	{
		if(selection1 == selection2)
			return true;
			
		if((selection1 instanceof IStructuredSelection) && (selection2 instanceof IStructuredSelection))
		{
			List list1 = new ArrayList(((IStructuredSelection)selection1).toList());
			List list2 = new ArrayList(((IStructuredSelection)selection2).toList());
			
			if(list1.size() == list2.size())
			{
				list1.removeAll(list2);
				return list1.isEmpty();
			}
		}
		
		return false;	
	}
	
	public static void integerStyledTextVerifyKey(VerifyEvent event)
	{
		if(event.doit)
		{
			switch(event.keyCode)
			{
				case SWT.ARROW_DOWN:			
				case SWT.ARROW_UP:
				case SWT.ARROW_LEFT:
				case SWT.ARROW_RIGHT:
				case SWT.HOME:
				case SWT.END:
					return;
			}
			
			switch(event.character)
			{
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case SWT.BS:
				case SWT.DEL:
					return;

				default:
					event.doit = false;
					return;
			}
		}		
	}

	/**
	 * Expands the tree contained in the given viewer in such a way that all
	 * the given elements are visible. This is useful for maintaining the
	 * expanded state of a tree after performing an operation that collapses
	 * it.
	 * 
	 * @param viewer The viewer containing the tree of elements.
	 * @param elements The elements to be made visible.
	 */
	public static void setVisibleViewerNodes(TreeViewer viewer, Set set) {
		/*
		 * This is necessary to force the creation of all the tree nodes. You
		 * cannot expand a node that hasn't been seen yet; this is a limitation
		 * of the JFace TreeViewer.
		 */
		viewer.expandAll();
		viewer.collapseAll();
		
		/*
		 * For each previously visible node, traverse through each top level
		 * node in the tree to search for it, while expanding its ancestors
		 * along the way, in order to make it visible.
		 */
		Iterator iter = set.iterator();
		while (iter.hasNext()) {
			Object obj = iter.next();
			ITreeContentProvider content = (ITreeContentProvider)viewer.getContentProvider();
			Object[] top = content.getElements(viewer.getInput());
			for (int i=0;i<top.length;++i) {
				hasElement(viewer, content, top[i], obj);
			}
		}
	}

	/**
	 * This method validates the file name to determine if it conforms to 
	 * OS file name restrictions.
	 * 
	 * At this point we are testing for valid file names on Windows and Linux 
	 * systems only.  In the future more system restrictions may need to be 
	 * added.
	 * 
	 * @param valueStr The file name string being passed in
	 * @return The invalid character or null 
	 */
	public static String validateOSFileName(String valueStr) {
		if (valueStr.startsWith("-")) {
			return new Character(valueStr.charAt(0)).toString();
		}

		for (int i = 0; i < valueStr.length(); i++) {
			char ch = valueStr.charAt(i);
			switch (ch) {
				case '<':
					return new Character(ch).toString();
				case '>':
					return new Character(ch).toString();
				case '+':
					return new Character(ch).toString();
				case '|':
					return new Character(ch).toString();
				case '*':
					return new Character(ch).toString();
				case '{':
					return new Character(ch).toString();
				case '}':
					return new Character(ch).toString();
				case '[':
					return new Character(ch).toString();
				case ']':
					return new Character(ch).toString();
				case ':':
					return new Character(ch).toString();
				case ';':
					return new Character(ch).toString();
				case ',':
					return new Character(ch).toString();
				case '"':
					return new Character(ch).toString();
				case '\'':
					return new Character(ch).toString();
				case '/':
					return new Character(ch).toString();
				case '\\':
					return new Character(ch).toString();
				case '`':
					return new Character(ch).toString();
				case '~':
					return new Character(ch).toString();
				case '=':
					return new Character(ch).toString();
				case '!':
					return new Character(ch).toString();
				case '?':
					return new Character(ch).toString();
				default:
					;
			}
		}
		return null;
}
	
	/**
	 * This method validates the file name does not contain any invalid
	 * characters determined by Java Naming Conventions.
	 * @param fileName The name of the file 
	 * @return The invalid character or null
	 */
	public static String validateJavaFileName(String valueStr) {
		if (Character.isDigit(valueStr.charAt(0))) {
			return new Character(valueStr.charAt(0)).toString();
		}

		for (int i = 0; i < valueStr.length(); i++) {
			char ch = valueStr.charAt(i);
			switch (ch) {
				case '!':
					return new Character(ch).toString();
				case '@':
					return new Character(ch).toString();
				case '#':
					return new Character(ch).toString();
				case '^':
					return new Character(ch).toString();
				case '\'':
					return new Character(ch).toString();
				case '`':
					return new Character(ch).toString();
				case '<':
					return new Character(ch).toString();
				case '>':
					return new Character(ch).toString();
				case '-':
					if (i != 0) { 
						return new Character(ch).toString();
					}
				case '+':
					return new Character(ch).toString();
				case ' ':
					return "<space>";
				case '|':
					return new Character(ch).toString();
				case '*':
					return new Character(ch).toString();
				case '{':
					return new Character(ch).toString();
				case '}':
					return new Character(ch).toString();
				case '[':
					return new Character(ch).toString();
				case ']':
					return new Character(ch).toString();
				case '(':
					return new Character(ch).toString();
				case ')':
					return new Character(ch).toString();
				case ':':
					return new Character(ch).toString();
				case ';':
					return new Character(ch).toString();
				case ',':
					return new Character(ch).toString();
				case '"':
					return new Character(ch).toString();
				case '/':
					return new Character(ch).toString();
				case '\\':
					return new Character(ch).toString();
				case '~':
					return new Character(ch).toString();
				case '=':
					return new Character(ch).toString();
				case '?':
					return new Character(ch).toString();
				default:
					;
			}
		}
		return null;
	}
	
	/**
	 * Recursively finds and collects all the visible elements in the viewer.
	 * An element is visible if its parent and all ancestors are expanded.
	 * 
	 * @param viewer The tree viewer containing the tree.
	 * @param content The content provider for the viewer.
	 * @param current The current element in the tree being traversed.
	 * @param set The set of visible elements to collect the elements into.
	 */
	private static void collectVisibleElements(TreeViewer viewer, ITreeContentProvider content, Object current, Set set) {
		set.add(current);
		if (viewer.getExpandedState(current)) {
			Object[] children = content.getChildren(current);
			for (int i=0;i<children.length;++i) {
				collectVisibleElements(viewer, content, children[i], set);
			}
		}
	}
	
	/**
	 * Returns whether or not the target node is a descendant of the current node,
	 * and if so, expands the current node. This is used to expand the tree in such
	 * a way that the target node is visible.
	 * 
	 * @param viewer The tree viewer containing the tree to traverse.
	 * @param content The viewer's content provider.
	 * @param current The current node being traversed.
	 * @param target The node being searched for and made visible.
	 * @return Whether or not the target node is a descendant of the current node.
	 */
	private static boolean hasElement(TreeViewer viewer, ITreeContentProvider content, Object current, Object target) {
		if (current == target) {
			return true;
		}
		else {
			// It has the target if one of its children has it
			Object[] children = content.getChildren(current);
			boolean found = false;
			for (int i=0;i<children.length;++i) {
				if (hasElement(viewer, content, children[i], target)) {
					found = true;
				}
			}
			// Expand the current node if it contains the target
			if (found) {
				viewer.setExpandedState(current, true);
			}
			return found;
		}
	}
}