/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LocationPage.java,v 1.12 2005/03/08 10:47:52 dguilbaud Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.wizard;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.dialogs.WizardNewFileCreationPage;
import org.eclipse.ui.internal.ide.IDEWorkbenchMessages;

/**
 * Extension of WizardNewFileCreationPage that allows the definition of a
 * required file extension to be used.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class LocationPage extends WizardNewFileCreationPage {
    private String fileExtension;
    private boolean allowOverwrite;

    /**
     * Constructor for LocationPage
     * @param pageName
     * @param selection
     */
    public LocationPage(String pageName, IStructuredSelection selection) {
        super(pageName, selection);
        allowOverwrite = false;
    }

    public void setAllowOverwrite(boolean allowOverwrite) {
        this.allowOverwrite = allowOverwrite;
    }

    public boolean allowOverwrite() {
        return allowOverwrite;
    }

    /**
     * Sets the required file extension.
     * @param fileExtension
     */
    public void setFileExtension(String fileExtension) {
        this.fileExtension = fileExtension;
    }

    /**
     * Returns the required file extension.
     * @return Sting
     */
    public String getFileExtension() {
        return fileExtension;
    }

    /**
     * @see org.eclipse.ui.dialogs.WizardNewFileCreationPage#validatePage()
     */
    protected boolean validatePage() {
        setMessage(null);
        //- Displaying a message (information or warning) directly after having had an error does not work without that:
        setErrorMessage(null);

        if (!allowOverwrite()) {
            IPath path = getContainerFullPath();
            if (path != null) {
                String fileName = getFileName();
                if (fileName.length() == 0 || fileName.equals("."+this.getFileExtension())) {
                    setErrorMessage(HyadesUIPlugin.getString("_ERROR_MIS_NAME")); //$NON-NLS-1$
                    return false;
                } else if (fileName.startsWith(".")) { //$NON-NLS-1$
                    setErrorMessage(HyadesUIPlugin.getString("_ERROR_FILE_NAME_EXT")); //$NON-NLS-1$
                    return false;
                }
                /*
                 * Test for any invalid characters within the file name determined
                 * by the Operating System.
                 */
                String invalidChar = UIUtil.validateOSFileName(fileName);
                if (invalidChar != null) {
                    if (Character.isDigit(invalidChar.charAt(0))) {
                        String[] errors = { HyadesUIPlugin.getString("LOGICAL_NAME"), HyadesUIPlugin.getString("NUMERIC_VALUE")}; //$NON-NLS-1$ //$NON-NLS-2$
                        setErrorMessage(HyadesUIPlugin.getString("_ERROR_NAME_EXT", errors)); //$NON-NLS-1$
                        return false;
                    } else if (invalidChar.trim().equals("-")) { //$NON-NLS-1$
                        String[] errors = { HyadesUIPlugin.getString("LOGICAL_NAME"), "-"}; //$NON-NLS-1$//$NON-NLS-2$
                        setErrorMessage(HyadesUIPlugin.getString("_ERROR_NAME_EXT", errors)); //$NON-NLS-1$
                        return false;
                    } else {
                        String[] errors = { HyadesUIPlugin.getString("LOGICAL_NAME"), invalidChar}; //$NON-NLS-1$
                        setErrorMessage(HyadesUIPlugin.getString("_ERROR_NAME_SPECIAL_CHAR", errors)); //$NON-NLS-1$
                        return false;
                    }
                }

                IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(path.append(fileName));
                if ((resource != null) && (resource.exists())) {
                    setErrorMessage(IDEWorkbenchMessages.format("ResourceGroup.nameExists", new String[] { fileName})); //$NON-NLS-1$);
                    return false;
                }

                String fileExtension = getFileExtension();
                if(fileExtension != null) {
	                int lastIndexOfExtension = fileName.lastIndexOf(fileExtension);
	                if (lastIndexOfExtension > 0) {
	                    String name = fileName.substring(0, lastIndexOfExtension - 1);
	                    if (name.endsWith("." + fileExtension)) { //$NON-NLS-1$
	                        setMessage(HyadesUIPlugin.getString("_WARNING_FILE_NAME", fileName), IStatus.WARNING); //$NON-NLS-1$
	                        return true;
	                    }
	                }
                }
            } else {
                setErrorMessage(HyadesUIPlugin.getString("_ERROR_MIS_PROJECT")); //$NON-NLS-1$
                return false;            	
            }
        }

        if (super.validatePage()) return true;

        return false;
    }

    /**
     * @see org.eclipse.ui.dialogs.WizardNewFileCreationPage#createAdvancedControls(org.eclipse.swt.widgets.Composite)
     * 
     * @author pnedelec
     */
    protected void createAdvancedControls(Composite parent) {
        return;
    }

    /**
     * @see org.eclipse.ui.dialogs.WizardNewFileCreationPage#getNewFileLabel()
     * 
     * @author pnedelec
     */
    protected String getNewFileLabel() {
        return HyadesUIPlugin.getString("LBL_NAME"); //$NON-NLS-1$
    }

    /**
     * @see org.eclipse.ui.dialogs.WizardNewFileCreationPage#validateLinkedResource()
     * 
     * @author pnedelec
     */
    protected IStatus validateLinkedResource() {
        //- Because the Advanced part is missing for this page, without stubbing this method a NPE would be thrown.
        return new Status(IStatus.OK, HyadesUIPlugin.getID(), IStatus.OK, "", null); //$NON-NLS-1$
    }

    /**
     * @see org.eclipse.ui.dialogs.WizardNewFileCreationPage#getFileName()
     * 
     * @author pnedelec
     */
    public String getFileName() {
        //- Bugzilla #82262
    	String name = super.getFileName();
    	// check if file extension is null. bugzilla 84626
    	if(fileExtension != null)
    		name += "." + fileExtension; //$NON-NLS-1$
    	return name;
    }

    /**
     * Returns the text displayed in the Name field.
     * 
     * @author pnedelec
     */
    public String getItemName() {
        return super.getFileName();
    }

}