/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CreationWizard.java,v 1.12 2005/04/30 04:02:36 bjiang Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.wizard.exampleproject;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.actions.WorkspaceModifyDelegatingOperation;
import org.eclipse.ui.dialogs.IOverwriteQuery;
import org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard;
import org.eclipse.ui.wizards.newresource.BasicNewResourceWizard;

import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.util.JavaUtil;
import org.eclipse.hyades.ui.internal.util.ResourceUtil;
import org.eclipse.hyades.ui.internal.util.StringUtil;
import org.eclipse.hyades.ui.internal.util.UIUtil;

/**
 * This wizard is designed to simplify the creation of project samples in Eclipse.
 * Most of the actions to be performed by this class can be defined in the
 * plugin registry when this wizard is defined.
 * 
 * <p>This is a sample of an extension point definition that uses this plugin
 * 
	<extension point="org.eclipse.ui.newWizards">
		<category
			id = "myCategory"
			parentCategory="org.eclipse.ui.Examples"
			name="%theNameOfMyCategory">
		</category>
	
		<wizard 
			id="idOfTheWizardThatCreatesMyExampleProject"
			name="%myWizardTitle"
			windowTitle="%windowTitle"
			class="me.MyWizard"
			category="org.eclipse.ui.Examples/myCategory"
			project="true"
			finalPerspective="idOfThePerspectiveThatShouldBeOpenedAfterCreatingTheProject"
			icon="icons/iconOfMyWizard.gif"
			banner="icons/bannerToBeOnTopOfMyWizard.gif">
			
			<description>%theDescriptionOfTheProjectToBeCreated</description>
			
			<!-- Here I am defining some values for JDT classpath variables -->
			<!-- This plugin supports some ${nn} variables as indicated after this sample --> 
			<classpathVariable name="A_JDT_VAR" value="e:\temp" overwrite="false"/>
			<classpathVariable name="MY_PLUGIN_DIR" value="${plugins}/myPlugin" overwrite="true"/>

			<!-- Each <i>projectsetup</i> element correspond to one project creation page. -->
			<!-- So a given Example Project Create Wizard can actually create more than one project.  -->
			<projectsetup
				pagetitle="%theTitleOfTheFirstProjectCreationPage"
				pagedescription="%page1Description"
				name="The Initial Name of The Project"
				helpId="me.ctst0310">
				
				<!-- Defines which file will be presented after the wizard creation -->
				<open file="readme.html" editorId="com.ibm.etools.webbrowser"/>
				
				<!-- Defines where the example project content is located.  -->
				<!-- The files are supposed to be located in the plugin defining this extension point. -->
				<import dest="" src="archive/httpSample/object"/>
			</projectsetup>

			<!-- This project is a more complicated -->
			<!-- It has a nature id and a builder --> 
			<projectsetup
				pagetitle="%sampleProjects.manual.page1.Name"
				pagedescription="%sampleProjects.manual.page1.Description"
				name="The initial name of this Java project"
				helpId="me.ctst123">
				
				<project
					label="test"
					name="JavaFiles">
					
					<open file="folder1/welcome.html" editorId="com.ibm.editor1"/>
					<open file="folder1/readme.html"/>
					<import dest="" src="archive/manualSample.zip"/>

					<nature id="org.eclipse.jdt.core.javanature"/>
					<buildCommand name="org.eclipse.jdt.core.javabuilder"/>
				</project>
			</projectsetup>
		</wizard>
	</extension>
	
 * 
 * <p>The ${nn} variables available are 
 *		${plugins} 			- Parent directory of the plugin that declares the extension point
 * 		${base.plugins}		- Parent directory of the "org.eclipse.boot" plugin
 * 		${pluginId="xxx"} 	- Directory of the xxx plugin (there should be no spaces before and after the equal signal
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class CreationWizard 
extends BasicNewResourceWizard implements INewWizard, IExecutableExtension
{
	private CreationWizardPage[] pages;
	private IConfigurationElement wizardConfigurationElement;
	private IConfigurationElement sampleConfigurationElement;
	private CreationOperation runnable;
	private IProjectAdjuster adjuster;
	private boolean disableOpenResources;
	
	/**
	 * Implementation of the IOverwriteQuery interface.
	 */
	private class ImportOverwriteQuery 
	implements IOverwriteQuery
	{
		/**
		 * @see org.eclipse.ui.dialogs.IOverwriteQuery#queryOverwrite(java.lang.String)
		 */
		public String queryOverwrite(String file)
		{
			String[] returnCodes = { YES, NO, ALL, CANCEL };
			int returnVal = openDialog(file);
			return returnVal < 0 ? CANCEL : returnCodes[returnVal];
		}

		/**
		 * Opens a dialog to confirm if a file should be overwritten.
		 * @param file
		 * @return the dialog result.
		 */
		private int openDialog(final String file)
		{
			final int[] result = { IDialogConstants.CANCEL_ID };
			getShell().getDisplay().syncExec(new Runnable()
			{
				public void run()
				{
					String msg = HyadesUIPlugin.getString("Q_OVERWRITE", file);
					String title = HyadesUIPlugin.getString("W_QUESTION"); 					
					String[] options = {IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL, IDialogConstants.YES_TO_ALL_LABEL, IDialogConstants.CANCEL_LABEL };
					MessageDialog dialog = new MessageDialog(getShell(), title, null, msg, MessageDialog.QUESTION, options, 0);
					result[0] = dialog.open();
				}
			});
			return result[0];
		}
	}

	/**
	 * Constructor for CreationWizard
	 */
	public CreationWizard()
	{
		super();
		setNeedsProgressMonitor(true);
	}
	
	/**
	 * @see org.eclipse.jface.wizard.IWizard#addPages()
	 */
	public void addPages()
	{
		super.addPages();

		disableOpenResources = false;

		if((getWindowTitle() == null) || (getWindowTitle().trim().length() == 0))
		{			
			String windowTitle = HyadesUIPlugin.getString("NEW_WIZARD_TITLE");
			if(windowTitle != null)
				setWindowTitle(windowTitle);
		}

		IConfigurationElement[] children = sampleConfigurationElement.getChildren("projectsetup");
		if (children == null || children.length == 0)
		{
			HyadesUIPlugin.logError(HyadesUIPlugin.getString("_ERROR_DUP_PROJECT_NAME_LOG"));
			return;
		}

		pages = new CreationWizardPage[children.length];

		for (int i = 0; i < children.length; i++)
		{
			CreationWizardPage page = createWizardPage("newProject" + i, children[i]);
			if(page != null)
			{
				pages[i] = page; 
				addPage(pages[i]);
			}
		}
	}
	
	/**
	 * Clients may overwrite this method to provide a customized implementation of the
	 * <code>CreationWizardPage</code>.
	 * @param suggestedName The suggested name for the page to be created.  This is not the 
	 * page title.
	 * @param configurationElement  The configuration element that describes this page.
	 * @return CreationWizardPage  If <code>null</code> then there won't be a page for 
	 * the specified configuration element.
	 */
	protected CreationWizardPage createWizardPage(String suggestedName, IConfigurationElement configurationElement)
	{
		return new CreationWizardPage(this, suggestedName, configurationElement);
	}
	
	/**
	 * Returns the <code>CreationWizardPage</code> pages added to this
	 * wizard.
	 * @return CreationWizardPage
	 */
	protected CreationWizardPage[] getCreationWizardPages()
	{
		return pages;
	}

	/**
	 * @see org.eclipse.jface.wizard.IWizard#createPageControls(Composite)
	 */
	public void createPageControls(Composite pageContainer)
	{
		super.createPageControls(pageContainer);
		
		for (int i = 0; i < pages.length; i++)
			pages[i].setHelpId();
	}

	/**
	 * Returns the adjuster.
	 * @return IBaseExampleProjectCreationAdjuster
	 */
	public IProjectAdjuster getAdjuster()
	{
		return adjuster;
	}

	/**
	 * Sets the adjuster.
	 * @param adjuster The adjuster to set
	 */
	public void setAdjuster(IProjectAdjuster adjuster)
	{
		this.adjuster = adjuster;
		if(runnable != null)
			runnable.setAdjuster(adjuster);
	}

	/**
	 * Adjusts the JDT's classpath variables if required by the extension point.	 *
	 */
	private void adjustClasspathVariable()
	{
		IConfigurationElement[] classpathVars = sampleConfigurationElement.getChildren("classpathVariable");
		for(int i = 0; i < classpathVars.length; i++)
		{
			String name = classpathVars[i].getAttribute("name");
			String value = classpathVars[i].getAttribute("value");
			boolean overwrite = Boolean.valueOf(classpathVars[i].getAttribute("overwrite")).booleanValue();
			
			if((name == null) || (value == null))
				continue;
				
			if(!overwrite)
			{
				if(JavaUtil.getVariableValue(name) != null)
					continue;
			}
			
			value = replaceClasspathVariables(value);
			if(value == null)
				continue;
			
			try
			{
				JavaUtil.setVariableValue(name, value, null);
			}
			catch(Exception e)
			{
				continue;
			}
		}
	}
	
	protected String replaceClasspathVariables(String value)
	{
		value = replaceVariableByPluginDirectory(value, "${plugins}", wizardConfigurationElement.getDeclaringExtension().getNamespace(), true);
		value = replaceVariableByPluginDirectory(value, "${base.plugins}", "org.eclipse.core.boot", true);
		
		String[] pluginIds = getPluginIdsToResolve(value);
		for(int i = 0; i < pluginIds.length; i++)
			value = replaceVariableByPluginDirectory(value, "${pluginId=\""+ pluginIds[i] +"\"}", pluginIds[i], false);
		
		return value;
	}
	
	protected String replaceVariableByPluginDirectory(String text, String var, String pluginId, boolean useParentDir)
	{
		if(text.indexOf(var) >= 0)
		{
			File dir = ResourceUtil.getPluginDirectory(pluginId);
			if(dir != null)
			{
				if(useParentDir)
					dir = dir.getAbsoluteFile().getParentFile();
				if(dir != null)
					text = StringUtil.replace(text, var, dir.getAbsolutePath());
			}
				
		}
		
		return text;
	}
	
	/**
	 * Returns the pluginsId that should be resolved.  These plugins are identified by
	 * {pluginId="xxx"} where xxx is the plugin id.  The variable {pluginId="xxx"} will be
	 * replaced by the plugin directory.
	 * 
	 * @param text where the expression {pluginId="xxx"} should be found
	 * @return String[] the plugin ids to be resolved.
	 */
	protected String[] getPluginIdsToResolve(String text)
	{
		String[] tokens = StringUtil.tokenizer(text, "${pluginId=\"", false);
		
		//tokens.length == 0 - No pluginsId to resolve
		//tokens.length == 1 - No pluginsId to resolve
		//	tokens[0] = from the beginning of the file to the 1st {pluginId="
		if(tokens.length <= 1)
			return new String[0];
			
		Set variables = new HashSet(tokens.length-1);
		for(int i = 1; i < (tokens.length); i++)
		{
			int index = tokens[i].indexOf("\"}");
			if(index < 0)
				continue;
			
			variables.add(tokens[i].substring(0, index));
		}
		
		return (String[]) variables.toArray(new String[variables.size()]);
	}

   /**
    * finishWork is a callback function that is called in the performWork method.  This function is called 
    * after copying the resources to the workspace and before any of the resources are opened in an associated
    * editor.  This is necessary to process the resources before opening up any associated editors.
    * 
    * @return boolean value that determines if the work was performed successfully.
    */
   protected boolean finishWork(){
      return true;
   }
	/**
	 * @see org.eclipse.jface.wizard.IWizard#performFinish()
	 */		
	public boolean performFinish()
	{
		adjustClasspathVariable();
		
		runnable = new CreationOperation(this, new ImportOverwriteQuery());
		runnable.setAdjuster(adjuster);
		
		IRunnableWithProgress op = new WorkspaceModifyDelegatingOperation(runnable);
		try
		{
			getContainer().run(false, true, op);
		}
		catch (InvocationTargetException e)
		{
			handleException(e.getTargetException());
			return false;
		}
		catch (InterruptedException e)
		{
			return false;
		}
		
		if(adjuster != null)
			adjuster.beforeWorspaceRefresh(pages);
			
		BasicNewProjectResourceWizard.updatePerspective(wizardConfigurationElement);

		if(adjuster != null)
			adjuster.afterWorspaceRefresh(pages);
		
		if (disableOpenResources == false) {
			IResource[] resources = runnable.getResourcesToOpen();
			
			finishWork();
			for(int i = 0; i < resources.length; i++)
			{
				if(resources[i] instanceof IFile)
				{
					String editorId = runnable.getEditorId(resources[i]);
					boolean editorOpened = false;
					
					try
					{
						editorOpened = (UIUtil.openEditor((IFile)resources[i], editorId, false) != null);
					}
					catch(Throwable t)
					{
						HyadesUIPlugin.logError(t);
					}				
					if(!editorOpened)
					{
						try
						{
							UIUtil.openEditor((IFile)resources[i], null, false);
						}
						catch(Throwable t)
						{
							HyadesUIPlugin.logError(t);
						}
					}
				}
			}
			if(resources.length > 0)
				selectAndReveal(resources[resources.length-1]);
		}
		runnable.dispose();
		return true;
	}

	private void handleException(Throwable throwable)
	{
		HyadesUIPlugin.logError(throwable);

		String[] options = {IDialogConstants.OK_LABEL};
		new MessageDialog(getShell(), HyadesUIPlugin.getString("W_ERROR"), null, HyadesUIPlugin.getString("_ERROR_UNABLE_CREATE"), MessageDialog.ERROR, options, 0);
	}

	/**
	 * Stores the configuration element for the wizard.  The configuration element 
	 * will be used in <code>performFinish</code> to set the result perspective.
	 */
	public void setInitializationData(IConfigurationElement cfig, String propertyName, Object data)
	{
		wizardConfigurationElement = cfig;
		if(wizardConfigurationElement != null)
		{
			String wizardId = wizardConfigurationElement.getAttribute("id");
			if(wizardId != null)
			{
				IConfigurationElement[] configurationElements = Platform.getExtensionRegistry().getConfigurationElementsFor(HyadesUIPlugin.getID(), HyadesUIPlugin.EP_SAMPLE_WIZARD);
				for(int i=0, max=configurationElements.length; i<max; i++)
				{
					if(!"sampleWizard".equals(configurationElements[i].getName()))
						continue;
					
					if(wizardId.equals(configurationElements[i].getAttribute("wizardId")))
					{	
						sampleConfigurationElement = configurationElements[i];
						break;
					}
				}
			}
		}
		
		if(sampleConfigurationElement == null)
			throw new RuntimeException("This sample doesn't have a Sample Project extension point associated with the wizard.");
	}
	
	/**
	 * Returns the id of the plugin that is declaring this wizard.
	 * @return String
	 */
	protected String getPluginId()
	{
		if(wizardConfigurationElement == null)
			return null;
			
		return wizardConfigurationElement.getDeclaringExtension().getNamespace();
	}
	
	/**
	 * @see org.eclipse.ui.wizards.newresource.BasicNewResourceWizard#initializeDefaultPageImageDescriptor()
	 */
	protected void initializeDefaultPageImageDescriptor()
	{
		String banner= sampleConfigurationElement.getAttribute("banner");
		if(banner != null)
		{
			ImageDescriptor imageDescriptor = UIUtil.getImageDescriptorFromPlugin(Platform.getBundle(wizardConfigurationElement.getDeclaringExtension().getNamespace()), banner);
			if(imageDescriptor != null)
			{
				setDefaultPageImageDescriptor(imageDescriptor);
				return;
			}
		}

		super.initializeDefaultPageImageDescriptor();
	}
	
	/**
	 * Sets whether or not the wizard should open designated resources after the wizard has finished
	 * all of the appropriate actions when the user clicks on the finish button. By default, resources
	 * are to be opened after the wizard has completed its operation. This method provides an override 
	 * to this functionality.
	 * 
	 * @param shouldOpenResource set true if the resources should NOT be opened. Set to false if resources should open (default behaviour).
	 */
	public void disableResourceOpening(boolean disableOpening) {
		disableOpenResources = disableOpening;
	}
}