/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CreationWizardPage.java,v 1.5 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.wizard.exampleproject;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.eclipse.ui.help.WorkbenchHelp;

import org.eclipse.hyades.ui.HyadesUIPlugin;

/**
 * Wizard page for the Example Project Creation Wizard.  This page allows the input of 
 * the project name and checks if this name is already in use.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class CreationWizardPage 
extends WizardNewProjectCreationPage
{
	private IConfigurationElement configurationElement;
	private String initialProjectName;
	private CreationWizard creationWizard;
	
	/**
	 * Constructor for CreationWizardPage.
	 * @param pageName
	 * @param configurationElement
	 */
	public CreationWizardPage(CreationWizard creationWizard, String pageName, IConfigurationElement configurationElement)
	{
		super(pageName);
		this.configurationElement = configurationElement;
		this.creationWizard = creationWizard;

		initialProjectName = getValue(configurationElement, "name");
		if(initialProjectName != null)
			setInitialProjectName(initialProjectName);
			
		setDescription(getValue(configurationElement, "pagedescription"));
		setTitle(getValue(configurationElement, "pagetitle"));
	}

	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#dispose()
	 */
	public void dispose()
	{
		configurationElement = null;
		creationWizard = null;
		
		super.dispose();
	}
	
	protected CreationWizard getCreationWizard()
	{
		return creationWizard;
	}

	/**
	 * Returns the configurationElement.
	 * @return IConfigurationElement
	 */
	public IConfigurationElement getConfigurationElement()
	{
		return configurationElement;
	}
	
	/**
	 * Returns the initialProjectName.
	 * @return String
	 */
	public String getInitialProjectName()
	{
		return initialProjectName;
	}
	
	/**
	 * Sets the help id of this wizard's page.
	 * @return <code>true</code> if the help id was set.
	 */
	protected boolean setHelpId()
	{
		String helpId = configurationElement.getAttribute("helpId");
		if(helpId == null)
			return true;
			
		if(getControl() == null)
			return false;
			
		WorkbenchHelp.setHelp(getControl(), helpId);
		return true;
	}

	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(Composite)
	 */
	public void createControl(Composite parent)
	{
		super.createControl(parent);
		
		Text projectText = getProjectText();
		if(projectText != null)
		{
			projectText.notifyListeners(SWT.Modify, new Event());
			projectText.setSelection(0, projectText.getText().length());
		}
	}
	
	/**
	 * Returns the project's text widget that is defined in this
	 * wizard's superclass.
	 * @return Text
	 */
	private Text getProjectText()
	{
		if(!(getControl() instanceof Composite))
			return null;
			
		if(getProjectName() == null)
			return null;
			
		Composite parent = (Composite)getControl();
		Control[] children = parent.getChildren();
		for(int i = 0; i < children.length; i++)
		{
			if(children[i] instanceof Composite)
			{
				Control[] grandChildren = ((Composite)children[i]).getChildren();
				for(int j = 0; j < grandChildren.length; j++)
				{
					if(grandChildren[j] instanceof Text)
					{
						Text text = (Text)grandChildren[j];
						if(getProjectName().equals(text.getText()))
							return text;
					}
				}
			}
		}
		
		return null;
	}
	
	/**
	 * @see org.eclipse.ui.dialogs.WizardNewProjectCreationPage#validatePage()
	 */
	protected boolean validatePage()
	{
		if(!super.validatePage())
			return false;
			
		String projectName = getProjectName();
		if(projectName == null)
			return false;
			
		IWizard wizard = getWizard();
		if(wizard instanceof CreationWizard)
		{
			IWizardPage[] pages = wizard.getPages();
			for(int i = 0; i < pages.length; i++)
			{
				if((pages[i] != this) && (pages[i] instanceof CreationWizardPage))
				{
					if(projectName.equals(((CreationWizardPage)pages[i]).getProjectName()))
					{
						setErrorMessage(HyadesUIPlugin.getString("_ERROR_DUP_PROJECT_NAME"));
						return false;
					}
				}
			}
		}
		
		return true;
	}
	
	/**
	 * Returns the value of a given attribute in a configuration
	 * element.  If the configuration element doesn't have such attribute then
	 * this method returns an empty string.
	 * @param configurationElement
	 * @param attribute
	 * @return A not null String
	 */
	private String getValue(IConfigurationElement configurationElement, String attribute)
	{
		String value = configurationElement.getAttribute(attribute);
		if (value == null)
			return "";
			
		return value;
	}
}