/**********************************************************************
Copyright (c) 2005 IBM Corporation and others.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Eclipse Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/epl-v10.html
$Id: JavaTaskRemoteHyadesComponentStub.java,v 1.4 2005/02/16 22:20:15 qiyanli Exp $

Contributors:
 IBM Corporation - initial implementation
**********************************************************************/
package org.eclipse.hyades.execution.local;

import org.eclipse.hyades.execution.core.ExecutionComponentStateException;
import org.eclipse.hyades.execution.core.IDataChannel;
import org.eclipse.hyades.execution.core.IDataProcessor;
import org.eclipse.hyades.execution.core.IRemoteHyadesComponent;
import org.eclipse.hyades.execution.core.TimeoutException;
import org.eclipse.hyades.execution.invocation.RemoteInvocationException;
import org.eclipse.hyades.execution.invocation.ReturnData;
import org.eclipse.hyades.internal.execution.local.common.DataProcessor;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;

/**
 * Most of the tasks of this monitoring component are handled on the local side.
 * Most of the tasks, also, are implemented in the stub layer only. For the
 * monitoring operations, an <code>Agent</code> is needed. The Hyades execution
 * core plugin knows nothing about <code>Agent</code>s, and most of the operations
 * provided by this class are only to be implemented on the local side.
 */
public class JavaTaskRemoteHyadesComponentStub	extends ExecutionComponentStub	implements IRemoteHyadesComponent {
	
	private Agent agent;
	private IDataProcessor dataProcessor;
	private boolean monitorRequested=false;
	private IDataProcessor processor;

	public JavaTaskRemoteHyadesComponentStub() {
		super();
	}
	
	/**
	 * Set the agent that will be used to communicate with the remote test
	 * agent. This method is automatically called when this component is
	 * added to a <code>JavaTaskExecutorStub</code> as a child.
	 * 
	 * This method is implemented at the stub layer only; it only makes
	 * sense on the local side.
	 * 
	 * @param agent should be the agent that has the communication channel to
	 *        the remote test agent.
	 * 
	 * @see org.eclipse.hyades.execution.local.JavaTaskExecutorStub.addChild(IExecutionComponent)
	 */
	protected void setAgent(Agent agent) {
		this.agent = agent;
		if(monitorRequested && !isMonitored()) {
			startMonitoring(processor);
		}
	}
	
	/**
	 * This implementation does not delegate to the implementation class. All
	 * agent state information is maintained at the stub level.
	 *  
	 * @see org.eclipse.hyades.execution.core.IRemoteHyadesComponent#isActive()
	 */
	public boolean isActive() {
		return agent.isActive();
	}

	/**
	 * This implementation does not delegate to the implementation class. All
	 * agent state information is maintained at the stub level.
	 *  
	 * @see org.eclipse.hyades.execution.core.IRemoteHyadesComponent#isMonitored()
	 */
	public boolean isMonitored() {
		return agent.isMonitored();
	}

	/**
	 * This method currently does not provide meaningful functionality.
	 * 
	 * @see org.eclipse.hyades.execution.core.IRemoteHyadesComponent#isLocked()
	 */
	public boolean isLocked() {
		/* Delegate to the remote instance */
		ReturnData invokeRtn = delegateRemoteCall(new Class[]{}, new Object[]{}, "isLocked");
		Object rtnValue = invokeRtn.getReturnValue();
		if(invokeRtn.isError()) {
			throw new RemoteInvocationException((Throwable)rtnValue);
		}
		if ( !(rtnValue instanceof Boolean) )
			throw new RemoteInvocationException("Incorrect datatype returned; expected Boolean");
			
		return ((Boolean)rtnValue).booleanValue();
	}

	/**
	 * This method currently does not provide meaningful functionality.
	 * 
	 * @see org.eclipse.hyades.execution.core.IRemoteHyadesComponent#lock(long)
	 */
	public void lock(long timeout) throws TimeoutException {
		ReturnData invokeRtn = delegateRemoteCall(
			new Class[]{Long.TYPE}, new Object[]{new Long(timeout)}, "timeout");
		Object rtnValue = invokeRtn.getReturnValue();
		if(invokeRtn.isError()) {
			throw new RemoteInvocationException((Throwable)rtnValue);
		}
	}

	/**
	 * This method currently does not provide meaningful functionality.
	 * 
	 * @see org.eclipse.hyades.execution.core.IRemoteHyadesComponent#unlock()
	 */
	public void unlock() {
		ReturnData invokeRtn = delegateRemoteCall(new Class[]{}, new Object[]{}, "unlock");
		Object rtnValue = invokeRtn.getReturnValue();
		if(invokeRtn.isError()) {
			throw new RemoteInvocationException((Throwable)rtnValue);
		}
	}

	/**
	 * This implementation always throws an <code>UnsupportedOperationException</code>.
	 * Use <code>startMonitoring(IDataProcessor)</code>.
	 * 
	 * @see org.eclipse.hyades.execution.core.IRemoteHyadesComponent#startMonitoring(org.eclipse.hyades.execution.core.IDataChannel)
	 */
	public void startMonitoring(IDataChannel processor) throws ExecutionComponentStateException {
		throw new UnsupportedOperationException("This operation has not yet been implemented");
	}

	/**
	 * This implementation does not delegate to the implementation class. All
	 * agent state information is maintained at the stub level.
	 *  
	 * @see org.eclipse.hyades.execution.core.IRemoteHyadesComponent#stopMonitoring()
	 */
	public void stopMonitoring() throws ExecutionComponentStateException {
		try {
			agent.stopMonitoring();
		}
		catch ( InactiveAgentException e ){
			throw new ExecutionComponentStateException(NOT_CONFIGURED, "The monitoring agent is not active");
		}
		dataProcessor = null;
	}
	
	/**
	 * This implementation does not delegate to the implementation class. All
	 * agent state information is maintained at the stub level.
	 *  
	 * @see org.eclipse.hyades.execution.core.IRemoteHyadesComponent#startMonitoring(org.eclipse.hyades.execution.core.IDataProcessor)
	 */
	public void startMonitoring(IDataProcessor dataProcessor) throws ExecutionComponentStateException {

		if(agent==null) {
			monitorRequested=true;
			processor=dataProcessor;
			return;
		}
		// We need to do *something* here to tie dataProcessor to the remote
		// side so that it receives "stuff". Is this it??
		if ( !(dataProcessor instanceof DataProcessor) )
			throw new IllegalArgumentException(
			"The data processor must be of type " + DataProcessor.class.getName());
		try {
			agent.startMonitoring((DataProcessor)dataProcessor);
		}
		catch ( InactiveAgentException e ){
			throw new ExecutionComponentStateException(NOT_CONFIGURED, "The monitoring agent is not active");
		}
		this.dataProcessor = dataProcessor;
	}

}
