/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileManagerFactory.java,v 1.9 2005/05/25 20:32:47 bjiang Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.local.file;

import java.util.StringTokenizer;

import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.internal.execution.local.common.SetNVPairCommand;
import org.eclipse.hyades.internal.execution.local.control.Connection;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.SecureConnectionImpl;

/**
 * The file manager factory is the recommended access point for obtaining an
 * instace of the file manager to use. The file manager will begin the starting
 * timeout at a low value and increment it by a constant amount as the new and
 * legacy file server protocol is attempted, if the communication succeeds then
 * the appropriate concrete file manager is created; if the communication fails
 * it will continue alternating between the legacy and new file server protocols
 * until a communication can be made. Each set of attempts will increment the
 * timeout value by a constant value until the maximum amount of query server
 * availability tries is exceeded.
 * 
 * @author Scott E. Schneider
 */
public final class FileManagerFactory implements IFileManagerFactory {

    /**
     * <code>MINIMUM_RAC_VERSION_FOR_NEW_SERVER</code> -- minimum RAC version that 
     * supports new file transfer protocol
     */
    private static final String MINIMUM_RAC_VERSION_FOR_NEW_SERVER = "3.3"; //$NON-NLS-1$

    /**
	 * <code>VERSION</code> -- string that is searched for when determining 
	 * version of RAC
	 */
	private static final String VERSION = "version"; //$NON-NLS-1$

	/**
	 * <code>ORG_ECLIPSE_HYADES_DATACOLLECTION</code> -- plugin that is queried
	 * when determining version of RAC before 4.0.
	 */
	private static final String ORG_ECLIPSE_HYADES_DATACOLLECTION = "org.eclipse.hyades.datacollection"; //$NON-NLS-1$
	
	/**
	 * <code>ORG_ECLIPSE_TPTP_PLATFORM_COLLECTION_FRAMEWORK</code> -- plugin that is queried
	 * when determining version of RAC in 4.0 or later
	 */
	private static final String ORG_ECLIPSE_TPTP_PLATFORM_COLLECTION_FRAMEWORK = "org.eclipse.tptp.platform.collection.framework";
	
    /**
     * The singleton file manager factory instance
     */
    private static final IFileManagerFactory factory;

    /**
     * Value to increment the timeout when alternating back and forth between
     * the legacy and new protocol to determine the proper protocol to use.
     */
    private static final int QUERY_SERVER_AVAILABILITY_TIMEOUT_INCREMENT = 250;

    /**
     * Starting Timeout used when determine what file manager implementation to
     * instantiate and use (it is assumed that if a response is not received in
     * a specified number of tries, that the older file manager should be used,
     * and that an older agent controller or situation exists that prevents the
     * new one from starting and servicing requests as needed.
     */
    private static final int QUERY_SERVER_AVAILABILITY_TIMEOUT_START = 500;

    /**
     * The number of times to interrogate the file server to determine to the
     * appropriate protocol to use. Each try includes an attempted ping to the
     * legacy server using the legacy protocol and an attempted ping to the new
     * server using the legacy protocol.
     */
    private static final int QUERY_SERVER_AVAILABILITY_TRIES = 5;

    /**
     * Create and store the singleton instance, not using on demand creation,
     * always create on class load
     */
    static {
        factory = new FileManagerFactory();
    }

    /**
     * Singleton accessor method
     * 
     * @return
	 * @provisional
     */
    public static IFileManagerFactory getInstance() {
        return FileManagerFactory.factory;
    }

    /**
     * Limit instantiation to same class for singleton pattern
     */
    private FileManagerFactory() {
    }

    /**
     * Create file manager factory method
     * 
     * @param connection
     *            connection to use for this file manager instance
     * @return the extended file manager interface to operate on, null if none
     *         could be found
	 * @provisional
     */
    public IFileManagerExtended create(Connection connection) {
    	boolean useLegacyImpl = false;
    	IFileManagerExtended fileManager = null;
    	int retries = 0;
    	
    	if (connection instanceof SecureConnectionImpl || isLegacyRACVersion(connection)) {
    		useLegacyImpl = true;
    	}
    	
    	while (retries < QUERY_SERVER_AVAILABILITY_TRIES && fileManager == null) {
	    	if (useLegacyImpl)
	    	{
	    		fileManager = new FileManagerExtendedImpl.Adapter(new FileManagerLegacy(connection));
	    	}
	    	else
	    	{
	    		fileManager = new FileManagerExtendedImpl(connection);
	    	}
	    	if ( fileManager == null )
	    	{
	    		try {
					Thread.sleep(750);
				} catch (InterruptedException e) {
				}
	    	}
    	}
    	
    	return fileManager;
    }

    /**
	 * @param connection
	 * @return
	 */
	private boolean isLegacyRACVersion(Connection connection) {
		// query the RAC to see if it's version string is high enough to support
		// the new file transfer service.
		String version = getRACversion(connection.getNode());
		if (version == null || !versionIsAtLeast(MINIMUM_RAC_VERSION_FOR_NEW_SERVER, version))
		{
			return true;
		}
		return false;
	}

	/**
	 * @param required
	 * @param version
	 * @return
	 */
	private static boolean versionIsAtLeast(String required, String version) {
		StringTokenizer stReq = new StringTokenizer(required, "."); //$NON-NLS-1$
		StringTokenizer stVer = new StringTokenizer(version, "."); //$NON-NLS-1$
		//System.out.println("testing that " + version + " is at least " + required); //$NON-NLS-1$
		/*
		 * Check major version
		 */
		int majReq = 0;
		int majVer = -1;
		// Get the major versions
		if(stReq.hasMoreTokens() && stVer.hasMoreTokens()) {
			majReq = Integer.valueOf(stReq.nextToken()).intValue();
			majVer = Integer.valueOf(stVer.nextToken()).intValue();
		}
		else {
			//System.out.println("returning false");	//$NON-NLS-1$
			return false; // can't find major version
		}

		if(majVer < majReq) {
			//System.out.println("returning false");	//$NON-NLS-1$
			return false;
		}
		else if(majVer > majReq) {
			//System.out.println("returning true");		//$NON-NLS-1$
			return true;
		}

		int min = java.lang.Math.min(stReq.countTokens(), stVer.countTokens());
		for ( int i = 0; i < min; i++ ) {
			int minReq = 0;
			int minVer = -1;
			// Get the minor versions
			if(stReq.hasMoreTokens()) { // do this only if minor version exists
				minReq = Integer.valueOf(stReq.nextToken()).intValue();
				if(stVer.hasMoreTokens()) {
					minVer = Integer.valueOf(stVer.nextToken()).intValue();
					//System.out.println("required = " + minReq + " version = " + minVer ); //$NON-NLS-1$
					if(minVer < minReq) {
						//System.out.println("returning false");  //$NON-NLS-1$
						return false;
					}
					else if(minVer > minReq) {
						//System.out.println("returning true");  //$NON-NLS-1$
						return true;
					}
					else if ( i == (min-1) )	// last time thru, check for equality
					{
						if ( !stReq.hasMoreTokens() ) {
							//System.out.println("returning " + (minVer == minReq ? "true":"false"));  //$NON-NLS-1$
							return minVer == minReq;
						}
						else {
							//System.out.println("returning false");	//$NON-NLS-1$
							return false;
						}
						
					}
				}

			}
		}
		//System.out.println("returning false at the end");	//$NON-NLS-1$
		return false;
	}
	
	/**
	 * @param node
	 * @return
	 */
	private String getRACversion(Node node) {
		String name_3_3 = ORG_ECLIPSE_HYADES_DATACOLLECTION;
		String name_4_0 = ORG_ECLIPSE_TPTP_PLATFORM_COLLECTION_FRAMEWORK;
        String type = VERSION;
		String value = getRACPropertyStringValue(node, name_3_3, type);
		if(value == null)
			value = getRACPropertyStringValue(node, name_4_0, type);
		
		return value;		
	}
	
	private String getRACPropertyStringValue(Node node, String name, String type)
	{
		SetNVPairCommand[] values = null;
		try {
			values = node.getPropertyValues(name, type);
		} catch (NotConnectedException e) {
		}
		if (values != null && values.length > 0)
		{
			return values[0].getValue();
		}
		return null;
	}

    /**
     * Creates a file manager used for performance timing measurements
     * 
     * @param connection
     *            the connection to the agent controller
     * @param iterations
     *            the number of times each method will be exercised per
     *            invocation in
     * @return the file manager configured to use
	 * @provisional
     */
    public IFileManagerExtended createTimed(Connection connection, int iterations) {

    	boolean useLegacyImpl = false;
    	IFileManagerExtended fileManager = null;
    	int retries = 0;
    	
    	if (connection instanceof SecureConnectionImpl || isLegacyRACVersion(connection)) {
    		useLegacyImpl = true;
    	}
    	
    	while (retries < QUERY_SERVER_AVAILABILITY_TRIES && fileManager == null) {
	    	if (useLegacyImpl)
	    	{
	    		fileManager = new FileManagerExtendedImpl.Adapter(new FileManagerLegacy(connection));
	    	}
	    	else
	    	{
	    		fileManager = new FileManagerExtendedTimedImpl(connection, iterations);
	    	}
	    	if ( fileManager == null )
	    	{
	    		try {
					Thread.sleep(750);
				} catch (InterruptedException e) {
				}
	    	}
    	}
    	
    	return fileManager;
    }

}